#include "clickhouse.h"

#include <infra/yasm/collector/python/deserializers/string_cache.h>

#include <library/cpp/clickhouse/client/columns/string.h>
#include <library/cpp/clickhouse/client/columns/numeric.h>
#include <library/cpp/clickhouse/client/block.h>
#include <library/cpp/clickhouse/client/client.h>

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/xrange.h>
#include <util/system/mutex.h>

namespace NCollector {
    namespace {
        const static TStringBuf EMPTY = "";
        const static TStringBuf TIER = "tier";
        const static TStringBuf HOST = "host";
        const static TStringBuf GROUP = "group";
        const static TStringBuf SELF = "self";
        const static TStringBuf COMMON = "common";

        const static size_t INSTANCE_THRESHOLD = 100 * 1000;
        const static size_t SIGNAL_THRESHOLD = 1000 * 1000;

        static TStringCache HostCache;

        template <class InputIt, class Distance>
        inline InputIt AdvanceIterator(InputIt it, Distance distance) {
            std::advance(it, distance);
            return it;
        }
    }

    class TClickhouseTransmitter::TImpl {
    public:
        TImpl(const TClickhouseOptions& options)
            : InstancesTableName(options.InstancesTableName)
            , SignalsTableName(options.SignalsTableName)
            , CommonTableName(options.CommonTableName)
            , Dummy(options.Dummy)
            , CurrentInstanceId(1)
        {
            InstanceIdValues.reserve(INSTANCE_THRESHOLD);
            MetaGroupValues.reserve(INSTANCE_THRESHOLD);
            GroupValues.reserve(INSTANCE_THRESHOLD);
            HostValues.reserve(INSTANCE_THRESHOLD);
            ItypeValues.reserve(INSTANCE_THRESHOLD);

            for (const auto& tag : options.Tags) {
                if (tag != HOST && tag != GROUP) {
                    TagValues.emplace_back(tag, TVector<TStringBuf>()).second.reserve(INSTANCE_THRESHOLD);
                }
            }
            SortBy(TagValues.begin(), TagValues.end(), [](const auto& pair) {
                return pair.first;
            });

            SignalInstanceIdValues.reserve(SIGNAL_THRESHOLD);
            SignalNameValues.reserve(SIGNAL_THRESHOLD);

            ClickHouseOptions
                .SetHost(options.Host)
                .SetPort(options.Port)
                .SetUser(options.User)
                .SetPassword(options.Password)
                .SetDefaultDatabase(options.DefaultDatabase)
                .SetRequestTimeout(TDuration::Seconds(options.Timeout));
        }

        void HistoryExtend(const TVector<TGroupInfo>& groups, const THserverResponse& response) {
            THashMap<TStringBuf, TStringBuf> groupToMetaGroup;
            for (const auto& info : groups) {
                groupToMetaGroup.emplace(HostCache.Get(info.GroupName), HostCache.Get(info.MetaGroupName));
            }

            for (const auto& hostSignals : response) {
                const auto groupInfo(groupToMetaGroup.find(hostSignals.GetHostname()));
                if (groupInfo == groupToMetaGroup.end()) {
                    continue;
                }

                auto variants(hostSignals.GetInstanceKey().CartesianProduct());
                for (const auto& instanceKey : variants) {
                    OnInstanceKey(
                        groupInfo->second, groupInfo->first, hostSignals.GetHostname(),
                        instanceKey, hostSignals.GetSignals()
                    );
                }
            }
        }

        void RealtimeExtend(const TVector<TGroupInfo>& groups, const THserverResponse& response) {
            // group -> metaGroup
            THashMap<TStringBuf, TVector<TStringBuf>> groupToMetagroups;
            THashSet<TStringBuf> hostNames;

            for (const auto& groupInfo : groups) {
                auto groupHostNames = response.GetGroupHostNames(groupInfo.GroupName);
                hostNames.insert(groupHostNames.begin(), groupHostNames.end());
                groupToMetagroups[groupInfo.GroupName].emplace_back(HostCache.Get(groupInfo.MetaGroupName));
            }

            for (const auto& hostSignals : response) {
                const auto groupName = hostSignals.GetInstanceKey().GetGroupName().GetName();
                const auto hostName = hostSignals.GetHostname();

                const auto groupInfo = groupToMetagroups.find(groupName);

                if (groupInfo == groupToMetagroups.end() || hostNames.find(hostName) == hostNames.end()) {
                    continue;
                }

                auto variants = hostSignals.GetInstanceKey().CartesianProduct();

                for (const auto& metagroupName : groupInfo->second){
                    for (const auto& instanceKey : variants) {
                        OnInstanceKey(
                            metagroupName, groupName, hostName,
                            instanceKey, hostSignals.GetSignals()
                        );
                    }
                }
            }
        }

        void DumpInstances(bool force) {
            const auto block(SafePrepareInstanceBlock(force));
            InsertBlock(InstancesTableName, block);
        }

        void DumpSignals(bool force) {
            const auto block(SafePrepareSignalBlock(force));
            InsertBlock(SignalsTableName, block);
        }

        void DumpCommon() {
            const auto block(SafePrepareCommon());
            InsertBlock(CommonTableName, block);
        }

        void Reset() {
            TGuard<TMutex> guard(Lock);

            ResetInstanceChunk();
            ResetSignalChunk();
            ResetCommon();

            CurrentInstanceId = 1;
        }

    private:
        void CreateBigSelfInstance(TStringBuf metaGroup, TStringBuf groupName, TStringBuf hostName, TStringBuf itype) {
            InstanceIdValues.emplace_back(0);
            MetaGroupValues.emplace_back(metaGroup);
            GroupValues.emplace_back(groupName);
            HostValues.emplace_back(hostName);
            ItypeValues.emplace_back(itype);

            for (auto& pair : TagValues) {
                if (pair.first == TIER) {
                    pair.second.emplace_back(SELF);
                } else {
                    pair.second.emplace_back(EMPTY);
                }
            }
        }

        void OnInstanceKey(TStringBuf metaGroup, TStringBuf groupName, TStringBuf hostName,
                           NTags::TInstanceKey instanceKey,
                           const TVector<TStringBuf>& signals) {
            TGuard<TMutex> guard(Lock);

            const auto itype(instanceKey.GetItype());
            if (SeenBigSelfs.find(std::make_pair(hostName, itype)) == SeenBigSelfs.end()) {
                CreateBigSelfInstance(metaGroup, groupName, hostName, itype);
                SeenBigSelfs.emplace(hostName, itype);
            }

            InstanceIdValues.emplace_back(CurrentInstanceId);
            MetaGroupValues.emplace_back(metaGroup);
            GroupValues.emplace_back(groupName);
            HostValues.emplace_back(hostName);
            ItypeValues.emplace_back(itype);

            const auto instanceTags(instanceKey.GetTags());
            auto leftIt(TagValues.begin());
            auto rightIt(instanceTags.begin());
            while (leftIt != TagValues.end() && rightIt != instanceTags.end()) {
                const int res = leftIt->first.compare(rightIt->Name);
                if (res == 0) {
                    // left == right
                    leftIt->second.emplace_back(rightIt->Value);
                    ++leftIt;
                    ++rightIt;
                } else if (res < 0) {
                    // left < right
                    leftIt->second.emplace_back(EMPTY);
                    ++leftIt;
                } else {
                    // left > right
                    ++rightIt;
                }
            }

            while (leftIt != TagValues.end()) {
                leftIt->second.emplace_back(EMPTY);
                ++leftIt;
            }

            // there are a lot of signals, dill with it
            SignalInstanceIdValues.resize(SignalInstanceIdValues.size() + signals.size());
            std::fill(
                AdvanceIterator(SignalInstanceIdValues.begin(), SignalInstanceIdValues.size() - signals.size()),
                SignalInstanceIdValues.end(),
                CurrentInstanceId
            );

            SignalNameValues.resize(SignalNameValues.size() + signals.size());
            std::copy(
                signals.begin(), signals.end(),
                AdvanceIterator(SignalNameValues.begin(), SignalNameValues.size() - signals.size())
            );

            if (itype == COMMON) {
                CommonSignals.insert(signals.begin(), signals.end());
            }

            CurrentInstanceId++;

            if (InstanceIdValues.size() >= INSTANCE_THRESHOLD) {
                const auto block(PrepareInstanceBlock(false));
                auto unguard = Unguard(guard);
                InsertBlock(InstancesTableName, block);
            }

            if (SignalInstanceIdValues.size() >= SIGNAL_THRESHOLD) {
                const auto block(PrepareSignalBlock(false));
                auto unguard = Unguard(guard);
                InsertBlock(SignalsTableName, block);
            }
        }

        void ResetInstanceChunk() {
            InstanceIdValues.clear();
            MetaGroupValues.clear();
            GroupValues.clear();
            HostValues.clear();
            ItypeValues.clear();
            for (auto& pair : TagValues) {
                pair.second.clear();
            }
        }

        NClickHouse::TBlock PrepareInstanceBlock(bool force) {
            NClickHouse::TBlock block;
            if (force || InstanceIdValues.size() >= INSTANCE_THRESHOLD) {
                block.AppendColumn("instance_id", NClickHouse::TColumnUInt64::Create(InstanceIdValues));
                block.AppendColumn("metagroup", NClickHouse::TColumnStringBuf::Create(MetaGroupValues));
                block.AppendColumn("group", NClickHouse::TColumnStringBuf::Create(GroupValues));
                block.AppendColumn("host", NClickHouse::TColumnStringBuf::Create(HostValues));
                block.AppendColumn("itype", NClickHouse::TColumnStringBuf::Create(ItypeValues));
                for (const auto& pair : TagValues) {
                    block.AppendColumn(pair.first, NClickHouse::TColumnStringBuf::Create(pair.second));
                }
                ResetInstanceChunk();
            }
            return block;
        }

        NClickHouse::TBlock SafePrepareInstanceBlock(bool force) {
            TGuard<TMutex> guard(Lock);
            return PrepareInstanceBlock(force);
        }

        void ResetSignalChunk() {
            SignalInstanceIdValues.clear();
            SignalNameValues.clear();
        }

        NClickHouse::TBlock PrepareSignalBlock(bool force) {
            NClickHouse::TBlock block;
            if (force || SignalInstanceIdValues.size() >= SIGNAL_THRESHOLD) {
                block.AppendColumn("instance_id", NClickHouse::TColumnUInt64::Create(SignalInstanceIdValues));
                block.AppendColumn("signal", NClickHouse::TColumnStringBuf::Create(SignalNameValues));
                ResetSignalChunk();
            }
            return block;
        }

        NClickHouse::TBlock SafePrepareSignalBlock(bool force) {
            TGuard<TMutex> guard(Lock);
            return PrepareSignalBlock(force);
        }

        void ResetCommon() {
            SeenBigSelfs.clear();
            CommonSignals.clear();
        }

        NClickHouse::TBlock SafePrepareCommon() {
            TGuard<TMutex> guard(Lock);
            NClickHouse::TBlock block;
            if (!CommonSignals.empty()) {
                auto signals(NClickHouse::TColumnStringBuf::Create());
                for (const auto& signal : CommonSignals) {
                    signals->Append(signal);
                }
                block.AppendColumn("signal", signals);
                CommonSignals.clear();
            }
            return block;
        }

        void InsertBlock(const TString& tableName, const NClickHouse::TBlock& block) {
            if (block.GetRowCount() && !Dummy) {
                NClickHouse::TClient client(ClickHouseOptions);
                client.Insert(tableName, block);
            }
        }

        TMutex Lock;

        // clickhouse client
        NClickHouse::TClientOptions ClickHouseOptions;
        const TString InstancesTableName;
        const TString SignalsTableName;
        const TString CommonTableName;
        const bool Dummy;

        // current instance id
        ui64 CurrentInstanceId;

        // hostname, itype
        THashSet<std::pair<TStringBuf, TStringBuf>> SeenBigSelfs;
        THashSet<TStringBuf> CommonSignals;

        // values for instances
        TVector<ui64> InstanceIdValues;
        TVector<TStringBuf> MetaGroupValues;
        TVector<TStringBuf> GroupValues;
        TVector<TStringBuf> HostValues;
        TVector<TStringBuf> ItypeValues;

        // first element - column (or tag) name, second element - column values
        TVector<std::pair<TString, TVector<TStringBuf>>> TagValues;

        // columns for signals
        TVector<ui64> SignalInstanceIdValues;
        TVector<TStringBuf> SignalNameValues;
    };

    TClickhouseTransmitter::TClickhouseTransmitter(const TClickhouseOptions& options)
        : Impl(MakeHolder<TImpl>(options))
    {
    }

    TClickhouseTransmitter::TClickhouseTransmitter()
        : TClickhouseTransmitter(TClickhouseOptions())
    {
    }

    TClickhouseTransmitter::~TClickhouseTransmitter() {
    }

    void TClickhouseTransmitter::HistoryExtend(const TVector<TGroupInfo>& groups, const THserverResponse& response) {
        Impl->HistoryExtend(groups, response);
    }

    void TClickhouseTransmitter::RealtimeExtend(const TVector<TGroupInfo>& groups, const THserverResponse& response) {
        Impl->RealtimeExtend(groups, response);
    }

    void TClickhouseTransmitter::DumpInstances(bool force) {
        Impl->DumpInstances(force);
    }

    void TClickhouseTransmitter::DumpSignals(bool force) {
        Impl->DumpSignals(force);
    }

    void TClickhouseTransmitter::DumpCommon() {
        Impl->DumpCommon();
    }

    void TClickhouseTransmitter::Reset() {
        Impl->Reset();
    }
}
