#include "dataproxy_reader.h"

#include <solomon/libs/cpp/string_pool/string_pool.h>

#include <infra/monitoring/common/perf.h>
#include <infra/yasm/stockpile_client/dataproxy_client.h>
#include <infra/yasm/common/labels/host/host.h>
#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/common/labels/tags/verification.h>

#include "metrics.h"

using namespace NHistDb::NStockpile;
using namespace NMonitoring;
using namespace NCollector;

namespace {
    constexpr TDuration TAGS_COLLECTOR_REQUEST_DEPTH = TDuration::Days(14);
    constexpr size_t CLUSTERS_TO_REQUEST = 2;
    constexpr size_t ATTEMPTS_PER_CLUSTER = 2;

    bool IsYasmInternalTag(TStringBuf tag) {
        return tag == HOST_LABEL_KEY ||
               tag == GROUP_LABEL_KEY ||
               tag == SIGNAL_LABEL_KEY ||
               tag == PROJECT_LABEL_KEY ||
               tag == CLUSTER_LABEL_KEY ||
               tag == SERVICE_LABEL_KEY;
    }
}

TSelectorBuilder::TSelectorBuilder() {
    KeysWithValues.emplace_back("service", NHistDb::NStockpile::STOCKPILE_YASM_SERVICE, ESelectorType::EQUAL_VALUE);
}

void TSelectorBuilder::AddValues(const TString& key, const TVector<TString>& values, ESelectorType selectorType, bool prepareForRegexp) {
    if (values.empty()) {
        return;
    }
    TVector<TString> convertedValues;
    for (auto& value: values) {
        if (value.Contains("\"")) {
            throw yexception() << "Selector '" << key << "' field has incorrect value '" << value << "'";
        }
        if (prepareForRegexp) {
            // Legacy weird logic for glob and regexp support at same time
            convertedValues.push_back(JoinSeq(".*", StringSplitter(value).Split('*').ToList<TString>()));
        } else {
            convertedValues.push_back(value);
        }
    }
    KeysWithValues.emplace_back(key, JoinSeq("|", convertedValues), selectorType);
}

void TSelectorBuilder::AddPattern(const TString& key, const TMaybe<TString>& pattern) {
    if (pattern.Defined()) {
        if (pattern.GetRef().Contains("\"")) {
            throw yexception() << "Selector '" << key << "' field has incorrect value '" << pattern.GetRef() << "'";
        }
        KeysWithValues.emplace_back(key, pattern.GetRef(), ESelectorType::EQUAL_REGEX);
    }
}

void TSelectorBuilder::AddSignalsAndTags(const TVector<TString>& signals, const TTagValuesMap& tags) {
    bool useExactMatch = true;
    for (auto& signal : signals) {
        if (!NZoom::NSignal::IsValidSignalName(signal)){
            useExactMatch = false;
        }
    }
    if (useExactMatch) {
        AddValues(SIGNAL_LABEL_KEY, signals, ESelectorType::EQUAL_VALUE, false);
    } else {
        AddValues(SIGNAL_LABEL_KEY, signals, ESelectorType::EQUAL_REGEX, true);
    }

    for (auto& [key, values]: tags) {
        useExactMatch = true;
        for (auto& value : values) {
            if (!NTags::IsCorrectInstanceTagValue(value)) {
                useExactMatch = false;
            }
        }
        if (useExactMatch) {
            AddValues(key, values, ESelectorType::EQUAL_VALUE, false);
        } else {
            AddValues(key, values, ESelectorType::EQUAL_REGEX, true);
        }
    }
}

TString TSelectorBuilder::ToString() const {
    TStringBuilder stringBuilder;
    TVector<TString> keyValues;

    stringBuilder << "{";
    for (auto& [key, value, selectorType]: KeysWithValues) {
        keyValues.emplace_back(TString::Join(key, SelectorTypeToString(selectorType), "\"", value, "\""));
    }
    stringBuilder << JoinSeq(",", keyValues);
    stringBuilder << "}";

    return stringBuilder;
}

TStringBuf TSelectorBuilder::SelectorTypeToString(ESelectorType selectorType) const {
    switch(selectorType) {
        case ESelectorType::EQUAL_GLOB:
            return TStringBuf("=");
        case ESelectorType::NOT_EQUAL_GLOB:
            return TStringBuf("!=");
        case ESelectorType::EQUAL_VALUE:
            return TStringBuf("==");
        case ESelectorType::NOT_EQUAL_VALUE:
            return TStringBuf("!==");
        case ESelectorType::EQUAL_REGEX:
            return TStringBuf("=~");
        case ESelectorType::NOT_EQUAL_REGEX:
            return TStringBuf("!=~");
    }
}

TDataProxyReader::TDataProxyReader(TDataProxyMultiClusterState& dataProxyState, TRequestLog& logger)
    : DataProxyState(dataProxyState)
    , Logger(logger)
    , GrpcHandler(Logger) {
}

THashMap<TString, TVector<TString>> TDataProxyReader::FindTagsForItypes(const TSet<TString>& itypes, TInstant deadline) {
    auto requester = NDataProxyClient::TDataProxyRequester(
        &TDataProxyService::Stub::PrepareAsyncLabelKeys,
        DataProxyState.GetAllHosts(),
        NMetrics::DATAPROXY_LABEL_KEYS,
        CLUSTERS_TO_REQUEST,
        ATTEMPTS_PER_CLUSTER,
        Logger
    );
    requester.Reserve(itypes.size());
    THashMap<TString, TVector<TString>> result;
    const auto now = TInstant::Now();
    for (auto& itype: itypes) {
        auto& request = requester.PrepareCall(
            [itype, &result](auto&, const auto* response) {
                if (response) {
                    auto& tagsVector = result[itype];
                    NSolomon::NStringPool::TStringPool stringPool(response->Getstring_pool());
                    for (auto idx: response->Getkeys_idx()) {
                        auto tag = stringPool[idx];
                        if (!IsYasmInternalTag(tag)) {
                            tagsVector.emplace_back(tag);
                        }
                    }
                    Sort(tagsVector);
                }
            });
        NDataProxyClient::FillLabelKeysRequest(MakeItypeProjectId(itype), TSelectorBuilder().ToString(), now - TAGS_COLLECTOR_REQUEST_DEPTH,
            now, request);
    }
    auto callsFailed = requester.Request(deadline);
    if (callsFailed) {
        throw yexception() << "Some calls to DataProxy failed";
    }
    return result;
}

THostsResult TDataProxyReader::FindHosts(const TSet<TString>& itypes, THostTypeArg hostTypes, const TTagValuesMap& tags,
                                         const TVector<TString>& signals, size_t limit, const TMaybe<TString>& hostPattern,
                                         const THashMap<TString, TString>& groupToMetagroupMap, TInstant deadline) {
    TSelectorBuilder selectorBuilder;
    selectorBuilder.AddSignalsAndTags(signals, tags);

    auto requester = NDataProxyClient::TDataProxyRequester(
        &TDataProxyService::Stub::PrepareAsyncLabelValues,
        DataProxyState.GetAllHosts(),
        NMetrics::DATAPROXY_LABEL_VALUES,
        CLUSTERS_TO_REQUEST,
        ATTEMPTS_PER_CLUSTER,
        Logger
    );
    auto requestedKeys = HostTypesToKeySet(hostTypes);
    requester.Reserve(itypes.size() * requestedKeys.size());
    THostsResult result = {};
    const auto now = TInstant::Now();

    for (auto& requestedKey: requestedKeys) {
        for (auto& itype: itypes) {
            auto& request = requester.PrepareCall(
                [&requestedKey, &result, &groupToMetagroupMap, &hostTypes, this](auto&, const auto* response) {
                    if (!response) {
                        return;
                    }
                    NSolomon::NStringPool::TStringPool stringPool(response->Getstring_pool());
                    bool needRequestedKeyValues = requestedKey.equal("host") || (requestedKey.equal("group") && hostTypes.HasFlags(EHostTypeArgFlags::GROUP));
                    bool needMetagroups = requestedKey.equal("group") && hostTypes.HasFlags(EHostTypeArgFlags::METAGROUP);

                    for (auto& label: response->Getlabels()) {
                        auto keyIdx = label.Getkey_idx();
                        auto key = stringPool[keyIdx];

                        if (!requestedKey.equal(key)) {
                            continue;
                        }
                        for (auto valueIdx: label.Getvalues_idx()) {
                            auto hostName = stringPool[valueIdx];
                            if (needRequestedKeyValues) {
                                result.Hosts.insert({TString(hostName), requestedKey});
                            }
                            if (needMetagroups) {
                                auto it = groupToMetagroupMap.find(hostName);
                                if (it) {
                                    result.Hosts.insert({it->second, "metagroup"});
                                } else {
                                    Logger << ELogPriority::TLOG_ERR << "Can not find metagroup for " << hostName;
                                }
                            }
                        }
                    }
                });
            TSelectorBuilder tmpSelectorBuilder = selectorBuilder;
            if (requestedKey.equal("host")) {
                tmpSelectorBuilder.AddValues(CLUSTER_LABEL_KEY, {"host_*"}, ESelectorType::EQUAL_GLOB);
            } else if (requestedKey.equal("group")) {
                tmpSelectorBuilder.AddValues(CLUSTER_LABEL_KEY, {"group_*"}, ESelectorType::EQUAL_GLOB);
            }
            tmpSelectorBuilder.AddPattern(requestedKey, hostPattern);
            NDataProxyClient::FillLabelValuesRequest(MakeItypeProjectId(itype), tmpSelectorBuilder.ToString(), {requestedKey},
                                                     now - TAGS_COLLECTOR_REQUEST_DEPTH, now, limit, request);
        }
    }

    auto callsFailed = requester.Request(deadline);
    if (callsFailed) {
        throw yexception() << "Some calls to DataProxy failed";
    }

    // TODO: wrong total in truncated requests or with limit
    result.Total = result.Hosts.size();

    return result;
}

TSet<TString> TDataProxyReader::FindKeyValues(const TSet<TString>& itypes, const TString& requestedKey, const TTagValuesMap& tags,
                                              const TVector<TString>& hosts, const TVector<TString>& signals,
                                              const TMaybe<TString>& valuePattern, const TSet<TString>& metagroups,
                                              size_t limit, TInstant deadline) {
    TSelectorBuilder selectorBuilder;
    selectorBuilder.AddSignalsAndTags(signals, tags);
    selectorBuilder.AddPattern(requestedKey, valuePattern);

    TVector<TSelectorBuilder> selectors = SelectorsFromHosts(hosts, metagroups, selectorBuilder, true);

    auto requester = NDataProxyClient::TDataProxyRequester(
        &TDataProxyService::Stub::PrepareAsyncLabelValues,
        DataProxyState.GetAllHosts(),
        NMetrics::DATAPROXY_LABEL_VALUES,
        CLUSTERS_TO_REQUEST,
        ATTEMPTS_PER_CLUSTER,
        Logger
    );
    requester.Reserve(itypes.size() * selectors.size());
    TSet<TString> result = {};
    const auto now = TInstant::Now();

    for (auto& tmpSelectorBuilder: selectors) {
        for (auto& itype: itypes) {
            auto& request = requester.PrepareCall(
                [&requestedKey, &result, limit](auto&, const auto* response) {
                    if (!response) {
                        return;
                    }
                    NSolomon::NStringPool::TStringPool stringPool(response->Getstring_pool());

                    for (auto& label: response->Getlabels()) {
                        auto keyIdx = label.Getkey_idx();
                        auto key = stringPool[keyIdx];

                        if (!key.equal(requestedKey)) {
                            continue;
                        }
                        for (auto valueIdx: label.Getvalues_idx()) {
                            auto value = stringPool[valueIdx];
                            if (value.equal(AGGREGATED_MARKER)) {
                                continue;
                            }
                            if (result.size() < limit) {
                                result.insert(TString(value));
                            }
                        }
                    }
                });
            // we request extra item because tags response may have internal /SELF value
            auto limitToRequest = limit + 1;
            NDataProxyClient::FillLabelValuesRequest(MakeItypeProjectId(itype), tmpSelectorBuilder.ToString(),
                                                     {requestedKey},
                                                     now - TAGS_COLLECTOR_REQUEST_DEPTH, now, limitToRequest, request);
        }
    }
    auto callsFailed = requester.Request(deadline);
    if (callsFailed) {
        throw yexception() << "Some calls to DataProxy failed";
    }
    return result;
}

TUniqLabelsResult TDataProxyReader::FindUniqueLabels(const TSet<TString>& itypes, const TTagValuesMap& tags,
                                                     const TVector<TString>& hosts, const TVector<TString>& signals,
                                                     const TSet<TString>& keys, bool needItype,
                                                     const TSet<TString>& metagroups, TInstant deadline) {
    TUniqLabelsResult result;
    TSelectorBuilder selectorBuilder;
    selectorBuilder.AddSignalsAndTags(signals, tags);

    auto requester = NDataProxyClient::TDataProxyRequester(
        &TDataProxyService::Stub::PrepareAsyncUniqueLabels,
        DataProxyState.GetAllHosts(),
        NMetrics::DATAPROXY_UNIQUE_LABELS,
        CLUSTERS_TO_REQUEST,
        ATTEMPTS_PER_CLUSTER,
        Logger
    );
    TVector<TSelectorBuilder> selectors = SelectorsFromHosts(hosts, metagroups, selectorBuilder, true);
    requester.Reserve(itypes.size() * selectors.size());
    for (auto& tmpSelectorBuilder: selectors) {
        for (auto& itype: itypes) {
            auto& request = requester.PrepareCall(
                [&itype, &needItype, &result](auto&, const auto* response) {
                    if (!response) {
                        return;
                    }
                    NSolomon::NStringPool::TStringPool stringPool(response->Getstring_pool());
                    for (auto& labels: response->Getlabels()) {
                        auto& labelsIdx = labels.Getlabels_idx();
                        if (labelsIdx.size() % 2) {
                            ythrow yexception() << "Dataproxy response has odd number of labels";
                        }
                        TVector<std::pair<TString, TString>> labelsVector;
                        labelsVector.reserve(labelsIdx.size()/2 + 1);
                        if (needItype) {
                            labelsVector.emplace_back("itype", itype);
                        }
                        for (int i = 0; i < labelsIdx.size(); i += 2) {
                            auto tagKey = stringPool[labelsIdx[i]];
                            auto tagValue = stringPool[labelsIdx[i + 1]];
                            if (!tagValue.equal(AGGREGATED_MARKER)) {
                                labelsVector.emplace_back(tagKey, tagValue);
                            } else {
                                labelsVector.emplace_back(tagKey, "");
                            }
                        }
                        if (!labelsVector.empty()) {
                            Sort(labelsVector);
                            result.insert(labelsVector);
                        }
                    }
                });
            NDataProxyClient::FillUniqueLabelsRequest(MakeItypeProjectId(itype), tmpSelectorBuilder.ToString(),
                                                      keys, request);
        }
    }
    auto callsFailed = requester.Request(deadline);
    if (callsFailed) {
        throw yexception() << "Some calls to DataProxy failed";
    }
    return result;
}

TSet<TString> TDataProxyReader::HostTypesToKeySet(THostTypeArg hostTypes) const {
    TSet<TString> keys;
    if (hostTypes.HasFlags(EHostTypeArgFlags::HOST)) {
        keys.insert("host");
    }
    if (hostTypes.HasFlags(EHostTypeArgFlags::GROUP) || hostTypes.HasFlags(EHostTypeArgFlags::METAGROUP)) {
        keys.insert("group");
    }
    return keys;
}

TVector<TSelectorBuilder> TDataProxyReader::SelectorsFromHosts(const TVector<TString>& hosts, const TSet<TString>& metagroups,
                                                               const TSelectorBuilder& selectorBuilder, bool onlyHostShards) const {
    TVector<TString> hostNames;
    TVector<TString> groupNames;
    bool ignoreHostFiltering = hosts.empty();
    for (auto& name: hosts) {
        if (NZoom::NHost::THostName(name).IsGroup()) {
            if (metagroups.contains(name)) {
                ignoreHostFiltering = true;
                break;
            } else {
                groupNames.emplace_back(name);
            }
        } else {
            hostNames.emplace_back(name);
        }
    }
    TVector<TSelectorBuilder> selectors;
    if (ignoreHostFiltering) {
        if (onlyHostShards) {
            TSelectorBuilder tmpSelectorBuilder = selectorBuilder;
            tmpSelectorBuilder.AddValues(CLUSTER_LABEL_KEY, {"host_*"}, ESelectorType::EQUAL_GLOB);
            selectors.emplace_back(tmpSelectorBuilder);
        } else {
            selectors.emplace_back(selectorBuilder);
        }
    } else {
        if (!hostNames.empty()) {
            TSelectorBuilder tmpSelectorBuilder = selectorBuilder;
            tmpSelectorBuilder.AddValues("host", hostNames, ESelectorType::EQUAL_GLOB);
            selectors.emplace_back(tmpSelectorBuilder);
        }
        if (!groupNames.empty()) {
            TSelectorBuilder tmpSelectorBuilder = selectorBuilder;
            tmpSelectorBuilder.AddValues("group", groupNames, ESelectorType::EQUAL_GLOB);
            selectors.emplace_back(tmpSelectorBuilder);
        }
    }
    return selectors;
}
