#include "dataproxy_reader.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NCollector;

Y_UNIT_TEST_SUITE(TDataProxySelectorBuilderTest) {
    Y_UNIT_TEST(EmptyBuilderHasService) {
        TSelectorBuilder selectorBuilder;
        UNIT_ASSERT_VALUES_EQUAL(selectorBuilder.ToString(), TStringBuf("{service==\"yasm\"}"));
    }

    Y_UNIT_TEST(AddValues) {
        TSelectorBuilder selectorBuilder;
        selectorBuilder.AddValues("key", {"value1", "value2"});
        UNIT_ASSERT_VALUES_EQUAL(selectorBuilder.ToString(), TStringBuf("{service==\"yasm\",key==\"value1|value2\"}"));
    }

    Y_UNIT_TEST(AddMultipleKeys) {
        TSelectorBuilder selectorBuilder;
        selectorBuilder.AddValues("key1", {"value1", "value2"});
        selectorBuilder.AddValues("key2", {"value3"});
        selectorBuilder.AddValues("key3", {"value"}, ESelectorType::NOT_EQUAL_VALUE);
        UNIT_ASSERT_VALUES_EQUAL(selectorBuilder.ToString(), TStringBuf("{service==\"yasm\",key1==\"value1|value2\",key2==\"value3\",key3!==\"value\"}"));
    }

    Y_UNIT_TEST(AddSameKeyTwice) {
        TSelectorBuilder selectorBuilder;
        selectorBuilder.AddValues("key1", {"value1", "value2"});
        selectorBuilder.AddValues("key1", {"value3"});
        UNIT_ASSERT_VALUES_EQUAL(selectorBuilder.ToString(), TStringBuf("{service==\"yasm\",key1==\"value1|value2\",key1==\"value3\"}"));
    }

    Y_UNIT_TEST(SelectorTypes) {
        TSelectorBuilder selectorBuilder;
        selectorBuilder.AddValues("key1", {"value"}, ESelectorType::EQUAL_GLOB);
        selectorBuilder.AddValues("key2", {"value"}, ESelectorType::NOT_EQUAL_GLOB);
        selectorBuilder.AddValues("key3", {"value"}, ESelectorType::EQUAL_VALUE);
        selectorBuilder.AddValues("key4", {"value"}, ESelectorType::NOT_EQUAL_VALUE);
        selectorBuilder.AddValues("key5", {"value"}, ESelectorType::EQUAL_REGEX);
        selectorBuilder.AddValues("key6", {"value"}, ESelectorType::NOT_EQUAL_REGEX);
        UNIT_ASSERT_VALUES_EQUAL(selectorBuilder.ToString(),
            TStringBuf("{service==\"yasm\",key1=\"value\",key2!=\"value\",key3==\"value\",key4!==\"value\",key5=~\"value\",key6!=~\"value\"}"));
    }

    Y_UNIT_TEST(SelectorOptimizationForSignalsAndTags) {
        TVector<std::tuple<TVector<TString>, TTagValuesMap, TStringBuf>> testCases = {
            {
                {"signal1"}, {{"geo", {"man"}}},
                TStringBuf("{service==\"yasm\",signal==\"signal1\",geo==\"man\"}")
            },
            {
                {"signal1", "signal2"}, {{"geo", {"man*"}}},
                TStringBuf("{service==\"yasm\",signal==\"signal1|signal2\",geo=~\"man.*\"}")
            },
            {
                {"signal1*"}, {{"geo", {"sas", "man*"}}},
                TStringBuf("{service==\"yasm\",signal=~\"signal1.*\",geo=~\"sas|man.*\"}")
            },
            {
                {"signal1", "signal2*"}, {{"geo", {"sas"}}, {"ctype", {"pr*"}}},
                TStringBuf("{service==\"yasm\",signal=~\"signal1|signal2.*\",geo==\"sas\",ctype=~\"pr.*\"}")
            },
        };
        for (auto& [signals, tags, result] : testCases) {
            TSelectorBuilder selectorBuilder;
            selectorBuilder.AddSignalsAndTags(signals, tags);
            UNIT_ASSERT_VALUES_EQUAL(selectorBuilder.ToString(), result);
        }
    }
}
