#include "handlers.h"

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/json/writer/json.h>
#include <library/cpp/unistat/unistat.h>
#include <util/generic/guid.h>

#include <infra/yasm/common/labels/tags/verification.h>
#include <infra/monitoring/common/perf.h>

using namespace NCollector;
using namespace NMonitoring;
using namespace NYasm::NGroups;
using namespace NHistDb::NStockpile;

namespace {
    static const size_t DEFAULT_LIMIT = 10000;
}

void TPingHandler::DoReply(const TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_PING_TIME);
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent("OK"));
}

TBaseCollectorHandler::TParsedRequest TBaseCollectorHandler::ParseCollectorArgs(const TString& requestBody, const TParsedHttpFull& meta) {
    TParsedRequest result;
    if (meta.Method == "GET") {
        TCgiParameters cgiParameters(meta.Cgi);
        for (auto& [key, value]: cgiParameters) {
            Split(value, ",", result[key]);
        }
    } else if (meta.Method == "POST") {
        NJson::TJsonValue root;
        try {
            NJson::ReadJsonTree(requestBody, &root, true);
        } catch(const NJson::TJsonException& exc) {
            ythrow TBadRequest() << exc.AsStrBuf();
        }
        if (root.IsMap()) {
            for (const auto& [key, value] : root.GetMap()) {
                auto& values = result[key];
                switch (value.GetType()) {
                    case NJson::JSON_UNDEFINED:
                    case NJson::JSON_NULL:
                    case NJson::JSON_MAP:
                        ythrow TBadRequest() << "Incorrect input data format at key '" << key << "'";
                    case NJson::JSON_ARRAY: {
                        for (const auto& subValue: value.GetArray()) {
                            values.push_back(subValue.GetStringRobust());
                        }
                        break;
                    }
                    default: {
                        values.push_back(value.GetStringRobust());
                        break;
                    }
                }
            }
        } else {
            ythrow TBadRequest() << "Incorrect input data format";
        }
    } else {
        throw TBadRequest() << "Only GET and POST methods are allowed";
    }
    return result;
}

void TBaseCollectorHandler::DoReply(TServiceRequest::TRef request, const TParsedHttpFull& meta) {
    DoReply(request, meta, {}, {});
}

void TBaseCollectorHandler::DoReply(TServiceRequest::TRef request, const TParsedHttpFull& meta, const TVector<TString>& pathGroups, const TMap<int, TString>&) {
    NJson::TJsonValue responseJson;
    TMaybe<TString> error;
    HttpCodes code = HttpCodes::HTTP_OK;
    TRequestLog requestLogger(Logger, "unknown");
    TString requestBody;
    try {
        requestBody = request->GetInput().ReadAll();
        auto args = ParseCollectorArgs(requestBody, meta);

        auto requestIdIt = args.find(REQUEST_ID_FIELD);
        auto requestId = (requestIdIt != args.end() && !requestIdIt->second.empty()) ? requestIdIt->second.front() : CreateGuidAsString();
        requestLogger.SetRequestId(requestId);
        TMeasuredMethod perf(requestLogger, HandlerTimeStatName, HandlerTimeStatName);
        try {
            requestLogger << ELogPriority::TLOG_INFO << "Start processing: " << meta.Method << " " << meta.Request;

            responseJson = HandleParsedRequest(std::move(args), pathGroups, requestLogger);
        } catch (const TBadRequest& e) {
            error = "Bad request: ";
            error.GetRef() += e.what();
            code = HttpCodes::HTTP_BAD_REQUEST;
            TUnistat::Instance().PushSignalUnsafe(Handler4xxStatName, 1);
        } catch (const yexception& e) {
            Logger << ELogPriority::TLOG_ERR << e;
            error = "Server error: ";
            error.GetRef() += e.what();
            code = HttpCodes::HTTP_INTERNAL_SERVER_ERROR;
            TUnistat::Instance().PushSignalUnsafe(Handler5xxStatName, 1);
        }
        requestLogger << ELogPriority::TLOG_INFO << "Request processed. Code: " << code;
    } catch (const TBadRequest& e) {
        error = "Bad request: ";
        error.GetRef() += e.what();
        code = HttpCodes::HTTP_BAD_REQUEST;
        TUnistat::Instance().PushSignalUnsafe(Handler4xxStatName, 1);
    } catch (...) {
        Logger << ELogPriority::TLOG_ERR << "Internal server error: " << CurrentExceptionMessage();
        error = "Internal server error";
        code = HttpCodes::HTTP_INTERNAL_SERVER_ERROR;
        TUnistat::Instance().PushSignalUnsafe(Handler5xxStatName, 1);
    }

    if (error.Defined() && !responseJson.IsDefined()) {
        responseJson = NJson::TJsonMap({{"error", NJson::TJsonValue(error.GetRef())}});
        requestLogger << ELogPriority::TLOG_ERR << meta.Method << " " << meta.Path << " finished with " << code << ": \"" << error << "\"\nBody:" << requestBody;
    }
    NJsonWriter::TBuf jsonBuf;
    jsonBuf.WriteJsonValue(&responseJson);
    request->Finish(THttpResponse(code).SetContentType("application/json").SetContent(jsonBuf.Str()));
}

void TBaseCollectorHandler::InitStatistics(const TStatsInitializer& initializer, TUnistat& creator) const {
    initializer.DrillHistogramHole(creator, HandlerTimeStatName);
    initializer.DrillSummHole(creator, Handler5xxStatName);
    initializer.DrillSummHole(creator, Handler4xxStatName);
}

TString TBaseCollectorHandler::MakeTimeSignalName(const TStringBuf& handlerStatName) {
    TString signalName(handlerStatName);
    signalName += "_time";
    return signalName;
}

TString TBaseCollectorHandler::Make5xxSignalName(const TStringBuf& handlerStatName) {
    TString signalName(handlerStatName);
    signalName += "_5xx";
    return signalName;
}

TString TBaseCollectorHandler::Make4xxSignalName(const TStringBuf& handlerStatName) {
    TString signalName(handlerStatName);
    signalName += "_4xx";
    return signalName;
}

void TReadyHandler::DoReply(const TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_READY_TIME);

    if (ItypeTagsLoader.IsReady()) {
        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent("OK"));
    } else {
        request->Finish(THttpResponse(HttpCodes::HTTP_SERVICE_UNAVAILABLE).SetContent("not ready"));
    }
}

NJson::TJsonValue TTagKeysHandler::HandleParsedRequest(TBaseCollectorHandler::TParsedRequest requestData, const TVector<TString>&, TRequestLog&) {
    TTagKeysHandleArgs args(requestData);
    NJson::TJsonValue result(NJson::JSON_MAP);
    for (auto& [itype, tags]: ItypeTagsLoader.GetTagKeys(args.Itypes)) {
        NJson::TJsonValue tagsResult(NJson::JSON_ARRAY);
        for (auto& tag: tags) {
            tagsResult.AppendValue(tag);
        }
        result.InsertValue(itype, std::move(tagsResult));
    }
    return result;
}

TCollectorHandleArgs::TCollectorHandleArgs(TBaseCollectorHandler::TParsedRequest& requestData,
                                           const TVector<TStringBuf>& requiredArgs, const TVector<TStringBuf>& optionalArgs, bool allowTags)
        : Types(EHostTypeArgFlags::HOST)
        , Limit(DEFAULT_LIMIT)
        , Offset(0)
        , Sorted(ESortType::ASC)
        , UnifiedTagFormat(false) {
    for (auto& arg: requiredArgs) {
        auto it = requestData.find(arg);
        if (it == requestData.end() || it->second.empty()) {
            throw TBadRequest() << "No " << arg << " specified";
        }
    }

    for (auto& [key, values]: requestData) {
        if (key.equal(REQUEST_ID_FIELD)) {
            continue;
        }
        bool keyIsTag = allowTags && Find(FORBIDDEN_TAG_NAMES, key) == FORBIDDEN_TAG_NAMES.end();
        if (Find(requiredArgs, key) == requiredArgs.end() && Find(optionalArgs, key) == optionalArgs.end() && !keyIsTag) {
            throw TBadRequest() << "Unknown argument " << key;
        }
        if (key.equal("itype")) {
            ParseItype(values);
        } else if (key.equal("parents")) {
            Parents = std::move(values);
        } else if (key.equal("host_pattern")) {
            ParseHostPattern(values);
        } else if (key.equal("signals")) {
            ParseSignals(values);
        } else if (key.equal("types")) {
            ParseTypes(values);
        } else if (key.equal("limit")) {
            ParseLimit(values);
        } else if (key.equal("offset")) {
            ParseOffset(values);
        } else if (key.equal("sorted")) {
            ParseSorted(values);
        } else if (key.equal("hosts")) {
            ParseHosts(values);
        } else if (key.equal("signal_pattern")) {
            ParseSignalPattern(values);
        } else if (key.equal("tag_pattern")) {
            ParseTagPattern(values);
        } else if (key.equal("fields")) {
            ParseFields(values);
        } else if (key.equal("tag_format")) {
            ParseTagFormat(values);
        } else if (keyIsTag) {
            ParseTags(key, values);
        } else {
            throw TBadRequest() << "Unknown argument " << key;
        }
    }
}

void TCollectorHandleArgs::ParseItype(TVector<TString>& values) {
    for (auto& value: values) {
        if (!NTags::IsCorrectItype(value)) {
            throw TBadRequest() << "'itype' field has incorrect value '" << value << "'";
        }
        Itypes.insert(value);
    }
}

void TCollectorHandleArgs::ParseHostPattern(TVector<TString>& values) {
    if (values.size() != 1) {
        throw TBadRequest() << "'host_pattern' field should be string";
    }
    if (values[0].Contains("\"")) {
        throw TBadRequest() << "'host_pattern' field has incorrect value '" << values[0] << "'";
    }
    HostPattern = values[0];
}

void TCollectorHandleArgs::ParseSignals(TVector<TString>& values) {
    for (auto& value: values) {
        if (value.Contains("\"")) {
            throw TBadRequest() << "'signals' field has incorrect value '" << value << "'";
        }
    }
    Signals = std::move(values);
}

void TCollectorHandleArgs::ParseTypes(TVector<TString>& values) {
    if (values.empty()) {
        throw TBadRequest() << "'types' array should have at least one 'host', 'group' or 'metagroup'";
    }
    THostTypeArg types;
    for (auto& value : values) {
        if (value.equal("host")) {
            types |= EHostTypeArgFlags::HOST;
        } else if (value.equal("group")) {
            types |= EHostTypeArgFlags::GROUP;
        } else if (value.equal("metagroup")) {
            types |= EHostTypeArgFlags::METAGROUP;
        } else {
            throw TBadRequest() << "'types' field has incorrect value '" << value << "'";
        }
    }
    Types = types;
}

void TCollectorHandleArgs::ParseLimit(TVector<TString>& values) {
    if (values.size() != 1) {
        throw TBadRequest() << "'limit' field should be integer";
    }
    try {
        Limit = ::FromString<size_t>(values[0]);
    } catch (TFromStringException&) {
        throw TBadRequest() << "'limit' field has incorrect value '" << values[0] << "'";
    }
}

void TCollectorHandleArgs::ParseOffset(TVector<TString>& values) {
    if (values.size() != 1) {
        throw TBadRequest() << "'offset' field should be integer";
    }
    try {
        Offset = ::FromString<size_t>(values[0]);
    } catch (TFromStringException&) {
        throw TBadRequest() << "'offset' field has incorrect value '" << values[0] << "'";
    }
}

void TCollectorHandleArgs::ParseSorted(TVector<TString>& values) {
    if (values.size() != 1) {
        throw TBadRequest() << "'sorted' field should be 'asc' or 'desc'";
    }
    values[0].to_lower();
    if (values[0].equal("asc")) {
        Sorted = ESortType::ASC;
    } else if (values[0].equal("desc")) {
        Sorted = ESortType::DESC;
    } else {
        throw TBadRequest() << "'sorted' field has incorrect value '" << values[0] << "'";
    }
}

void TCollectorHandleArgs::ParseTags(const TString& key, TVector<TString>& values) {
    if (!NTags::IsCorrectTagName(key)) {
        throw TBadRequest() << "'" << key << "' is not correct tag name";
    }
    for (auto& value: values) {
        if (value.Contains("\"")) {
            throw TBadRequest() << "'" << key << "' tag has incorrect value '" << value << "'";
        }
    }
    Tags[key] = std::move(values);
}

void TCollectorHandleArgs::ParseHosts(TVector<TString>& values) {
    for (auto& value: values) {
        if (value.Contains("\"")) {
            throw TBadRequest() << "'hosts' field has incorrect value '" << value << "'";
        }
    }
    Hosts = std::move(values);
}

void TCollectorHandleArgs::ParseSignalPattern(TVector<TString>& values) {
    if (values.size() != 1) {
        throw TBadRequest() << "'signal_pattern' field should be string";
    }
    if (values[0].Contains("\"")) {
        throw TBadRequest() << "'signal_pattern' field has incorrect value '" << values[0] << "'";
    }
    SignalPattern = values[0];
}

void TCollectorHandleArgs::ParseTagPattern(TVector<TString>& values) {
    if (values.size() != 1) {
        throw TBadRequest() << "'tag_pattern' field should be string";
    }
    if (values[0].Contains("\"")) {
        throw TBadRequest() << "'tag_pattern' field has incorrect value '" << values[0] << "'";
    }
    TagPattern = values[0];
}

void TCollectorHandleArgs::ParseFields(TVector<TString>& values) {
    for (auto& value: values) {
        if (!NTags::IsCorrectTagName(value)) {
            throw TBadRequest() << "'" << value << "' is not correct tag name";
        }
    }
    Fields = std::move(values);
}

void TCollectorHandleArgs::ParseTagFormat(TVector<TString>& values) {
    if (values.size() != 1) {
        throw TBadRequest() << "'tag_format' field should be string";
    }
    values[0].to_lower();
    if (values[0].equal("unified")) {
        UnifiedTagFormat = true;
    } else {
        throw TBadRequest() << "'tag_format' field has incorrect value '" << values[0] << "'";
    }
}

NJson::TJsonValue THostsHandler::HandleParsedRequest(TBaseCollectorHandler::TParsedRequest requestData, const TVector<TString>&, TRequestLog& logger) {
    THostsHandleArgs args(requestData);
    THostsResult findHostsResult;

    if (args.Itypes.empty()) {
        // ignore most of args
        if (args.Types != EHostTypeArgFlags::METAGROUP) {
            throw TBadRequest() << "host request without itypes allowed only for metagroups";
        }
        for (auto& metagroup : Metagroups) {
            findHostsResult.Hosts.insert({metagroup, "metagroup"});
        }
        findHostsResult.Total = Metagroups.size();
    } else {
        TDataProxyReader reader(DataProxyMultiClusterState, logger);
        findHostsResult = reader.FindHosts(args.Itypes, args.Types, args.Tags, args.Signals, args.Limit,
                                           args.HostPattern, GroupToMetagroupMap, TInstant::Now() + TIMEOUT);
    }
    NJson::TJsonValue result(NJson::JSON_MAP);
    NJson::TJsonValue hostsArrayValue(NJson::JSON_ARRAY);
    for (auto& [name, type]: findHostsResult.Hosts) {
        NJson::TJsonValue hostValue(NJson::JSON_MAP);
        hostValue.InsertValue("name", name);
        hostValue.InsertValue("type", type);
        hostsArrayValue.AppendValue(std::move(hostValue));
    }

    result.InsertValue("result", std::move(hostsArrayValue));
    result.InsertValue("total", findHostsResult.Total);
    return result;
}

NJson::TJsonValue TSignalsHandler::HandleParsedRequest(TBaseCollectorHandler::TParsedRequest requestData, const TVector<TString>&, TRequestLog& logger) {
    TSignalsHandleArgs args(requestData);
    TDataProxyReader reader(DataProxyMultiClusterState, logger);
    NJson::TJsonValue result(NJson::JSON_MAP);
    auto findResult = reader.FindKeyValues(args.Itypes, "signal", args.Tags, args.Hosts, {}, args.SignalPattern,
                                           Metagroups, args.Limit, TInstant::Now() + TIMEOUT);

    NJson::TJsonValue signalsArray(NJson::JSON_ARRAY);
    for (auto& signal: findResult) {
        signalsArray.AppendValue(signal);
    }

    result.InsertValue("result", std::move(signalsArray));
    result.InsertValue("total", findResult.size());
    return result;
}

NJson::TJsonValue TTagsHandler::HandleParsedRequest(TBaseCollectorHandler::TParsedRequest requestData, const TVector<TString>&, TRequestLog& logger) {
    TTagsHandleArgs args(requestData);
    TDataProxyReader reader(DataProxyMultiClusterState, logger);
    NJson::TJsonValue result(NJson::JSON_MAP);
    TInstant deadline = TInstant::Now() + TIMEOUT;

    TSet<TString> tagsToRequest;
    bool needItype = false;
    if (args.Fields.empty()) {
        needItype = true;
        for (auto& [itype, tags] : reader.FindTagsForItypes(args.Itypes, deadline)) {
            for (auto& tag : tags) {
                tagsToRequest.insert(tag);
            }
        }
    } else {
        for (auto& tag : args.Fields) {
            if (tag.equal("itype")) {
                needItype = true;
            } else {
                tagsToRequest.insert(tag);
            }
        }
    }

    auto findResult = reader.FindUniqueLabels(args.Itypes, args.Tags, args.Hosts, args.Signals, tagsToRequest, needItype,
                                              Metagroups, deadline);

    NJson::TJsonValue tagsArray(NJson::JSON_ARRAY);

    for (auto& tags: findResult) {
        NJson::TJsonValue tagsMap(NJson::JSON_MAP);
        for (auto& [k, v] : tags) {
            tagsMap.InsertValue(k, v);
        }
        tagsArray.AppendValue(tagsMap);
    }

    result.InsertValue("result", std::move(tagsArray));
    result.InsertValue("total", findResult.size());
    return result;
}

NJson::TJsonValue TTagsItypeHandler::HandleParsedRequest(TBaseCollectorHandler::TParsedRequest requestData, const TVector<TString>&, TRequestLog&) {
    NJson::TJsonValue result(NJson::JSON_MAP);
    TTagsSpecificHandleArgs args(requestData, true);
    // ignore most of args
    TSet<TString> findResult = ItypeTagsLoader.GetItypes(args.Itypes, args.Limit);

    NJson::TJsonValue tagsArray(NJson::JSON_ARRAY);
    for (auto& tag: findResult) {
        tagsArray.AppendValue(tag);
    }

    result.InsertValue("result", std::move(tagsArray));
    result.InsertValue("total", findResult.size());
    return result;
}

NJson::TJsonValue TTagsSpecificHandler::HandleParsedRequest(TBaseCollectorHandler::TParsedRequest requestData, const TVector<TString>& pathGroups, TRequestLog& logger) {
    TTagsSpecificHandleArgs args(requestData, false);
    TDataProxyReader reader(DataProxyMultiClusterState, logger);
    NJson::TJsonValue result(NJson::JSON_MAP);
    TString tagName;
    if (pathGroups.size() == 1) {
        tagName = pathGroups[0];
    } else {
        throw TBadRequest() << "empty tag name";
    }
    auto findResult = reader.FindKeyValues(args.Itypes, tagName, args.Tags, args.Hosts, args.Signals, args.TagPattern,
                                           Metagroups, args.Limit, TInstant::Now() + TIMEOUT);

    NJson::TJsonValue tagsArray(NJson::JSON_ARRAY);
    for (auto& tag: findResult) {
        tagsArray.AppendValue(tag);
    }

    result.InsertValue("result", std::move(tagsArray));
    result.InsertValue("total", findResult.size());
    return result;
}

THashMap<TString, TString> TCollectorHandlersCollection::BuildGroupToMetagroupMap(const TMetagroupGroupsConfig& groupsConfig,
                                                                                  EStockpileClusterType clusterType)
{
    THashMap<TString, TString> result;
    for (auto& [metagroup, groups]: groupsConfig.GetMetagroupToGroupsListProdMap()) {
        for (auto& group: groups) {
            result[group] = metagroup;
        }
    }
    if (clusterType == EStockpileClusterType::Prestable) {
        for (auto& [metagroup, groups]: groupsConfig.GetMetagroupToGroupsListPrestableMap()) {
            for (auto& group: groups) {
                result[group] = metagroup;
            }
        }
    }
    return result;
}

TSet<TString> TCollectorHandlersCollection::InitMetagroups(const TMetagroupGroupsConfig& groupsConfig,
                                                           EStockpileClusterType clusterType)
{
    TSet<TString> result;
    for (auto&[metagroup, groups]: groupsConfig.GetMetagroupToGroupsListProdMap()) {
        result.insert(metagroup);
    }
    if (clusterType == EStockpileClusterType::Prestable) {
        for (auto&[metagroup, groups]: groupsConfig.GetMetagroupToGroupsListPrestableMap()) {
            result.insert(metagroup);
        }
    }
    return result;
}
