#pragma once

#include "settings.h"

#include <infra/monitoring/common/application.h>
#include <infra/yasm/common/groups/metagroup_groups.h>
#include <infra/yasm/collector/server/lib/dataproxy_reader.h>
#include <infra/yasm/collector/server/lib/itype_tags_loader.h>
#include <infra/yasm/collector/server/lib/metrics.h>
#include <infra/yasm/stockpile_client/state.h>

namespace NCollector {
    static const TString REQUEST_ID_FIELD = "request_id";

    class TPingHandler : public NMonitoring::IServiceReplier {
    public:
        explicit TPingHandler(TLog& logger)
            : Logger(logger) {
        }

        void DoReply(NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TLog& Logger;
    };

    class TSharedServiceReplier: public NMonitoring::IServiceReplier {
    public:
        explicit TSharedServiceReplier(TAtomicSharedPtr<NMonitoring::IServiceReplier> actualReplier)
            : ActualReplier(std::move(actualReplier)) {
        }

        void DoReply(NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override {
            ActualReplier->DoReply(request, meta);
        }

    private:
        TAtomicSharedPtr<NMonitoring::IServiceReplier> ActualReplier;
    };

    class TSharedServiceRegexpReplier: public NMonitoring::IServiceRegexpReplier {
    public:
        explicit TSharedServiceRegexpReplier(TAtomicSharedPtr<NMonitoring::IServiceRegexpReplier> actualReplier)
            : ActualReplier(std::move(actualReplier)) {
        }

        void DoReply(NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta,
            const TVector<TString>& pathGroups, const TMap<int, TString>& groupNames) override {
            ActualReplier->DoReply(request, meta, pathGroups, groupNames);
        }

    private:
        TAtomicSharedPtr<NMonitoring::IServiceRegexpReplier> ActualReplier;
    };

    class TBaseCollectorHandler: public NMonitoring::IServiceReplier, public NMonitoring::IServiceRegexpReplier {
    public:
        using TParsedRequest = TMap<TString, TVector<TString>>;

        explicit TBaseCollectorHandler(TLog& logger, const TStringBuf& handlerStatName)
            : Logger(logger)
            , HandlerTimeStatName(MakeTimeSignalName(handlerStatName))
            , Handler5xxStatName(Make5xxSignalName(handlerStatName))
            , Handler4xxStatName(Make4xxSignalName(handlerStatName)){
        }

        void DoReply(NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;
        void DoReply(NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta,
                     const TVector<TString>& pathGroups, const TMap<int, TString>& groupNames) override;
        virtual NJson::TJsonValue HandleParsedRequest(TParsedRequest requestData, const TVector<TString>& pathGroups, NMonitoring::TRequestLog& logger) = 0;

        static TParsedRequest ParseCollectorArgs(const TString& requestBody, const TParsedHttpFull& meta);

        void InitStatistics(const NMonitoring::TStatsInitializer& initializer, TUnistat& creator) const;

    private:
        static TString MakeTimeSignalName(const TStringBuf& handlerStatName);
        static TString Make5xxSignalName(const TStringBuf& handlerStatName);
        static TString Make4xxSignalName(const TStringBuf& handlerStatName);

        TLog& Logger;
        const TString HandlerTimeStatName;
        const TString Handler5xxStatName;
        const TString Handler4xxStatName;
    };

    struct TCollectorHandleArgs {
        TSet<TString> Itypes;

        TTagValuesMap Tags;
        TVector<TString> Parents;           // unused argument
        TMaybe<TString> HostPattern;
        TVector<TString> Signals;
        THostTypeArg Types;
        size_t Limit;
        size_t Offset;                      // unused argument
        ESortType Sorted;                   // unused argument
        TVector<TString> Hosts;
        TMaybe<TString> SignalPattern;
        TMaybe<TString> TagPattern;
        TVector<TString> Fields;
        bool UnifiedTagFormat;              // unused argument
    protected:
        TCollectorHandleArgs(TBaseCollectorHandler::TParsedRequest& requestData,
                             const TVector<TStringBuf>& requiredArgs, const TVector<TStringBuf>& optionalArgs, bool allowTags);
    private:
        void ParseItype(TVector<TString>& values);
        void ParseHostPattern(TVector<TString>& values);
        void ParseSignals(TVector<TString>& values);
        void ParseTypes(TVector<TString>& values);
        void ParseLimit(TVector<TString>& values);
        void ParseOffset(TVector<TString>& values);
        void ParseSorted(TVector<TString>& values);
        void ParseTags(const TString& key, TVector<TString>& values);
        void ParseHosts(TVector<TString>& values);
        void ParseSignalPattern(TVector<TString>& values);
        void ParseTagPattern(TVector<TString>& values);
        void ParseFields(TVector<TString>& values);
        void ParseTagFormat(TVector<TString>& values);
    };

    static const TVector<TStringBuf> FORBIDDEN_TAG_NAMES = {
        NHistDb::NStockpile::HOST_LABEL_KEY,
        NHistDb::NStockpile::GROUP_LABEL_KEY,
        NHistDb::NStockpile::SIGNAL_LABEL_KEY,
        NHistDb::NStockpile::PROJECT_LABEL_KEY,
        NHistDb::NStockpile::CLUSTER_LABEL_KEY,
        NHistDb::NStockpile::SERVICE_LABEL_KEY,
        TStringBuf("itype"),
        TStringBuf("limit"),
        TStringBuf("offset"),
        TStringBuf("sorted"),
        TStringBuf("parents"),
        TStringBuf("host_pattern"),
        TStringBuf("signals"),
        TStringBuf("types"),
        TStringBuf("hosts"),
        TStringBuf("signal_pattern"),
        TStringBuf("tag_pattern"),
        TStringBuf("fields"),
        TStringBuf("tag_format"),
    };
    static const TVector<TStringBuf> ITYPE_ARG = {"itype"};
    static const TVector<TStringBuf> HOSTS_HANDLE_OPT_ARGS = {
        TStringBuf("itype"),
        TStringBuf("parents"),
        TStringBuf("host_pattern"),
        TStringBuf("signals"),
        TStringBuf("types"),
        TStringBuf("limit"),
        TStringBuf("offset"),
        TStringBuf("sorted"),
    };
    static const TVector<TStringBuf> SIGNALS_HANDLE_OPT_ARGS = {
        TStringBuf("hosts"),
        TStringBuf("signal_pattern"),
        TStringBuf("limit"),
        TStringBuf("offset"),
        TStringBuf("sorted"),
    };
    static const TVector<TStringBuf> TAGS_HANDLE_OPT_ARGS = {
        TStringBuf("hosts"),
        TStringBuf("signals"),
        TStringBuf("fields"),
        TStringBuf("tag_format"),
        TStringBuf("limit"),
        TStringBuf("offset"),
        TStringBuf("sorted"),
    };
    static const TVector<TStringBuf> TAGS_SPECIFIC_HANDLE_OPT_ARGS = {
        TStringBuf("itype"),
        TStringBuf("tag_pattern"),
        TStringBuf("hosts"),
        TStringBuf("signals"),
        TStringBuf("limit"),
        TStringBuf("offset"),
        TStringBuf("sorted"),
    };

    struct TTagKeysHandleArgs : public TCollectorHandleArgs {
        TTagKeysHandleArgs(TBaseCollectorHandler::TParsedRequest& requestData)
            : TCollectorHandleArgs(requestData, {}, ITYPE_ARG, false) {
        }
    };
    struct THostsHandleArgs : public TCollectorHandleArgs {
        THostsHandleArgs(TBaseCollectorHandler::TParsedRequest& requestData)
            : TCollectorHandleArgs(requestData, {}, HOSTS_HANDLE_OPT_ARGS, true) {
        }
    };
    struct TSignalsHandleArgs : public TCollectorHandleArgs {
        TSignalsHandleArgs(TBaseCollectorHandler::TParsedRequest& requestData)
            : TCollectorHandleArgs(requestData, ITYPE_ARG, SIGNALS_HANDLE_OPT_ARGS, true) {
        }
    };
    struct TTagsHandleArgs : public TCollectorHandleArgs {
        TTagsHandleArgs(TBaseCollectorHandler::TParsedRequest& requestData)
            : TCollectorHandleArgs(requestData, ITYPE_ARG, TAGS_HANDLE_OPT_ARGS, true) {
        }
    };
    struct TTagsSpecificHandleArgs : public TCollectorHandleArgs {
        TTagsSpecificHandleArgs(TBaseCollectorHandler::TParsedRequest& requestData, bool itypeOptional)
            : TCollectorHandleArgs(requestData, itypeOptional ? TVector<TStringBuf>() : ITYPE_ARG, TAGS_SPECIFIC_HANDLE_OPT_ARGS, true) {
        }
    };

    class TReadyHandler : public NMonitoring::IServiceReplier {
    public:
        explicit TReadyHandler(const TBackgroundItypeTagsLoader& itypeTagsLoader, TLog& logger)
            : ItypeTagsLoader(itypeTagsLoader)
            , Logger(logger) {
        }

        void DoReply(NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        const TBackgroundItypeTagsLoader& ItypeTagsLoader;
        TLog& Logger;
    };

    class TTagKeysHandler: public TBaseCollectorHandler {
    public:
        static constexpr TStringBuf SIGNAL_NAME_BASE = "handlers.tagkeys";
        explicit TTagKeysHandler(const TBackgroundItypeTagsLoader& itypeTagsLoader, TLog& logger)
            : TBaseCollectorHandler(logger, SIGNAL_NAME_BASE)
            , ItypeTagsLoader(itypeTagsLoader) {
        }

        NJson::TJsonValue HandleParsedRequest(TParsedRequest requestData, const TVector<TString>& pathGroups, NMonitoring::TRequestLog& logger) override;

    private:
        const TBackgroundItypeTagsLoader& ItypeTagsLoader;
    };

    class THostsHandler: public TBaseCollectorHandler {
    public:
        static constexpr TStringBuf SIGNAL_NAME_BASE = "handlers.hosts";
        static constexpr TDuration TIMEOUT = TDuration::Seconds(30);
        explicit THostsHandler(NHistDb::NStockpile::TDataProxyMultiClusterState& dataProxyState,
                               const THashMap<TString, TString>& groupToMetagroup,
                               const TSet<TString>& metagroups,
                               TLog& logger)
            : TBaseCollectorHandler(logger, SIGNAL_NAME_BASE)
            , DataProxyMultiClusterState(dataProxyState)
            , GroupToMetagroupMap(groupToMetagroup)
            , Metagroups(metagroups) {
        }

        NJson::TJsonValue HandleParsedRequest(TParsedRequest requestData, const TVector<TString>& pathGroups, NMonitoring::TRequestLog& logger) override;

    private:
        NHistDb::NStockpile::TDataProxyMultiClusterState& DataProxyMultiClusterState;
        const THashMap<TString, TString>& GroupToMetagroupMap;
        const TSet<TString>& Metagroups;
    };

    class TSignalsHandler: public TBaseCollectorHandler {
    public:
        static constexpr TStringBuf SIGNAL_NAME_BASE = "handlers.signals";
        static constexpr TDuration TIMEOUT = TDuration::Seconds(30);
        explicit TSignalsHandler(NHistDb::NStockpile::TDataProxyMultiClusterState& dataProxyState, const TSet<TString>& metagroups, TLog& logger)
            : TBaseCollectorHandler(logger, SIGNAL_NAME_BASE)
            , DataProxyMultiClusterState(dataProxyState)
            , Metagroups(metagroups) {
        }

        NJson::TJsonValue HandleParsedRequest(TParsedRequest requestData, const TVector<TString>& pathGroups, NMonitoring::TRequestLog& logger) override;

    private:
        NHistDb::NStockpile::TDataProxyMultiClusterState& DataProxyMultiClusterState;
        const TSet<TString>& Metagroups;
    };

    class TTagsHandler: public TBaseCollectorHandler {
    public:
        static constexpr TStringBuf SIGNAL_NAME_BASE = "handlers.tags";
        static constexpr TDuration TIMEOUT = TDuration::Seconds(30);
        explicit TTagsHandler(NHistDb::NStockpile::TDataProxyMultiClusterState& dataProxyState, const TSet<TString>& metagroups, TLog& logger)
            : TBaseCollectorHandler(logger, SIGNAL_NAME_BASE)
            , DataProxyMultiClusterState(dataProxyState)
            , Metagroups(metagroups) {
        }

        NJson::TJsonValue HandleParsedRequest(TParsedRequest requestData, const TVector<TString>& pathGroups, NMonitoring::TRequestLog& logger) override;

    private:
        NHistDb::NStockpile::TDataProxyMultiClusterState& DataProxyMultiClusterState;
        const TSet<TString>& Metagroups;
    };

    class TTagsItypeHandler: public TBaseCollectorHandler {
    public:
        static constexpr TStringBuf SIGNAL_NAME_BASE = "handlers.tags_specific";
        explicit TTagsItypeHandler(const TBackgroundItypeTagsLoader& itypeTagsLoader, TLog& logger)
            : TBaseCollectorHandler(logger, SIGNAL_NAME_BASE)
            , ItypeTagsLoader(itypeTagsLoader) {
        }

        NJson::TJsonValue HandleParsedRequest(TParsedRequest requestData, const TVector<TString>& pathGroups, NMonitoring::TRequestLog& logger) override;

    private:
        const TBackgroundItypeTagsLoader& ItypeTagsLoader;
    };

    class TTagsSpecificHandler: public TBaseCollectorHandler {
    public:
        static constexpr TStringBuf SIGNAL_NAME_BASE = "handlers.tags_specific";
        static constexpr TDuration TIMEOUT = TDuration::Seconds(30);
        explicit TTagsSpecificHandler(NHistDb::NStockpile::TDataProxyMultiClusterState& dataProxyState, const TSet<TString>& metagroups, TLog& logger)
            : TBaseCollectorHandler(logger, SIGNAL_NAME_BASE)
            , DataProxyMultiClusterState(dataProxyState)
            , Metagroups(metagroups) {
        }

        NJson::TJsonValue HandleParsedRequest(TParsedRequest requestData, const TVector<TString>& pathGroups, NMonitoring::TRequestLog& logger) override;

    private:
        NHistDb::NStockpile::TDataProxyMultiClusterState& DataProxyMultiClusterState;
        const TSet<TString>& Metagroups;
    };

    class TCollectorHandlersCollection {
    public:
        explicit TCollectorHandlersCollection(NHistDb::NStockpile::TDataProxyMultiClusterState& dataProxyState,
                                              const TBackgroundItypeTagsLoader& itypeTagsLoader,
                                              const TServerSettings& settings,
                                              const NYasm::NGroups::TMetagroupGroupsConfig& groupsConfig,
                                              TLog& logger)
            : DataProxyMultiClusterState(dataProxyState)
            , ItypeTagsLoader(itypeTagsLoader)
            , GroupToMetagroupMap(BuildGroupToMetagroupMap(groupsConfig, settings.GetDataProxyClusterType()))
            , Metagroups(InitMetagroups(groupsConfig, settings.GetDataProxyClusterType()))
            , ReadyHandler(MakeAtomicShared<TReadyHandler>(ItypeTagsLoader, logger))
            , TagKeysHandler(MakeAtomicShared<TTagKeysHandler>(ItypeTagsLoader, logger))
            , HostsHandler(MakeAtomicShared<THostsHandler>(DataProxyMultiClusterState, GroupToMetagroupMap, Metagroups, logger))
            , SignalsHandler(MakeAtomicShared<TSignalsHandler>(DataProxyMultiClusterState, Metagroups, logger))
            , TagsHandler(MakeAtomicShared<TTagsHandler>(DataProxyMultiClusterState, Metagroups, logger))
            , TagsItypeHandler(MakeAtomicShared<TTagsItypeHandler>(ItypeTagsLoader, logger))
            , TagsSpecificHandler(MakeAtomicShared<TTagsSpecificHandler>(DataProxyMultiClusterState, Metagroups, logger)) {

            // Debug handle. Checks that collector is running.
            Handlers.emplace_back("/ping", MakeHolder<TPingHandler>(logger));

            // ready handle checks that collector is ready to respond with meaningful data on all of it's handles.
            Handlers.emplace_back("/ready", MakeHolder<TSharedServiceReplier>(ReadyHandler));

            // tagkeys handler returns tags for itypes
            Handlers.emplace_back("/tagkeys", MakeHolder<TSharedServiceReplier>(TagKeysHandler));
            Handlers.emplace_back("/tagkeys/", MakeHolder<TSharedServiceReplier>(TagKeysHandler));

            // hosts handler searches for host names
            Handlers.emplace_back("/hosts", MakeHolder<TSharedServiceReplier>(HostsHandler));
            Handlers.emplace_back("/hosts/", MakeHolder<TSharedServiceReplier>(HostsHandler));

            // signals handler searches for signal names
            Handlers.emplace_back("/signals", MakeHolder<TSharedServiceReplier>(SignalsHandler));
            Handlers.emplace_back("/signals/", MakeHolder<TSharedServiceReplier>(SignalsHandler));

            // tags handler searches for tag value combinations
            Handlers.emplace_back("/tags", MakeHolder<TSharedServiceReplier>(TagsHandler));
            Handlers.emplace_back("/tags/", MakeHolder<TSharedServiceReplier>(TagsHandler));

            // tags/itype handler searches for itype values
            Handlers.emplace_back("/tags/itype", MakeHolder<TSharedServiceReplier>(TagsItypeHandler));
            Handlers.emplace_back("/tags/itype/", MakeHolder<TSharedServiceReplier>(TagsItypeHandler));

            // tags/<tag> handler searches for a specified tag's values
            HandlersWithRegexp.emplace_back("^/tags/([^/]+)/?$", "/tags/:tag", MakeHolder<TSharedServiceRegexpReplier>(TagsSpecificHandler));
        }

        void Register(NMonitoring::TWebServer& server) {
            for (const auto& pair : Handlers) {
                server.Add(pair.first, *pair.second);
            }
            for (const auto& endpoint : HandlersWithRegexp) {
                server.AddRegexp(endpoint.Regexp, endpoint.MetricPath, *endpoint.Replier);
            }
        }

        void InitStatistics(const NMonitoring::TStatsInitializer& initializer, TUnistat& creator) const {
            initializer.DrillHistogramHole(creator, NMetrics::HANDLERS_PING_TIME);
            initializer.DrillHistogramHole(creator, NMetrics::HANDLERS_READY_TIME);

            TagKeysHandler->InitStatistics(initializer, creator);
            HostsHandler->InitStatistics(initializer, creator);
            SignalsHandler->InitStatistics(initializer, creator);
            TagsHandler->InitStatistics(initializer, creator);
            TagsItypeHandler->InitStatistics(initializer, creator);
            TagsSpecificHandler->InitStatistics(initializer, creator);
        }

    private:
        struct TRegexpEndpoint {
            TRegexpEndpoint(TString regexp, TString endpoint, THolder<NMonitoring::IServiceRegexpReplier> replier)
                : Regexp(std::move(regexp))
                , MetricPath(std::move(endpoint))
                , Replier(std::move(replier))
            {
            }

            TString Regexp;
            TString MetricPath;
            THolder<NMonitoring::IServiceRegexpReplier> Replier;
        };
        static THashMap<TString, TString> BuildGroupToMetagroupMap(const NYasm::NGroups::TMetagroupGroupsConfig& groupsConfig,
                                                                   NHistDb::NStockpile::EStockpileClusterType clusterType);
        static TSet<TString> InitMetagroups(const NYasm::NGroups::TMetagroupGroupsConfig& groupsConfig,
                                            NHistDb::NStockpile::EStockpileClusterType clusterType);

        NHistDb::NStockpile::TDataProxyMultiClusterState& DataProxyMultiClusterState;
        const TBackgroundItypeTagsLoader& ItypeTagsLoader;
        const THashMap<TString, TString> GroupToMetagroupMap;
        const TSet<TString> Metagroups;

        TAtomicSharedPtr<TReadyHandler> ReadyHandler;
        TAtomicSharedPtr<TTagKeysHandler> TagKeysHandler;
        TAtomicSharedPtr<THostsHandler> HostsHandler;
        TAtomicSharedPtr<TSignalsHandler> SignalsHandler;
        TAtomicSharedPtr<TTagsHandler> TagsHandler;
        TAtomicSharedPtr<TTagsItypeHandler> TagsItypeHandler;
        TAtomicSharedPtr<TTagsSpecificHandler> TagsSpecificHandler;

        TVector<std::pair<TString, THolder<NMonitoring::IServiceReplier>>> Handlers;
        TVector<TRegexpEndpoint> HandlersWithRegexp;
    };
}
