#include "handlers.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NCollector;
using namespace NMonitoring;

void CheckArgs(const TCollectorHandleArgs& args, TSet<TString> itypes, TTagValuesMap tags = {}, TVector<TString>parents = {},
               TMaybe<TString> hostPattern = Nothing(), TVector<TString> signals = {}, THostTypeArg types = EHostTypeArgFlags::HOST,
               size_t limit = 10000, size_t offset = 0, ESortType sorted = ESortType::ASC, TVector<TString> hosts = {},
               TMaybe<TString> signalPattern = Nothing(), TMaybe<TString> tagPattern = Nothing(), TVector<TString> fields = {},
               bool unifiedTagFormat = false) {
    UNIT_ASSERT_EQUAL(args.Itypes, itypes);
    UNIT_ASSERT_EQUAL(args.Tags, tags);
    UNIT_ASSERT_EQUAL(args.Parents, parents);
    UNIT_ASSERT_EQUAL(args.HostPattern, hostPattern);
    UNIT_ASSERT_EQUAL(args.Signals, signals);
    UNIT_ASSERT_EQUAL(args.Types, types);
    UNIT_ASSERT_EQUAL(args.Limit, limit);
    UNIT_ASSERT_EQUAL(args.Offset, offset);
    UNIT_ASSERT_EQUAL(args.Sorted, sorted);
    UNIT_ASSERT_EQUAL(args.Hosts, hosts);
    UNIT_ASSERT_EQUAL(args.SignalPattern, signalPattern);
    UNIT_ASSERT_EQUAL(args.TagPattern, tagPattern);
    UNIT_ASSERT_EQUAL(args.Fields, fields);
    UNIT_ASSERT_EQUAL(args.UnifiedTagFormat, unifiedTagFormat);
}

template <typename T>
void ThrowsBadRequest(const TString& key, const TVector<TString>& values, bool itypeDefined = true) {
    TBaseCollectorHandler::TParsedRequest requestData = {};
    if (itypeDefined) {
        requestData.insert({"itype", {"yasmagent"}});
    }
    requestData[key] = values;
    UNIT_ASSERT_EXCEPTION(T(requestData), TBadRequest);
}

template <typename T>
void ShouldNotBeEmpty(const TString& key) {
    ThrowsBadRequest<T>(key, {});
}

template <typename T>
void ShouldHaveOneValue(const TString& key) {
    ThrowsBadRequest<T>(key, {});
    ThrowsBadRequest<T>(key, {"one", "two"});
}

Y_UNIT_TEST_SUITE(THostsHandleArgsTest) {

    Y_UNIT_TEST(DefaultValues) {
        TBaseCollectorHandler::TParsedRequest requestData = {
            {
                "itype", {"yasmagent"},
            },
        };
        CheckArgs(THostsHandleArgs(requestData), {"yasmagent"});
    }

    Y_UNIT_TEST(ArgsParsing) {
        TBaseCollectorHandler::TParsedRequest requestData = {
            {
                "itype", {"yasmagent", "yasmambry"},
            },
            {
                "ctype", {"prod"},
            },
            {
                "geo", {"man", "sas"},
            },
            {
                "parents", {"ASEARCH"},
            },
            {
                "host_pattern", {"sas"},
            },
            {
                "signals", {"counter-instance_tmmv", "signal1"},
            },
            {
                "types", {"group", "metagroup"},
            },
            {
                "limit", {"100"},
            },
            {
                "offset", {"10"},
            },
            {
                "sorted", {"desc"},
            },
        };
        CheckArgs(THostsHandleArgs(requestData), {"yasmagent", "yasmambry"},
                  {{"ctype", {"prod"}}, {"geo", {"man", "sas"}}},
                  {"ASEARCH"},
                  "sas",
                  {"counter-instance_tmmv", "signal1"},
                  EHostTypeArgFlags::GROUP | EHostTypeArgFlags::METAGROUP,
                  100, 10, ESortType::DESC
        );
    }

    Y_UNIT_TEST(Validation) {
        ThrowsBadRequest<THostsHandleArgs>("itype", {"bad#itype"});

        ThrowsBadRequest<THostsHandleArgs>("geo", {"bad\"value"});
        ThrowsBadRequest<THostsHandleArgs>("geo#", {"bad_key"});

        ShouldHaveOneValue<THostsHandleArgs>("host_pattern");

        ThrowsBadRequest<THostsHandleArgs>("signals", {"bad\"signal"});

        ShouldNotBeEmpty<THostsHandleArgs>("types");
        ThrowsBadRequest<THostsHandleArgs>("types", {"bad_value"});

        ShouldHaveOneValue<THostsHandleArgs>("limit");
        ThrowsBadRequest<THostsHandleArgs>("limit", {"bad_value"});
        ThrowsBadRequest<THostsHandleArgs>("limit", {"-1"});

        ShouldHaveOneValue<THostsHandleArgs>("offset");
        ThrowsBadRequest<THostsHandleArgs>("offset", {"bad_value"});
        ThrowsBadRequest<THostsHandleArgs>("offset", {"-1"});

        ShouldHaveOneValue<THostsHandleArgs>("sorted");
        ThrowsBadRequest<THostsHandleArgs>("sorted", {"bad_value"});
    }
}

Y_UNIT_TEST_SUITE(TSignalsHandleArgsTest) {

    Y_UNIT_TEST(ItypeRequired) {
        TBaseCollectorHandler::TParsedRequest requestData1;
        UNIT_ASSERT_EXCEPTION(TSignalsHandleArgs(requestData1), TBadRequest);
    }

    Y_UNIT_TEST(DefaultValues) {
        TBaseCollectorHandler::TParsedRequest requestData = {
            {
                "itype", {"yasmagent"},
            },
        };
        CheckArgs(TSignalsHandleArgs(requestData), {"yasmagent"});
    }

    Y_UNIT_TEST(ArgsParsing) {
        TBaseCollectorHandler::TParsedRequest requestData = {
            {
                "itype", {"yasmagent", "yasmambry"},
            },
            {
                "ctype", {"prod"},
            },
            {
                "geo", {"man", "sas"},
            },
            {
                "signal_pattern", {"counter"},
            },
            {
                "limit", {"100"},
            },
            {
                "offset", {"10"},
            },
            {
                "sorted", {"desc"},
            },
        };
        CheckArgs(TSignalsHandleArgs(requestData), {"yasmagent", "yasmambry"},
                  {{"ctype", {"prod"}}, {"geo", {"man", "sas"}}},
                  {}, Nothing(), {}, EHostTypeArgFlags::HOST,
                  100, 10, ESortType::DESC, {}, "counter"
        );
    }

    Y_UNIT_TEST(Validation) {
        ShouldNotBeEmpty<TSignalsHandleArgs>("itype");
        ThrowsBadRequest<TSignalsHandleArgs>("itype", {"bad#itype"});

        ThrowsBadRequest<TSignalsHandleArgs>("geo", {"bad\"value"});
        ThrowsBadRequest<TSignalsHandleArgs>("geo#", {"bad_key"});

        ShouldHaveOneValue<TSignalsHandleArgs>("signal_pattern");

        ShouldHaveOneValue<TSignalsHandleArgs>("limit");
        ThrowsBadRequest<TSignalsHandleArgs>("limit", {"bad_value"});
        ThrowsBadRequest<TSignalsHandleArgs>("limit", {"-1"});

        ShouldHaveOneValue<TSignalsHandleArgs>("offset");
        ThrowsBadRequest<TSignalsHandleArgs>("offset", {"bad_value"});
        ThrowsBadRequest<TSignalsHandleArgs>("offset", {"-1"});

        ShouldHaveOneValue<TSignalsHandleArgs>("sorted");
        ThrowsBadRequest<TSignalsHandleArgs>("sorted", {"bad_value"});
    }
}
