#include "itype_tags_loader.h"
#include "metrics.h"
#include "dataproxy_reader.h"

#include <library/cpp/http/client/query.h>
#include <infra/monitoring/common/perf.h>
#include <infra/monitoring/common/web_server.h>
#include <infra/yasm/common/labels/tags/verification.h>

using namespace NCollector;
using namespace NMonitoring;
using namespace NHistDb::NStockpile;

void TBackgroundItypeTagsLoader::InitStatistics(const TStatsInitializer& initializer, TUnistat& creator) const {
    initializer.DrillAmmxHole(creator, NMetrics::TAGKEYS_CACHE_SIZE);
    initializer.DrillAmmxHole(creator, NMetrics::TAGKEYS_REMAINS_TO_INIT);

    initializer.DrillHistogramHole(creator, NMetrics::SOLOMON_API_REQUEST_TIME);
    initializer.DrillSummHole(creator, NMetrics::SOLOMON_API_ERROR);
}

THashMap<TString, TVector<TString>> TBackgroundItypeTagsLoader::GetTagKeys(const TSet<TString>& itypes) const {
    if (!AtomicGet(Ready)) {
        throw TServiceUnavailableError() << "not ready";
    }
    THashMap<TString, TVector<TString>> result;
    TLightReadGuard guard(Mutex);
    if (itypes.empty()) {
        result = TagKeysCache;
    } else {
        auto defaultTagKeys = TagKeysCache.find(DEFAULT_ITYPE);
        for (const auto& itype: itypes) {
            auto it = TagKeysCache.find(itype);
            if (it != TagKeysCache.end()) {
                result[itype] = it->second;
            } else if (defaultTagKeys != TagKeysCache.end()) {
                result[itype] = defaultTagKeys->second;
            } else {
                result[itype] = {};
            }
        }
    }
    return result;
}

TSet<TString> TBackgroundItypeTagsLoader::GetItypes(const TSet<TString>& requestedItypes, size_t limit) const {
    if (!AtomicGet(Ready)) {
        throw TServiceUnavailableError() << "not ready";
    }
    TSet<TString> result;
    size_t count = 0;
    TLightReadGuard guard(Mutex);
    for (const auto& itype : ItypesCache) {
        if (!requestedItypes.empty() && !requestedItypes.contains(itype)) {
            continue;
        }
        result.insert(itype);
        if (limit && ++count >= limit) {
            break;
        }
    }
    return result;
}

bool TBackgroundItypeTagsLoader::IsReady() const {
    return AtomicGet(Ready);
}

void TBackgroundItypeTagsLoader::Start() {
    ItypeTagsFetchJob = NMonitoring::StartPeriodicJob(
        [this]() {
            Tick();
        },
        {TAGS_UPDATE_MIN_INTERVAL, true, TAGS_UPDATE_JITTER}
    );
}

void TBackgroundItypeTagsLoader::Stop() {
    ItypeTagsFetchJob.Reset();
}

void TBackgroundItypeTagsLoader::Tick() {
    if (!AtomicGet(Ready)) {
        TRequestLog iterationLog(Logger, "tagkeys loader initialization");
        while (!FetchItypesFromMetabaseProjects(iterationLog)) {
            Sleep(TDuration::Seconds(5));
        }
        TUnistat::Instance().PushSignalUnsafe(NMetrics::TAGKEYS_REMAINS_TO_INIT, ItypesCache.size());
        for (size_t offset = 0; offset < ItypesCache.size(); offset += DATAPROXY_ITYPES_REQUEST_MAX_SIZE) {
            while (!FetchTagKeysFromDataproxy(offset, iterationLog)) {
                Sleep(TDuration::Seconds(1));
            }
            TUnistat::Instance().PushSignalUnsafe(NMetrics::TAGKEYS_REMAINS_TO_INIT, ItypesCache.size() - TagKeysCache.size());
        }
        AtomicSet(Ready, true);
    } else {
        TRequestLog iterationLog(Logger, TStringBuilder() << "tagkeys loader iteration " << TInstant::Now().Seconds());
        FetchItypesFromMetabaseProjects(iterationLog);
        FetchTagKeysFromDataproxy(CurrentOffset, iterationLog);
        CurrentOffset += DATAPROXY_ITYPES_REQUEST_MAX_SIZE;
        if (CurrentOffset >= ItypesCache.size()) {
            CurrentOffset = 0;
        }
    }
}

bool TBackgroundItypeTagsLoader::FetchItypesFromMetabaseProjects(TRequestLog& iterationLog) {
    bool success = false;
    try {
        TVector<TString> itypes;
        auto response = RequestSolomonApi(iterationLog);

        if (response->Success()) {
            NJson::TJsonValue responseJson;
            NJson::ReadJsonTree(response->Data, &responseJson, true);

            auto& array = responseJson.GetArraySafe();
            itypes.reserve(array.size());
            for (const auto& item : array) {
                const TString& projectId = item.GetMapSafe().at(TStringBuf("id")).GetStringSafe();
                if (projectId && projectId.StartsWith(STOCKPILE_YASM_PROJECTS_PREFIX)) {
                    const TString& itype = projectId.substr(STOCKPILE_YASM_PROJECTS_PREFIX.size());
                    if (!NTags::IsCorrectItype(itype)) {
                        iterationLog << TLOG_NOTICE << "Got bad itype " << itype;
                        continue;
                    }
                    itypes.push_back(itype);
                }
            }
            TLightWriteGuard guard(Mutex);
            if (ItypesCache != itypes) {
                iterationLog << TLOG_INFO << "Itypes cache updated. " << itypes.size() << " itypes loaded";
                ItypesCache = std::move(itypes);
                CurrentOffset = 0;
            }
            success = true;
        } else {
            iterationLog << TLOG_WARNING << "Error loading projects from Solomon API";
            TUnistat::Instance().PushSignalUnsafe(NMetrics::SOLOMON_API_ERROR, 1);
        }
    } catch (...) {
        iterationLog << TLOG_WARNING << "Can't load itypes: " << CurrentExceptionMessage();
        TUnistat::Instance().PushSignalUnsafe(NMetrics::SOLOMON_API_ERROR, 1);
    }
    return success;
}

bool TBackgroundItypeTagsLoader::FetchTagKeysFromDataproxy(size_t offset, TRequestLog& iterationLog) {
    bool success = false;
    try {
        TDataProxyReader reader(DataProxyMultiClusterState, iterationLog);
        size_t begin = Min(offset, ItypesCache.size());
        size_t end = Min(offset + DATAPROXY_ITYPES_REQUEST_MAX_SIZE, ItypesCache.size());
        iterationLog << TLOG_INFO << "Updating tagkeys cache for itypes " << begin << "-" << end - 1 << " (of " << ItypesCache.size() << ")";
        TSet<TString> itypesChunk(ItypesCache.cbegin() + begin, ItypesCache.cbegin() + end);
        if (!itypesChunk.empty()) {
            THashMap<TString, TVector<TString>> findResult = reader.FindTagsForItypes(itypesChunk, TInstant::Now() + DATAPROXY_TIMEOUT);
            TLightWriteGuard guard(Mutex);
            for (auto& [itype, tags]: findResult) {
                TagKeysCache[itype] = std::move(tags);
            }
        }
        success = true;
    } catch (...) {
        iterationLog << TLOG_WARNING << "Can't load tagkeys for itypes: " << CurrentExceptionMessage();
    }
    TUnistat::Instance().PushSignalUnsafe(NMetrics::TAGKEYS_CACHE_SIZE, TagKeysCache.size());
    return success;
}

NHttpFetcher::TResultRef TBackgroundItypeTagsLoader::RequestSolomonApi(TRequestLog& iterationLog) const {
    NHttp::TFetchOptions options{};
    options.SetOAuthToken(AuthToken);
    options.SetContentType({"application/json"});
    options.SetUserAgent(YASM_USER_AGENT);
    options.SetTimeout(SOLOMON_API_TIMEOUT);
    NHttp::TFetchQuery query{ApiProjectsUrl, options};

    iterationLog << TLOG_INFO << "Fetching projects from " << ApiProjectsUrl;
    TMeasuredMethod perf(iterationLog, NMetrics::SOLOMON_API_REQUEST_TIME, NMetrics::SOLOMON_API_REQUEST_TIME);
    return NHttp::Fetch(query);
}
