#pragma once

#include <library/cpp/http/client/fetch/fetch_result.h>
#include <infra/monitoring/common/logging.h>
#include <infra/monitoring/common/metrics.h>
#include <infra/yasm/collector/server/lib/metrics.h>
#include <infra/yasm/stockpile_client/common/base_types.h>
#include <infra/yasm/stockpile_client/metabase_shard_manager.h>
#include <infra/yasm/stockpile_client/state.h>

namespace NCollector {
    static const TString DEFAULT_ITYPE = "default";

    class TBackgroundItypeTagsLoader {
    public:
        static constexpr TDuration SOLOMON_API_TIMEOUT = TDuration::Seconds(10);
        static constexpr TDuration DATAPROXY_TIMEOUT = TDuration::Seconds(10);
        static constexpr TDuration TAGS_UPDATE_MIN_INTERVAL = TDuration::Seconds(55);
        static constexpr TDuration TAGS_UPDATE_JITTER = TDuration::Seconds(10);
        static constexpr size_t DATAPROXY_ITYPES_REQUEST_MAX_SIZE = 100;

        TBackgroundItypeTagsLoader(NHistDb::NStockpile::TDataProxyMultiClusterState& dataProxyState,
                                   NHistDb::NStockpile::EStockpileClusterType clusterType,
                                   TLog &logger)
            : DataProxyMultiClusterState(dataProxyState)
            , CurrentOffset(0)
            , ApiProjectsUrl(TStringBuilder() << DetectBaseUrl(clusterType) << "/projects?text="
                                              << NHistDb::NStockpile::STOCKPILE_YASM_PROJECTS_PREFIX)
            , AuthToken(GetEnv("OAUTH_TOKEN", ""))
            , Logger(logger)
            , Ready(false) {
            if (AuthToken.empty()) {
                throw yexception() << "no OAuth token provided, can't perform stockpile API request";
            }
        }

        void InitStatistics(const NMonitoring::TStatsInitializer& initializer, TUnistat& creator) const;
        THashMap<TString, TVector<TString>> GetTagKeys(const TSet<TString>& itypes) const;
        TSet<TString> GetItypes(const TSet<TString> &requestedItypes, size_t limit) const;

        void Start();
        void Stop();
        bool IsReady() const;

    private:
        void Tick();
        bool FetchItypesFromMetabaseProjects(NMonitoring::TRequestLog& iterationLog);
        bool FetchTagKeysFromDataproxy(size_t offset, NMonitoring::TRequestLog& iterationLog);
        NHttpFetcher::TResultRef RequestSolomonApi(NMonitoring::TRequestLog& iterationLog) const;

        NHistDb::NStockpile::TDataProxyMultiClusterState& DataProxyMultiClusterState;
        TLightRWLock Mutex;
        TVector<TString> ItypesCache;
        THashMap<TString, TVector<TString>> TagKeysCache;
        size_t CurrentOffset;
        const TString ApiProjectsUrl;
        const TString AuthToken;
        TLog& Logger;
        TAtomic Ready;

        NMonitoring::TPeriodicTaskPtr ItypeTagsFetchJob;
    };

}
