#include <infra/monitoring/common/application.h>
#include <infra/yasm/collector/server/lib/handlers.h>
#include <infra/yasm/collector/server/lib/settings.h>
#include <infra/yasm/common/groups/metagroup_groups.h>
#include <infra/yasm/stockpile_client/state.h>
#include <infra/yasm/stockpile_client/rpc.h>
#include <infra/yasm/stockpile_client/metrics.h>

using namespace NMonitoring;
using namespace NCollector;

class TCollectorStatsInitializer final : public TStatsInitializer {
public:
    explicit TCollectorStatsInitializer(const TCollectorHandlersCollection& handlersCollection,
                                        const TBackgroundItypeTagsLoader& itypeTagsLoader)
        : HandlersCollection(handlersCollection)
        , ItypeTagsLoader(itypeTagsLoader) {
    }

    void Init(TUnistat& creator) const override {
        TStatsInitializer::Init(creator);

        HandlersCollection.InitStatistics(*this, creator);
        ItypeTagsLoader.InitStatistics(*this, creator);

        NHistDb::NStockpile::NMetrics::TStockpileStatsInitializer stockpileStatsInitializer;
        stockpileStatsInitializer.InitGrpcRequestMetrics(creator, NMetrics::DATAPROXY_LABEL_KEYS);
        stockpileStatsInitializer.InitGrpcRequestMetrics(creator, NMetrics::DATAPROXY_LABEL_VALUES);
        stockpileStatsInitializer.InitGrpcRequestMetrics(creator, NMetrics::DATAPROXY_UNIQUE_LABELS);
    }

private:
    const TCollectorHandlersCollection& HandlersCollection;
    const TBackgroundItypeTagsLoader& ItypeTagsLoader;
};

class TCollectorServer final : public TBaseApplication<TServerSettings> {
public:
    TCollectorServer(TLog& logger, const TServerSettings& settings)
        : TBaseApplication(logger, settings)
        , DataProxyMultiClusterState(settings.GetDataProxyClusterType(), logger)
        , ItypeTagsLoader(DataProxyMultiClusterState, settings.GetDataProxyClusterType(), logger)
        , MetagroupGroupsConfig()
        , HttpHandlers(DataProxyMultiClusterState, ItypeTagsLoader, settings, MetagroupGroupsConfig, logger)
    {
        NHistDb::NStockpile::TGrpcSettings::Init("yasmcollector", logger, true);
    }

    void Run() override {
        DataProxyMultiClusterState.Start();
        ItypeTagsLoader.Start();
        HttpHandlers.Register(HttpServer);
        HttpServer.StartServing();
        Event.Wait();
    }

    void Stop(int signal) override {
        TBaseApplication::Stop(signal);
        ItypeTagsLoader.Stop();
        DataProxyMultiClusterState.Stop();
    }

    static void FillParser(NLastGetopt::TOpts& opts) {
        TServerSettings::FillParser(opts);
    }

    static void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed, TServerSettings& settings) {
        settings.ProcessParsedOptions(parsed);
        settings.SetEnableCompression(true);
        settings.SetClientTimeout(TDuration::Seconds(10));
    }

    THolder<TStatsInitializer> GetStatsInitializer() override {
        return MakeHolder<TCollectorStatsInitializer>(HttpHandlers, ItypeTagsLoader);
    }

private:
    NHistDb::NStockpile::TDataProxyMultiClusterState DataProxyMultiClusterState;
    TBackgroundItypeTagsLoader ItypeTagsLoader;
    NYasm::NGroups::TMetagroupGroupsConfig MetagroupGroupsConfig;
    TCollectorHandlersCollection HttpHandlers;
};

int main(int argc, const char** argv) {
    return RunFromMain<TCollectorServer, TServerSettings, TThreadedLoggerFactory>(argc, argv);
}
