#include <infra/yasm/common/config/fast_config.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/logger/global/global.h>

#include <util/folder/path.h>
#include <util/folder/dirut.h>
#include <util/string/strip.h>
#include <util/system/env.h>
#include <util/system/shellcommand.h>

static const TString LOCAL_FAST_CONFIG_PATH {"fast_config.json"};
static const TString LOCAL_TEST_FAST_CONFIG_PATH {"test_fast_config.json"};

class TFastConfigHelper: public NYasm::NCommon::TFastConfig {
public:
    TFastConfigHelper(TLog& logger, NYasm::NCommon::TFastConfigSettings settings)
        : NYasm::NCommon::TFastConfig(logger, settings)
    {
        CreateClient();
    }

    void Fetch() {
        Tick();
    };

    void UploadLocalFile() {
        Upload(ReadFromDisk());
    }
};

class TFastConfigManager {
public:
    TFastConfigManager(const TStringBuf oauthToken, bool useTestConfig)
        : UseTestConfig(useTestConfig)
        , Logger(TLog())
        , Helper(Logger, NYasm::NCommon::TFastConfigSettings {
            TString{oauthToken},
            NYasm::NCommon::YASM_YT_CLUSTER,
            (useTestConfig) ? NYasm::NCommon::YASM_YT_FAST_CONFIG_TEST_PATH : NYasm::NCommon::YASM_YT_FAST_CONFIG_PATH,
            (useTestConfig) ? LOCAL_TEST_FAST_CONFIG_PATH : LOCAL_FAST_CONFIG_PATH
        }){
    };

    void Fetch() {
        Helper.Fetch();
    }

    void Commit() {
        if (!UseTestConfig) {
            Exec("svn", {"up"});
        }
        Helper.UploadLocalFile();
        if (!UseTestConfig) {
            Exec("svn", {
                "commit", "-m",
                R"("[YASM]: fast config flags modification, see wiki.y-t.ru/golovan/fast-config SKIP_REVIEW SKIP_CHECK")"
            });
        }
    }

private:
    void Exec(const TStringBuf cmd, const TList<TString>& args) {
        TShellCommandOptions options{};
        options.OutputStream = &Cout;
        options.ErrorStream = &Cerr;
        TShellCommand shellCommand(cmd, args, options);
        if (shellCommand.Run().Wait().GetStatus() != TShellCommand::SHELL_FINISHED) {
            ythrow yexception() << "shell shellCommand " << cmd << " finished with errors: " << shellCommand.GetError();
        }
    }

    const bool UseTestConfig;
    TLog Logger;
    TFastConfigHelper Helper;
};

TString findYtOauthToken() {
    TString ytToken {GetEnv("YT_TOKEN")};
    if (ytToken.empty()) {
        try {
            return StripString(TFileInput(GetHomeDir() + "/.yt/token").ReadAll());
        } catch (...) {
            ythrow yexception() << "You should save oauth token in YT_TOKEN environment variable or in ~/.yt/token";
        }
    }
    return ytToken;
}

int main(int argc, const char** argv) {
    using namespace NLastGetopt;
    TOpts options{TOpts::Default()};

    options.SetTitle(
        "Utility for manipulate YASM fast config.\n"
        "Work pipeline:\n"
        "  > ya make && ./fast-config fetch && vim fast_config.json && ./fast-config commit\n"
        "Do not forget to set up YT_TOKEN environment variable or save in ~/.yt/token (https://oauth.yt.yandex.net).\n"
        "https://wiki.yandex-team.ru/golovan/fast-config/ for details"
    );
    options.AddHelpOption();
    const TString testConfigOption = "test-config";
    options.AddLongOption(testConfigOption, "Operate on test-config file only.").Optional().HasArg(NLastGetopt::NO_ARGUMENT);
    options.SetFreeArgsNum(1);
    options.SetFreeArgTitle(0, "command", (
            "Valid commands: \n"
            "    fetch - fetch actual config from YT\n"
            "    commit - upload local config to YT and commit it to SVN"
    ));

    InitGlobalLog2Console();

    TOptsParseResult parsedOptions(&options, argc, argv);

    size_t freeArgsPos = parsedOptions.GetFreeArgsPos();
    argc -= freeArgsPos;
    argv += freeArgsPos;

    try {
        TFastConfigManager fastConfigManager(findYtOauthToken(), parsedOptions.Has(testConfigOption));
        if (stricmp(*argv, "fetch") == 0) {
            fastConfigManager.Fetch();
        } else if (stricmp(*argv, "commit") == 0) {
            fastConfigManager.Commit();
        } else {
            Cerr << "ERROR: '" << *argv << "' is invalid command (you should use 'fetch' or 'commit')" << Endl;
            return EXIT_FAILURE;
        }
    } catch (std::exception) {
        Cerr << "ERROR: " << CurrentExceptionMessage() << Endl;
        return EXIT_FAILURE;
    } catch (...) {
        Cerr << "ERROR: Unknown exception" << Endl;
        return EXIT_FAILURE;
    }
}
