#include "fast_config.h"

#include <infra/yasm/common/labels/tags/request_key.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/generic/algorithm.h>
#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/random/random.h>

using namespace NYasm::NCommon;

namespace {
    struct TFastConfigLookupTables {
        TMap<NYasm::NFastConfig::EFlag, EFastConfigFlag> LegacyFlagToFlag;
        TMap<EFastConfigFlag, TString> FlagToName;
        TMap<TString, EFastConfigFlag> NameToFlag;

        static TFastConfigLookupTables Make() {
            TFastConfigLookupTables result;
            for (const auto&[flag, name, legacyFlag]: FlagWithNameAndLegacyFlag) {
                if (legacyFlag.Defined()) {
                    if (!result.LegacyFlagToFlag.emplace(*legacyFlag, flag).second) {
                        Y_FAIL("Duplicated fastconfig legacy flag");
                    }
                }
                if (!result.FlagToName.emplace(flag, name).second) {
                    Y_FAIL("Duplicated fastconfig flag");
                }
                if (!result.NameToFlag.emplace(name, flag).second) {
                    Y_FAIL("Duplicated fastconfig flag name");
                }
            }
            return result;
        };

        static const TFastConfigLookupTables& Instance() {
            static const TFastConfigLookupTables table = Make();
            return table;
        }

        EFastConfigFlag GetFlagByLegacyFlag(NYasm::NFastConfig::EFlag legacyFlag) const {
            return LegacyFlagToFlag.at(legacyFlag);
        }

        TMaybe<EFastConfigFlag> GetFlagByName(const TString& flagName) const {
            TMaybe<EFastConfigFlag> result;
            const auto& nameToFlag = NameToFlag;
            auto it = nameToFlag.find(flagName);
            if (it != nameToFlag.end()) {
                result = it->second;
            }
            return result;
        }

        const TString& GetFlagName(EFastConfigFlag flag) const {
            return FlagToName.at(flag);
        }
    };
}

TFastConfigSettings::TFastConfigSettings(
    TString ytOauthToken,
    TString ytCluster,
    NYT::TYPath ytPath,
    TFsPath localPath,
    const NTags::TInstanceKey instanceKey
)
    : YtOauthToken(std::move(ytOauthToken))
    , YtCluster(std::move(ytCluster))
    , YtPath(std::move(ytPath))
    , LocalPath(std::move(localPath))
    , InstanceKey(instanceKey)
    , YtPathRevision(YtPath + "/@revision")
{
}

TFastConfig::TFastConfig(
    TLog& logger,
    TFastConfigSettings settings,
    TDuration interval
)
    : Settings(std::move(settings))
    , Logger(logger)
    , CurrentFlags{}
    , Interval{interval}
{
}

void TFastConfig::CreateClient() {
    Client = NYT::CreateClient(
        Settings.YtCluster,
        NYT::TCreateClientOptions().Token(Settings.YtOauthToken)
    );
}

void TFastConfig::Start() {
    CreateClient();
    try {
        try {
            ApplyConfig(ReadFromDisk());
        } catch (...) {
            Logger << TLOG_ERR << "Can't load fast config from file: " << CurrentExceptionMessage();
            Tick();
        }
    } catch (...) {
        Logger << TLOG_ERR << "Initial load fast config error: " << CurrentExceptionMessage();
    }
    auto startOffset = TDuration::MilliSeconds(RandomNumber(Interval.MilliSeconds()));
    FetchFastConfigJob = NMonitoring::StartPeriodicJob(
        [this]() {
            Tick();
        },
        {TInstant::Now() - startOffset, Interval}
    );
}

void TFastConfig::Stop() {
    FetchFastConfigJob.Reset();
}

void TFastConfig::Tick() {
    try {
        auto maybeNewConfig {GetFromYt()};
        if (maybeNewConfig.Defined()) {
            ApplyConfig(*maybeNewConfig);
            TOFStream stream {Settings.LocalPath};
            Serialize(*maybeNewConfig, stream);
            stream << "\n";
        }
    } catch (const yexception& e) {
        Logger << TLOG_ERR << "FastConfig fetch error:" << CurrentExceptionMessage();
    }
}

void TFastConfig::Upload(const TConfig& config) const {
    auto stream = Client->CreateFileWriter(Settings.YtPath);
    Serialize(config, *stream);
}

bool TFastConfig::Has(const EFastConfigFlag flag) const {
    TLightReadGuard rg(CurrentFlagsLock);
    return CurrentFlags.contains(flag);
}

TVector<TString> TFastConfig::Get(const EFastConfigFlag flag) const {
    TLightReadGuard rg(CurrentFlagsLock);
    return CurrentFlags.at(flag);
}

bool TFastConfig::Has(const TString& flagStr) const {
    auto flag = TFastConfigLookupTables::Instance().GetFlagByName(flagStr);
    return flag.Defined() && Has(*flag);
}

TMaybe<TConfig> TFastConfig::GetFromYt() {
    const auto newRevision = Client->Get(Settings.YtPathRevision).IntCast<TYtRevision>();
    if (newRevision != PrevRevisionFromYt) {
        PrevRevisionFromYt = newRevision;
        NYT::IFileReaderPtr reader = Client->CreateFileReader(Settings.YtPath);
        TConfig config {};
        NProtobufJson::Json2Proto(*reader, config);
        return {std::move(config)};
    } else {
        return {};
    }
}

TConfig TFastConfig::ReadFromDisk() const {
    TFileInput localFile {Settings.LocalPath};
    TConfig config {};
    NProtobufJson::Json2Proto(localFile, config);
    return config;
}

void TFastConfig::Serialize(const TConfig& fastConfigProto, IOutputStream& out) const {
    NProtobufJson::TProto2JsonConfig jsonConfig{};
    jsonConfig.SetFormatOutput(true);
    jsonConfig.SetAddMissingFields(true);
    jsonConfig.SetMissingSingleKeyMode(NProtobufJson::TProto2JsonConfig::MissingKeyDefault);
    jsonConfig.SetMissingRepeatedKeyMode(NProtobufJson::TProto2JsonConfig::MissingKeyDefault);
    jsonConfig.SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);
    NProtobufJson::Proto2Json(fastConfigProto, out, jsonConfig);
}

template<typename TOption>
void TFastConfig::EmplaceFlags(TFlagMap& flags, const TOption& options, EFastConfigFlag flag) {
    for(const auto& option: options) {
        if (Match(option.GetMatcher())) {
            flags.emplace(flag, TValuesVector());
        }
    }
}

template<typename TOption>
void TFastConfig::EmplaceFlagsWithValues(TFlagMap& flags, const TOption& options, EFastConfigFlag flag) {
    for(const auto& option: options) {
        if (Match(option.GetMatcher())) {
            flags.emplace(flag, TValuesVector(std::begin(option.GetValues()), std::end(option.GetValues())));
        }
    }
}

void TFastConfig::ApplyConfig(const TConfig& config) {
    TFlagMap flags{};

    // deprecated, backward compatibility
    for(const auto& option: config.GetOptions()) {
        if (Match(option.GetMatcher())) {
            flags.emplace(
                TFastConfigLookupTables::Instance().GetFlagByLegacyFlag(option.GetFlag()),
                TValuesVector(std::begin(option.GetValues()), std::end(option.GetValues()))
            );
        }
    }

    EmplaceFlagsWithValues(flags, config.GetNoneOptions(), EFastConfigFlag::NONE);
    EmplaceFlagsWithValues(flags, config.GetDisableStockpileDumpOptions(), EFastConfigFlag::DISABLE_STOCKPILE_DUMP);
    EmplaceFlagsWithValues(flags, config.GetDisableStockpileReadOptions(), EFastConfigFlag::DISABLE_STOCKPILE_READ);
    EmplaceFlagsWithValues(flags, config.GetBanItypeSubscriptionsOptions(), EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS);
    EmplaceFlagsWithValues(flags, config.GetMirrorHistdbReadsOptions(), EFastConfigFlag::MIRROR_HISTDB_READS);
    EmplaceFlags(flags, config.GetForceLegacyTypesConversion(), EFastConfigFlag::FORCE_LEGACY_TYPES_CONVERSION);
    EmplaceFlags(flags, config.GetDisableNewCollector(), EFastConfigFlag::DISABLE_NEW_COLLECTOR);
    EmplaceFlags(flags, config.GetIgnoreNewCollectorResponses(), EFastConfigFlag::IGNORE_NEW_COLLECTOR_RESPONSES);
    EmplaceFlags(flags, config.GetDisableNewCollectorStableHandles(), EFastConfigFlag::DISABLE_NEW_COLLECTOR_STABLE_HANDLES);

    TStringBuilder logString {};
    logString << "New FastConfig flags will be applied: ";
    for (const auto& [flag, values]: flags) {
        logString << TFastConfigLookupTables::Instance().GetFlagName(flag) << ": ";
        for (const auto& value: values) {
            logString << value << ", ";
        }
        logString << "; ";
    }
    Logger << ELogPriority::TLOG_INFO << logString;

    TLightWriteGuard rg(CurrentFlagsLock);
    CurrentFlags = flags;
}

bool TFastConfig::Match(const NFastConfig::TMatcher& matcher) const {
    auto requestKey {NTags::TRequestKey::FromString(matcher.GetRequestKey())};
    if (!requestKey.Match(Settings.InstanceKey)) {
        return false;
    }
    return (THash<TString>()(Settings.InstanceKey.ToNamed()) % 100) < matcher.GetPercent();
}
