#pragma once
#include <infra/yasm/common/config/proto/fast_config.pb.h>

#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/monitoring/common/periodic_task.h>

#include <mapreduce/yt/interface/client.h>

#include <library/cpp/logger/log.h>

#include <util/generic/set.h>
#include <util/folder/path.h>
#include <util/system/env.h>

namespace NYasm::NCommon {

    using TConfig = NFastConfig::TConfig;

    static const TString YASM_YT_CLUSTER {"locke"};
    static const TString YASM_YT_FAST_CONFIG_PATH {"//home/yasm/fast_config.json"};
    static const TString YASM_YT_FAST_CONFIG_TEST_PATH {"//home/yasm/test_fast_config.json"};

    struct TFastConfigSettings {
        TFastConfigSettings(
            TString YtOauthToken,
            TString YtCluster,
            NYT::TYPath YtPath,
            TFsPath LocalPath,
            NTags::TInstanceKey InstanceKey = NTags::TInstanceKey()
        );

        TString YtOauthToken;
        TString YtCluster;
        const NYT::TYPath YtPath;
        const TFsPath LocalPath;
        const NTags::TInstanceKey InstanceKey;
        const NYT::TYPath YtPathRevision;
    };

    inline TFastConfigSettings MakeYasmFastConfigSettings(const TFsPath& localPath) {
        return {
            GetEnv("YT_OAUTH_TOKEN"),
            YASM_YT_CLUSTER,
            YASM_YT_FAST_CONFIG_PATH,
            localPath,
            NTags::TInstanceKey::FromRtcEnviron()
        };
    }

    class TIncorrectFlagString : public yexception {};

    enum class EFastConfigFlag {
        NONE = 0,
        DISABLE_STOCKPILE_DUMP = 1,
        DISABLE_STOCKPILE_READ = 2,
        BAN_ITYPE_SUBSCRIPTIONS = 3,
        MIRROR_HISTDB_READS = 4,
        FORCE_LEGACY_TYPES_CONVERSION = 5,
        DISABLE_NEW_COLLECTOR = 6,
        IGNORE_NEW_COLLECTOR_RESPONSES = 7,
        DISABLE_NEW_COLLECTOR_STABLE_HANDLES = 8
    };

    const TVector<std::tuple<EFastConfigFlag, TString, TMaybe<NYasm::NFastConfig::EFlag>>> FlagWithNameAndLegacyFlag{
        {EFastConfigFlag::NONE, "NONE", NYasm::NFastConfig::EFlag::NONE},
        {EFastConfigFlag::DISABLE_STOCKPILE_DUMP, "DISABLE_STOCKPILE_DUMP", NYasm::NFastConfig::EFlag::DISABLE_STOCKPILE_DUMP},
        {EFastConfigFlag::DISABLE_STOCKPILE_READ, "DISABLE_STOCKPILE_READ", NYasm::NFastConfig::EFlag::DISABLE_STOCKPILE_READ},
        {EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS, "BAN_ITYPE_SUBSCRIPTIONS", NYasm::NFastConfig::EFlag::BAN_ITYPE_SUBSCRIPTIONS},
        {EFastConfigFlag::MIRROR_HISTDB_READS, "MIRROR_HISTDB_READS", NYasm::NFastConfig::EFlag::MIRROR_HISTDB_READS},
        {EFastConfigFlag::FORCE_LEGACY_TYPES_CONVERSION, "FORCE_LEGACY_TYPES_CONVERSION", Nothing()},
        {EFastConfigFlag::DISABLE_NEW_COLLECTOR, "DISABLE_NEW_COLLECTOR", Nothing()},
        {EFastConfigFlag::IGNORE_NEW_COLLECTOR_RESPONSES, "IGNORE_NEW_COLLECTOR_RESPONSES", Nothing()},
        {EFastConfigFlag::DISABLE_NEW_COLLECTOR_STABLE_HANDLES, "DISABLE_NEW_COLLECTOR_STABLE_HANDLES", Nothing()}
    };

    class TFastConfig {
    public:
        TFastConfig(
            TLog& logger,
            TFastConfigSettings settings,
            TDuration interval = DEFAULT_INTERVAL
        );
        virtual ~TFastConfig() = default;

        void Start();
        void Stop();
        bool Has(EFastConfigFlag flag) const;
        TVector<TString> Get(EFastConfigFlag flag) const;

        // TODO(rocco66) replace it to Enum usage after histfront will have migrated to arcadia
        bool Has(const TString& flagStr) const;

    protected:
        void Tick();
        TConfig ReadFromDisk() const;
        void Upload(const TConfig&) const;
        void CreateClient();
    private:
        using TValuesVector = TVector<TString>;
        using TFlagMap = THashMap<EFastConfigFlag, TVector<TString>>;
        template<typename TOption>
        void EmplaceFlags(TFlagMap& flags, const TOption& options, EFastConfigFlag flag);
        template<typename TOption>
        void EmplaceFlagsWithValues(TFlagMap& flags, const TOption& options, EFastConfigFlag flag);
        using TYtRevision = ui64; // yt revision attribute value will be cast to this type

        virtual TMaybe<TConfig> GetFromYt();
        void Serialize(const TConfig& fastConfigProto, IOutputStream& stream) const;
        void ApplyConfig(const TConfig& config);

        bool Match(const NFastConfig::TMatcher& matcher) const;

        static constexpr TDuration DEFAULT_INTERVAL = TDuration::Seconds(15);

        TFastConfigSettings Settings;
        NYT::IClientPtr Client;

        TMaybe<TYtRevision> PrevRevisionFromYt;

        TLog& Logger;

        TLightRWLock CurrentFlagsLock;
        TFlagMap CurrentFlags;

        NMonitoring::TPeriodicTaskPtr FetchFastConfigJob;
        const TDuration Interval;
    };
}
