#include "fast_config.h"
#include "fast_config_ut.h"

#include <library/cpp/logger/global/global.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/system/tempfile.h>

using namespace NYasm::NCommon;

Y_UNIT_TEST_SUITE(FastConfigTest) {
    void checkEmptyConfig(const TFastConfig& fastConfig) {
        UNIT_ASSERT(!fastConfig.Has(EFastConfigFlag::DISABLE_STOCKPILE_DUMP));
        UNIT_ASSERT(!fastConfig.Has(EFastConfigFlag::DISABLE_STOCKPILE_READ));
        UNIT_ASSERT(!fastConfig.Has(EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS));
    }

    void checkMockedConfig(const TFastConfig& fastConfig) {
        UNIT_ASSERT(fastConfig.Has(EFastConfigFlag::DISABLE_STOCKPILE_DUMP));
        UNIT_ASSERT(!fastConfig.Has(EFastConfigFlag::DISABLE_STOCKPILE_READ));
        UNIT_ASSERT(fastConfig.Has(EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS));
        UNIT_ASSERT_EQUAL(fastConfig.Get(EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS), TVector<TString>({"banneditype"}));
    }

    Y_UNIT_TEST(TestFetchAndMatchDeprecated) {
        TLog log {};
        TString localFileName {"fast_config.json"};
        TTempFile tempFile {localFileName};
        TFastConfigSettings settings {
            "oauth_token",
            "locke",
            "ytPath",
            localFileName,
            NTags::TInstanceKey::FromNamed("itype|ctype=prod")
        };
        NTest::TYtMockedFastConfig fastConfig {log, settings, NTest::MOCKED_CONFIG_STRING_DEPRECATED};
        fastConfig.Start();
        checkMockedConfig(fastConfig);
        fastConfig.Stop();

        //read from local file
        NTest::TYtBrokenMockedFastConfig brokenFastConfig {log, settings};
        brokenFastConfig.Start();
        UNIT_ASSERT(fastConfig.Has(EFastConfigFlag::DISABLE_STOCKPILE_DUMP));
        UNIT_ASSERT(!fastConfig.Has(EFastConfigFlag::DISABLE_STOCKPILE_READ));
        brokenFastConfig.Stop();
    }

    Y_UNIT_TEST(TestFetchAndMatchDeprecatedExtraField) {
        TLog log {};
        TString localFileName {"fast_config.json"};
        TTempFile tempFile {localFileName};
        TFastConfigSettings settings {
            "oauth_token",
            "locke",
            "ytPath",
            localFileName,
            NTags::TInstanceKey::FromNamed("itype|ctype=prod")
        };
        NTest::TYtMockedFastConfig fastConfig {log, settings, NTest::MOCKED_CONFIG_STRING_DEPRECATED_EXTRA_FIELDS};
        fastConfig.Start();
        checkMockedConfig(fastConfig);
        fastConfig.Stop();
    }

    Y_UNIT_TEST(TestFetchAndMatchActual) {
        TLog log {};
        TString localFileName {"fast_config.json"};
        TTempFile tempFile {localFileName};
        TFastConfigSettings settings {
            "oauth_token",
            "locke",
            "ytPath",
            localFileName,
            NTags::TInstanceKey::FromNamed("itype|ctype=prod")
        };
        NTest::TYtMockedFastConfig fastConfig {log, settings, NTest::MOCKED_CONFIG_STRING_ACTUAL};
        fastConfig.Start();
        checkMockedConfig(fastConfig);
        fastConfig.Stop();
    }

    Y_UNIT_TEST(TestUpdate) {
        TLog log {};
        TString localFileName {"fast_config.json"};
        TTempFile tempFile {localFileName};
        TFastConfigSettings settings {
            "oauth_token",
            "locke",
            "ytPath",
            localFileName,
            NTags::TInstanceKey::FromNamed("itype|ctype=prod")
        };
        NTest::TYtRefreshingFastConfig fastConfig {
            log, settings, TDuration::Hours(24),
            {NTest::SIMPLE_CONFIG_STRING, NTest::SIMPLE_CONFIG_STRING_WITH_OPTIONS, NTest::MOCKED_CONFIG_STRING_DEPRECATED, NTest::MOCKED_CONFIG_STRING_ACTUAL}
        };
        fastConfig.Start();
        checkEmptyConfig(fastConfig);
        fastConfig.CompulsoryTick();
        checkEmptyConfig(fastConfig);
        fastConfig.CompulsoryTick();
        checkMockedConfig(fastConfig);
        fastConfig.CompulsoryTick();
        checkMockedConfig(fastConfig);
        fastConfig.Stop();
    }

    Y_UNIT_TEST(TestUpdateChange) {
        TLog log {};
        TString localFileName {"fast_config.json"};
        TTempFile tempFile {localFileName};
        TFastConfigSettings settings {
            "oauth_token",
            "locke",
            "ytPath",
            localFileName,
            NTags::TInstanceKey::FromNamed("itype|ctype=prod")
        };
        NTest::TYtRefreshingFastConfig fastConfig {
            log, settings, TDuration::Hours(24),
            {NTest::MOCKED_CONFIG_STRING_DEPRECATED, NTest::MOCKED_CONFIG_STRING_ACTUAL_CHANGED}
        };
        fastConfig.Start();
        checkMockedConfig(fastConfig);
        fastConfig.CompulsoryTick();

        UNIT_ASSERT(!fastConfig.Has(EFastConfigFlag::DISABLE_STOCKPILE_DUMP));
        UNIT_ASSERT(fastConfig.Has(EFastConfigFlag::DISABLE_STOCKPILE_READ));
        UNIT_ASSERT(fastConfig.Has(EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS));
        UNIT_ASSERT_EQUAL(fastConfig.Get(EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS), TVector<TString>({"banneditype"}));
        UNIT_ASSERT(fastConfig.Has(EFastConfigFlag::FORCE_LEGACY_TYPES_CONVERSION));

        fastConfig.Stop();
    }

    Y_UNIT_TEST(TestUpdateFailed) {
        TLog log {};
        TString localFileName {"fast_config.json"};
        TTempFile tempFile {localFileName};
        TFastConfigSettings settings {
            "oauth_token",
            "locke",
            "ytPath",
            localFileName,
            NTags::TInstanceKey::FromNamed("itype|ctype=prod")
        };
        NTest::TYtRefreshingFastConfig fastConfig {
            log, settings, TDuration::Hours(24),
            {NTest::MOCKED_CONFIG_STRING_DEPRECATED, NTest::MOCKED_CONFIG_STRING_ACTUAL_FAILED}
        };
        fastConfig.Start();
        checkMockedConfig(fastConfig);
        fastConfig.CompulsoryTick();
        checkMockedConfig(fastConfig);
        fastConfig.Stop();
    }

    Y_UNIT_TEST(TestUpdateTransition) {
        TLog log {};
        TString localFileName {"fast_config.json"};
        TTempFile tempFile {localFileName};
        TFastConfigSettings settings {
            "oauth_token",
            "locke",
            "ytPath",
            localFileName,
            NTags::TInstanceKey::FromNamed("itype|ctype=prod")
        };
        NTest::TYtRefreshingFastConfig fastConfig {
            log, settings, TDuration::Hours(24),
            {NTest::MOCKED_CONFIG_STRING_DEPRECATED, NTest::MOCKED_CONFIG_STRING_DEPRECATED_TRANSITION_TO_ACTUAL, NTest::MOCKED_CONFIG_STRING_ACTUAL_FAILED}
        };
        fastConfig.Start();
        checkMockedConfig(fastConfig);
        fastConfig.CompulsoryTick();
        checkMockedConfig(fastConfig);
        fastConfig.CompulsoryTick();
        checkMockedConfig(fastConfig);
        fastConfig.Stop();
    }
}
