#include "host.h"

#include <util/generic/vector.h>
#include <util/generic/xrange.h>

using namespace NZoom::NHost;
using NZoom::NSignal::TCache;

namespace {
    const THostRef* CreateInterned(TStringBuf key) {
        const THostRef* res = Singleton<TCache<THostRef>>()->Get(key);
        if (res) {
            return res;
        } else {
            return Singleton<TCache<THostRef>>()->Insert(THostRef(TString(key)));
        }
    }

    static const THostRef* EMPTY_HOST_REF = CreateInterned(TStringBuf(""));
}

THostRef::THostRef(TStringBuf name)
    : Name(name)
    , GroupFlag(!AnyOf(Name, IsAsciiLower<unsigned char>))
{
}

bool THostRef::IsGroup() const noexcept {
    return GroupFlag;
}

const TString& THostRef::GetName() const noexcept {
    return Name;
}

THostName::THostName()
    : Impl(EMPTY_HOST_REF)
{
}

THostName::THostName(const TString& key)
    : Impl(CreateInterned(key))
{
}

THostName::THostName(TStringBuf key)
    : Impl(CreateInterned(key))
{
}

THostName::THostName(const THostRef* impl)
    : Impl(impl)
{
}

TVector<THostName> THostName::CreateMany(std::span<const TString* const> names) {
    TVector<THostName> result(Reserve(names.size()));
    TVector<size_t> toCreate(Reserve(names.size()));

    {
        TCache<THostRef>::TReader reader(Singleton<TCache<THostRef>>());
        for (const auto position : xrange(names.size())) {
            const TString* name = names[position];
            if (nullptr == result.emplace_back(THostName(reader.Get(*name))).Impl) {
                toCreate.emplace_back(position);
            }
        }
        if (toCreate.empty()) {
            return result;
        }
    }

    {
        TCache<THostRef>::TWriter writer(Singleton<TCache<THostRef>>());
        for (const auto position : toCreate) {
            result[position].Impl = writer.Insert(THostRef(*names[position]));
        }
    }

    return result;
}

const TString& THostName::GetName() const noexcept {
    return Impl->GetName();
}

bool THostName::IsGroup() const noexcept {
    return Impl->IsGroup();
}

bool THostName::Empty() const noexcept {
    return Impl == EMPTY_HOST_REF;
}

template <>
void Out<THostName>(IOutputStream& stream,
                    TTypeTraits<THostName>::TFuncParam key) {
    stream.Write(key.GetName());
}
