#pragma once

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/generic/hash.h>
#include <util/generic/string.h>

namespace NZoom {
    namespace NSignal {
        template<typename T>
        class TCache {
        private:
            TLightRWLock Mutex;
            absl::flat_hash_map<TStringBuf, std::unique_ptr<T>, THash<TStringBuf>> Cache;

            const T* GetUnsafe(TStringBuf key) {
                const auto it = Cache.find(key);
                if (it != Cache.end()) {
                    return it->second.get();
                }
                return nullptr;
            }

            const T* InsertUnsafe(T&& item) {
                TStringBuf name = item.GetName();
                if (auto it = Cache.find(name); it != Cache.end()) {
                    return it->second.get();
                }

                // we have to move item to heap to make sure that raw pointer will
                // not be invalidated after rehashing of Cache table
                auto itemCopy = std::make_unique<T>(std::move(item));
                name = itemCopy->GetName();
                auto [it, _] = Cache.emplace(name, std::move(itemCopy));
                return it->second.get();
            }

        public:
            class TWriter : public TNonCopyable {
            public:
                TWriter(TCache<T>* cache)
                    : Cache(cache)
                    , Guard(Cache->Mutex)
                {
                }

                const T* Insert(T&& item) {
                    return Cache->InsertUnsafe(std::move(item));
                }

            private:
                TCache<T>* Cache;
                TLightWriteGuard Guard;
            };

            class TReader : public TNonCopyable {
            public:
                TReader(TCache<T>* cache)
                    : Cache(cache)
                    , Guard(Cache->Mutex)
                {
                }

                template<typename KeyType>
                const T* Get(const KeyType& key) {
                    return Cache->GetUnsafe(key);
                }

            private:
                TCache<T>* Cache;
                TLightReadGuard Guard;
            };

            template<typename KeyType>
            const T* Get(const KeyType& key) {
                TLightReadGuard rg(Mutex);
                return GetUnsafe(key);
            }

            const T* Insert(T&& item) {
                TLightWriteGuard rg(Mutex);
                return InsertUnsafe(std::move(item));
            }
        };
    }
}
