#include "signal_name.h"
#include "cache.h"

#include <util/generic/vector.h>
#include <util/generic/xrange.h>

#include <re2/re2.h>

using namespace NZoom::NSignal;

//Do not want to cache them since matching should have the same complexity as hash()
bool NZoom::NSignal::IsValidSignalName(const TString& signalNameWithPrefix) {
    // https://wiki.yandex-team.ru/golovan/userdocs/tagsandsignalnaming/
    static const re2::RE2 VALID_SIGNAL_NAME_REGEX(R"([a-zA-Z0-9_\-./@]{1,128})");
    static const re2::RE2 PROHIBIT_AVVX_ETC_SUFFIXES_REGEX(R"(.+_[xnta]vv[xnt])");

    re2::StringPiece value(signalNameWithPrefix);
    return RE2::FullMatch(value, VALID_SIGNAL_NAME_REGEX) &&
        !RE2::FullMatch(value, PROHIBIT_AVVX_ETC_SUFFIXES_REGEX);
}

namespace {

    using namespace NZoom::NAccumulators;
    //TODO: use memory pool for names if possible
    class TNewSignalRef : public TAbstractSignalRef, TNonCopyable {
    private:
        TAggregationRules Aggregation;

    public:
        TNewSignalRef(const TString& name, const TAggregationRules aggregation)
            : TAbstractSignalRef(name)
            , Aggregation(aggregation)
        {
        }

        TNewSignalRef(TStringBuf name, const TAggregationRules aggregation)
            : TAbstractSignalRef(TString{name})
            , Aggregation(aggregation)
        {
        }

        TNewSignalRef(TNewSignalRef&& other)
            : TAbstractSignalRef(std::move(other.Name))
            , Aggregation(std::move(other.Aggregation))
        {
        }

        bool IsOld() const noexcept final {
            return false;
        }

        const TAggregationRules* GetAggregationRules() const noexcept final {
            return &Aggregation;
        }
    };


    class TOldSignalRef : public TAbstractSignalRef, TNonCopyable {
    public:
        TOldSignalRef(const TString& name)
            : TAbstractSignalRef(name)
        {
        }

        TOldSignalRef(TOldSignalRef&& other)
            : TAbstractSignalRef(std::move(other.Name))
        {
        }

        bool IsOld() const noexcept final {
            return true;
        }

        const TAggregationRules* GetAggregationRules() const noexcept final {
            return nullptr;
        }
    };


    template<typename K>
    bool TryCreateInternedNew(const K& key, const TAbstractSignalRef** dst) {
        const auto aggregationMaybe = NZoom::NAccumulators::TAggregationRules::FromFullName(key);
        if (aggregationMaybe.Defined()) {
            const TAbstractSignalRef* res = Singleton<TCache<TNewSignalRef>>()->Get(key);
            if (res) {
                *dst = res;
                return true;
            }
            *dst = Singleton<TCache<TNewSignalRef>>()->Insert(TNewSignalRef(TString(key), aggregationMaybe.GetRef()));
            return true;
        }
        return false;
    }

    template<typename K>
    const TAbstractSignalRef* CreateInterned(const K& key) {
        const TAbstractSignalRef* res = nullptr;
        if (!TryCreateInternedNew(key, &res)) {
            const TAbstractSignalRef* res = Singleton<TCache<TOldSignalRef>>()->Get(key);
            if (res) {
                return res;
            }
            return Singleton<TCache<TOldSignalRef>>()->Insert(TOldSignalRef(TString(key)));
        }
        return res;
    }
}


TAbstractSignalRef::TAbstractSignalRef(TString name)
    : Name(std::move(name))
{
}

const TString& TAbstractSignalRef::GetName() const noexcept {
    return Name;
}


TSignalName::TSignalName(const TString& key)
    : Impl(CreateInterned(key))
{
}

TSignalName::TSignalName(TStringBuf key)
    : Impl(CreateInterned(key))
{
}

TSignalName::TSignalName(const TAbstractSignalRef* impl)
    : Impl(impl)
{
}

TMaybe<TSignalName> TSignalName::TryNew(const TString& name) {
    const TAbstractSignalRef* res = nullptr;
    if (TryCreateInternedNew(name, &res)) {
        return TSignalName(res);
    }
    return Nothing();
}

TVector<TSignalName> TSignalName::CreateMany(std::span<const TString* const> names) {
    TVector<TSignalName> result(Reserve(names.size()));
    TVector<size_t> toCreate(Reserve(names.size()));
    TVector<size_t> toIntern(Reserve(names.size()));

    {
        TCache<TNewSignalRef>::TReader reader(Singleton<TCache<TNewSignalRef>>());
        for (const auto position : xrange(names.size())) {
            const TString* name = names[position];
            if (nullptr == result.emplace_back(TSignalName(reader.Get(*name))).Impl) {
                toCreate.emplace_back(position);
            }
        }
        if (toCreate.empty()) {
            return result;
        }
    }

    {
        TCache<TNewSignalRef>::TWriter writer(Singleton<TCache<TNewSignalRef>>());
        for (const auto position : toCreate) {
            const TString* name = names[position];
            const auto aggregationMaybe = NZoom::NAccumulators::TAggregationRules::FromFullName(*name);
            if (aggregationMaybe.Defined()) {
                result[position].Impl = writer.Insert(TNewSignalRef(*name, aggregationMaybe.GetRef()));
            } else {
                toIntern.emplace_back(position);
            }
        }
        if (toIntern.empty()) {
            return result;
        }
    }

    // this is very rare case, don't optimize it
    for (const auto position : toIntern) {
        result[position].Impl = CreateInterned(*names[position]);
    }

    return result;
}

const TString& TSignalName::GetName() const noexcept {
    return Impl->GetName();
}

bool TSignalName::IsOld() const noexcept {
    return Impl->IsOld();
}

TMaybe<NZoom::NAccumulators::EAccumulatorType> TSignalName::GetAggregationType(NZoom::NAccumulators::EAggregationMethod method) const {
    if (GetAggregationRules()) {
        return GetAggregationRules()->GetAccumulatorType(method);
    }
    return Nothing();
}

const TAggregationRules* TSignalName::GetAggregationRules() const noexcept {
    return Impl->GetAggregationRules();
}

template <>
void Out<TSignalName>(IOutputStream& stream,
                      TTypeTraits<TSignalName>::TFuncParam signal) {
    stream.Write(signal.GetName());
}
