#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/tags/request_key.h>
#include <infra/yasm/common/labels/tags/dynamic_filter.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NTags;

class TDynamicFilterTestCaseBuilder {
public:
    TDynamicFilterTestCaseBuilder& SetFilter(THolder<TDynamicFilter>&& filterPtr) {
        FilterPtr = std::move(filterPtr);
        return *this;
    }

    TVector<TInstanceKey> LoadInstances(const TVector<TString>& instancesNamed) {
        TVector<TInstanceKey> instances;
        for (const auto& instance: instancesNamed) {
            instances.emplace_back(TInstanceKey::FromNamed(instance));
        }
        return instances;
    }

    TDynamicFilterTestCaseBuilder& SetInstances(TVector<TString>&& instancesNamed) {
        Instances = LoadInstances(instancesNamed);
        return *this;
    }

    TDynamicFilterTestCaseBuilder& SetExpectedValidInstances(TVector<TString>&& expectedValidInstancesNamed) {
        ExpectedValidInstances = LoadInstances(expectedValidInstancesNamed);
        return *this;
    }

    TDynamicFilterTestCaseBuilder& SetExpectedInvalidInstances(TVector<TString>&& expectedInvalidInstancesNamed) {
        ExpectedInvalidInstances = LoadInstances(expectedInvalidInstancesNamed);
        return *this;
    }

    TDynamicFilterTestCaseBuilder& SetExpectedKeys(THashSet<TString>&& expectedKeysNamed) {
        THashSet<TInstanceKey> expectedKeys;
        for (const auto& key: expectedKeysNamed) {
            expectedKeys.insert(TInstanceKey::FromNamed(key));
        }

        ExpectedKeys = std::move(expectedKeys);
        return *this;
    }

    TDynamicFilterTestCaseBuilder& AssertIsValid() {
        TVector<TInstanceKey> validInstances;
        TVector<TInstanceKey> invalidInstances;

        TVector<NTags::TInstanceKey> allInstanceKeys;

        for (const auto& instance : Instances) {
            if (FilterPtr->Feed(instance)) {
                validInstances.push_back(instance);
            } else {
                invalidInstances.push_back(instance);
            }
            allInstanceKeys.emplace_back(instance);
        }

        auto keys = FilterPtr->Resolve();

        THashSet<TInstanceKey> foundKeys(keys.begin(), keys.end());

        UNIT_ASSERT_VALUES_EQUAL(validInstances, ExpectedValidInstances);
        UNIT_ASSERT_VALUES_EQUAL(invalidInstances, ExpectedInvalidInstances);
        UNIT_ASSERT_VALUES_EQUAL(foundKeys, ExpectedKeys);

        return *this;
    }

private:
    THolder<TDynamicFilter> FilterPtr;
    TVector<TInstanceKey> Instances;
    TVector<TInstanceKey> ExpectedValidInstances;
    TVector<TInstanceKey> ExpectedInvalidInstances;
    THashSet<TInstanceKey> ExpectedKeys;
};

class TDynamicFilterTest: public TTestBase {
    UNIT_TEST_SUITE(TDynamicFilterTest);
    UNIT_TEST(DynamicFilterTests)
    UNIT_TEST_SUITE_END();

private:
    void DynamicFilterTests() {
        TDynamicFilterTestCaseBuilder dynamicFilterTestCaseBuilder;

        dynamicFilterTestCaseBuilder
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=mdsproxy;ctype=production;geo=vla")))
            .SetInstances(
                {
                    "mdsproxy|ctype=production;geo=vla;prj=none;tier=default",
                    "mdsproxy|ctype=production;geo=vla;prj=none|tier",
                    "mdsproxy|ctype=production;geo=vla;prj=npm|tier",
                })
            .SetExpectedValidInstances(
                {
                    "mdsproxy|ctype=production;geo=vla;prj=none;tier=default",
                    "mdsproxy|ctype=production;geo=vla;prj=none|tier",
                    "mdsproxy|ctype=production;geo=vla;prj=npm|tier",
                })
            .SetExpectedInvalidInstances({})
            .SetExpectedKeys(
                {
                     "mdsproxy|ctype=production;geo=vla;prj=none|tier",
                     "mdsproxy|ctype=production;geo=vla;prj=npm|tier",
                })
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetInstances(
                {
                    "base|ctype=prod;geo=none;prj=none|tier",
                    "base|ctype=prod;geo=none;prj=none;tier=tier1",
                    "base|ctype=prod;geo=none;prj=none;tier=tier2",
                    "upper|ctype=prod;geo=none;prj=none;tier=tier2",
                })
            .SetExpectedValidInstances(
                {
                    "base|ctype=prod;geo=none;prj=none|tier",
                    "base|ctype=prod;geo=none;prj=none;tier=tier1",
                    "base|ctype=prod;geo=none;prj=none;tier=tier2",
                })
            .SetExpectedInvalidInstances(
                {
                    "upper|ctype=prod;geo=none;prj=none;tier=tier2",
                })
            .SetExpectedKeys(
                {
                     "base|ctype=prod;geo=none;prj=none|tier",
                })
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=base")))
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetInstances(
                {
                    "base|ctype=prod;geo=none;prj=none|tier",
                    "base|ctype=prod;geo=none;prj=none;tier=tier1",
                    "base|ctype=prod;geo=none;prj=none;tier=tier2",
                    "upper|ctype=prod;geo=none;prj=none;tier=tier2",
                })
            .SetExpectedValidInstances(
                {
                    "base|ctype=prod;geo=none;prj=none|tier",
                    "base|ctype=prod;geo=none;prj=none;tier=tier1",
                    "base|ctype=prod;geo=none;prj=none;tier=tier2",
                })
            .SetExpectedInvalidInstances(
                {
                    "upper|ctype=prod;geo=none;prj=none;tier=tier2",
                })
            .SetExpectedKeys(
                {
                     "base|ctype=prod;geo=none;prj=none|tier",
                })
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=base;ctype=p*")))
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetInstances(
                {
                    "base|ctype=prestable;geo=none;prj=none;tier=tier1;nanny_service=prod_hamster",
                    "base|ctype=prestable;geo=none;prj=none;tier=tier2;nanny_service=prod_hamster;gencfg_group=ALL_HAMSTER",
                    "base|ctype=prestable;geo=none;prj=none;nanny_service=prod_hamster|tier",
                    "base|ctype=prestable;geo=none;prj=none;gencfg_group=ALL_HAMSTER;nanny_service=prod_hamster|tier",
                })
            .SetExpectedValidInstances(
                {
                    "base|ctype=prestable;geo=none;prj=none;tier=tier1;nanny_service=prod_hamster",
                    "base|ctype=prestable;geo=none;prj=none;tier=tier2;nanny_service=prod_hamster;gencfg_group=ALL_HAMSTER",
                    "base|ctype=prestable;geo=none;prj=none;nanny_service=prod_hamster|tier",
                    "base|ctype=prestable;geo=none;prj=none;gencfg_group=ALL_HAMSTER;nanny_service=prod_hamster|tier",
                })
            .SetExpectedInvalidInstances({})
            .SetExpectedKeys(
                {
                    "base|ctype=prestable;geo=none;prj=none;gencfg_group=ALL_HAMSTER;nanny_service=prod_hamster|tier",
                     "base|ctype=prestable;geo=none;prj=none;nanny_service=prod_hamster|tier"
                })
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=base;nanny_service=prod_hamster")))
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetInstances(
                {
                    "yasmtestit|ctype=none;geo=none;prj=none;service=unstable-yasm-testit|tier",
                    "yasmtestit||ctype,geo,prj,tier"
                })
            .SetExpectedValidInstances(
                {
                    "yasmtestit|ctype=none;geo=none;prj=none;service=unstable-yasm-testit|tier",
                    "yasmtestit||ctype,geo,prj,tier"
                })
            .SetExpectedInvalidInstances({})
            .SetExpectedKeys(
                {
                    "yasmtestit|ctype=none;geo=none;prj=none;service=unstable-yasm-testit|tier",
                    "yasmtestit||ctype,geo,prj,tier"
                })
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=yasmtestit")))
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetInstances(
                {
                    "yasmtestit|ctype=prod;geo=none;prj=none;service=unstable-yasm-testit|tier",
                    "yasmtestit|ctype=prod;geo=none;prj=none;tier=tier1",
                    "yasmtestit|ctype=unstable;geo=none;prj=none|tier",
                })
            .SetExpectedValidInstances(
                {
                    "yasmtestit|ctype=prod;geo=none;prj=none;service=unstable-yasm-testit|tier",
                    "yasmtestit|ctype=prod;geo=none;prj=none;tier=tier1"
                })
            .SetExpectedInvalidInstances(
                {
                    "yasmtestit|ctype=unstable;geo=none;prj=none|tier"
                })
            .SetExpectedKeys(
                {
                    "yasmtestit|ctype=prod;geo=none;prj=none;service=unstable-yasm-testit|tier",
                    "yasmtestit|ctype=prod;geo=none;prj=none;tier=tier1"
                })
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=yasmtestit;ctype=p*")))
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=yasmtestit;ctype=p*")))
            .SetInstances(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=a;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=a;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=p|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=p;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t"
                })
            .SetExpectedValidInstances(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=a;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=a;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=p|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=p;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t"
                })
            .SetExpectedInvalidInstances({})
            .SetExpectedKeys(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c|prj,tier"
                })
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=yasmtestit;ctype=p*")))
            .SetInstances(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=d|prj,tier,from",

                    "yasmtestit|ctype=prod;prj=p|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier",
                    "yasmtestit|ctype=prod;prj=q|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=q|geo,tier",

                    "yasmtestit|ctype=prod;geo=a;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=a;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=p|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=p;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t",
                    "yasmtestit|ctype=prod;geo=d;prj=q;tier=t;from=f"
                })
            .SetExpectedValidInstances(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=d|prj,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier",
                    "yasmtestit|ctype=prod;prj=q|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=q|geo,tier",
                    "yasmtestit|ctype=prod;geo=a;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=a;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=p|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=p;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t",
                    "yasmtestit|ctype=prod;geo=d;prj=q;tier=t;from=f"
                })
            .SetExpectedInvalidInstances({})
            .SetExpectedKeys(
                {
                    // select 2 pre-aggregates from ctype,prj,geo,tier,from tag name set
                    "yasmtestit|ctype=prod;prj=p|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=q|geo,tier,from",

                    // select a single pre-aggregate from ctype,geo,prj,tier tag name set
                    "yasmtestit|ctype=prod;geo=c|prj,tier"
                })
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=yasmtestit;ctype=p*;geo=c*")))
            .SetInstances(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=cc|prj,tier,from",

                    "yasmtestit|ctype=prod;prj=p|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier",
                    "yasmtestit|ctype=prod;prj=q|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=q|geo,tier",

                    "yasmtestit|ctype=prod;geo=a;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=a;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=p|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=p;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t",
                    "yasmtestit|ctype=prod;geo=cc;prj=q;tier=t;from=f"
                })
            .SetExpectedValidInstances(
                {
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=cc|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c;prj=p|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=p;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t",
                    "yasmtestit|ctype=prod;geo=cc;prj=q;tier=t;from=f"
                })
            .SetExpectedInvalidInstances(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier",
                    "yasmtestit|ctype=prod;prj=q|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=q|geo,tier",
                    "yasmtestit|ctype=prod;geo=a;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=a;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=q;tier=t;from=f"
                })
            .SetExpectedKeys(
                {
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=cc|prj,tier,from",
                })
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetFilter(MakeHolder<TDynamicFilter>(TRequestKey::FromString("itype=yasmtestit;ctype=p*;geo=c*;prj=q")))
            .SetInstances(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=cc|prj,tier,from",

                    "yasmtestit|ctype=prod;prj=p|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier",
                    "yasmtestit|ctype=prod;prj=q|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=q|geo,tier",

                    "yasmtestit|ctype=prod;geo=a;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=a;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=p|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=p;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t",
                    "yasmtestit|ctype=prod;geo=cc;prj=q;tier=t;from=f"
                })
            .SetExpectedValidInstances(
                {
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t",
                    "yasmtestit|ctype=prod;geo=cc;prj=q;tier=t;from=f"
                })
            .SetExpectedInvalidInstances(
                {
                    "yasmtestit|ctype=prod;geo=a|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=b|prj,tier,from",
                    "yasmtestit|ctype=prod;geo=c|prj,tier",
                    "yasmtestit|ctype=prod;geo=cc|prj,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=p|geo,tier",
                    "yasmtestit|ctype=prod;prj=q|geo,tier,from",
                    "yasmtestit|ctype=prod;prj=q|geo,tier",
                    "yasmtestit|ctype=prod;geo=a;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=a;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=a;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=p|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=p;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=b;prj=q|tier,from",
                    "yasmtestit|ctype=prod;geo=b;prj=q;tier=t;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=p|tier",
                    "yasmtestit|ctype=prod;geo=c;prj=p;tier=t"
                })
            .SetExpectedKeys(
                {
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=cc;prj=q;tier=t;from=f"
                })
            .AssertIsValid();

        dynamicFilterTestCaseBuilder
            .SetFilter(MakeHolder<TDynamicFilter>())
            .SetInstances(
                {
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=tt",
                    "yasmtestit|ctype=prod;geo=cc;prj=q;tier=zz;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;tier=t|prj",
                    "yasmtestit|ctype=prod;geo=c;tier=tt|prj"
                })
            .SetExpectedValidInstances(
                {
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=t",
                    "yasmtestit|ctype=prod;geo=c;prj=q;tier=tt",
                    "yasmtestit|ctype=prod;geo=cc;prj=q;tier=zz;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                    "yasmtestit|ctype=prod;geo=c;tier=t|prj",
                    "yasmtestit|ctype=prod;geo=c;tier=tt|prj"
                })
            .SetExpectedInvalidInstances({})
            .SetExpectedKeys(
                {
                    "yasmtestit|ctype=prod;geo=cc;prj=q;tier=zz;from=f",
                    "yasmtestit|ctype=prod;geo=c;prj=q|tier",
                })
            .AssertIsValid();
    }
};

UNIT_TEST_SUITE_REGISTRATION(TDynamicFilterTest);
