#include <infra/yasm/common/labels/tags/request_key.h>
#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/tags/dynamic_filter.h>
#include <infra/monitoring/common/msgpack.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/xrange.h>
#include <util/generic/maybe.h>
#include <util/generic/set.h>
#include <util/stream/file.h>
#include <util/stream/zlib.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

using namespace NTags;

const std::size_t CHUNK_SIZE = 8192;

using NMonitoring::TMapIterator;
using NMonitoring::TArrayIterator;

struct TRecord {
    TVector<TInstanceKey> Keys;
    TVector<TRequestKey> Subs;
    TVector<std::pair<TRequestKey, TVector<TInstanceKey>>> Results;
};

class TReader {
public:
    TReader(const TString& name)
        : File(name, EOpenModeFlag::RdOnly | EOpenModeFlag::Seq)
        , FileStream(File)
        , DecodedStream(&FileStream)
    {
    }

    TMaybe<TRecord> Next() {
        while (true) {
            Unpacker.reserve_buffer(CHUNK_SIZE);
            const size_t readed = DecodedStream.Read(Unpacker.buffer(), CHUNK_SIZE);
            if (!readed) {
                return Nothing();
            }

            Unpacker.buffer_consumed(readed);
            msgpack::unpacked result;
            while (Unpacker.next(result)) {
                msgpack::object obj(result.get());
                return CreateRecord(obj.via.map);
            }
        }
    }

private:
    TRecord CreateRecord(msgpack::object_map root) {
        TRecord record;
        for (const auto& pair : TMapIterator(root)) {
            const auto key(pair.key.as<TStringBuf>());
            if (key == TStringBuf("keys")) {
                record.Keys = ToInstanceKeys(pair.val.via.array);
            } else if (key == TStringBuf("subs")) {
                for (const auto& requestKey : TArrayIterator(pair.val.via.array)) {
                    record.Subs.emplace_back(TRequestKey::FromString(requestKey.as<TStringBuf>()));
                }
            } else if (key == TStringBuf("result")) {
                for (const auto& resultPair : TArrayIterator(pair.val.via.array)) {
                    const auto resultArray = resultPair.via.array;
                    record.Results.emplace_back(
                        TRequestKey::FromString(resultArray.ptr[0].as<TStringBuf>()),
                        ToInstanceKeys(resultArray.ptr[1].via.array)
                    );
                }
            } else {
                ythrow yexception() << "unknown key " << key << " given";
            }
        }
        return record;
    }

    TVector<TInstanceKey> ToInstanceKeys(msgpack::object_array root) {
        TVector<TInstanceKey> result;
        for (const auto& instanceKey : TArrayIterator(root)) {
            result.emplace_back(TInstanceKey::FromNamed(instanceKey.as<TStringBuf>()));
            UNIT_ASSERT(!result.back().Empty());
        }
        return result;
    }

    TFile File;
    TFileInput FileStream;
    TBufferedZLibDecompress DecodedStream;
    msgpack::unpacker Unpacker;
};

class TDynamicMatcherTest: public TTestBase {
    UNIT_TEST_SUITE(TDynamicMatcherTest);
    UNIT_TEST(TestOnPreparedData)
    UNIT_TEST_SUITE_END();

private:
    inline void TestOnPreparedData() {
        TReader reader("dyn.log.gz");

        bool hasRecords = false;
        while (true) {
            TMaybe<TRecord> record = reader.Next();
            if (!record) {
                break;
            }

            AssertRecordIsValid(record.GetRef());
            hasRecords = true;
        }

        UNIT_ASSERT(hasRecords);
    }

    inline void AssertRecordIsValid(const TRecord& record) {
        TVector<NTags::TDynamicFilter> filters;
        for (const auto& result : record.Results) {
            filters.emplace_back(result.first);
        }

        bool failed = false;
        for (const auto idx : xrange(filters.size())) {
            filters[idx].FeedMany(record.Keys);
            const auto actual = filters[idx].Resolve();
            const auto& expected = record.Results[idx].second;
            if (expected.empty() && actual.empty()) {
                continue;
            }

            TSet<TString> actualSet;
            for (const auto& instanceKey : actual) {
                actualSet.emplace(instanceKey.ToNamed());
            }

            TSet<TString> expectedSet;
            for (const auto& instanceKey : expected) {
                expectedSet.emplace(instanceKey.ToNamed());
            }

            if (actualSet != expectedSet) {
                Cerr << "request key\t" << record.Results[idx].first.ToNamed() << Endl;

                for (const auto& instanceKey : actual) {
                    Cerr << "actual\t" << instanceKey << Endl;
                }

                for (const auto& instanceKey : expected) {
                    Cerr << "expected\t" << instanceKey << Endl;
                }

                failed = true;
            }
        }

        if (failed) {
            for (const auto& instanceKey : record.Keys) {
                Cerr << "instance key\t" << instanceKey << Endl;
            }
        }

        UNIT_ASSERT(!failed);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TDynamicMatcherTest);
