#include "instance_key.h"
#include "verification.h"

#include <util/generic/hash.h>
#include <util/generic/xrange.h>
#include <util/system/env.h>

using NZoom::NHost::THostName;

namespace NTags {
    namespace NPrivate {
        static const TString EMPTY_STRING;
        static const TInstanceKeyTags EMPTY_TAGS;
        static const TTagValuePairs EMPTY_TAG_VALUE_PAIRS;
        static const TAggregated EMPTY_AGGREGATED;
        static const TInternedTagNameSet EMPTY_TAG_NAME_SET;

        class TAccumulatingVisitor;

        bool ParseNamedStringUnsafe(TStringBuf, TAccumulatingVisitor&);

        class TInstanceKeyImpl {
        public:
            TInstanceKeyImpl(TString named, TStringBuf itype, bool hasMultipleValues)
                : Named(named)
                , Itype(itype)
                , HasMultipleValues_(hasMultipleValues)
                , GrouplessKeyCached(nullptr)
            {
            }

            virtual ~TInstanceKeyImpl() = default;

            TStringBuf GetItype() const noexcept {
                return Itype;
            }

            const TString& GetNamed() const noexcept {
                return Named;
            }

            bool HasMultipleValues() const noexcept {
                return HasMultipleValues_;
            }

            virtual TArrayRef<const TInstanceKeyTag> GetTags() const noexcept = 0;
            virtual TAggregated GetAggregated() const noexcept = 0;

            virtual TInternedTagNameSet GetTagNameSet() const noexcept = 0;
            virtual TInternedTagNameSet GetTagsAndAggregatedNameSet() const noexcept = 0;

            virtual THostName GetHostName() const noexcept = 0;
            virtual THostName GetGroupName() const noexcept = 0;

            const TInstanceKeyImpl* FindAggregatedBy(TInternedTagNameSet fingerprint) const noexcept {
                TLightReadGuard guard(CacheMutex);
                const auto it(AggregatedByCache.find(fingerprint));
                if (it != AggregatedByCache.end()) {
                    return it->second;
                } else {
                    return nullptr;
                }
            }

            void SetAggregatedBy(TInternedTagNameSet fingerprint, const TInstanceKeyImpl* impl) const noexcept {
                TLightWriteGuard guard(CacheMutex);
                AggregatedByCache.emplace(fingerprint, impl);
            }

            const TInstanceKeyImpl* FindByGroupAndHost(THostName group, THostName host) const noexcept {
                TLightReadGuard guard(CacheMutex);
                const auto it(GroupAndHostCache.find(std::make_pair(group, host)));
                if (it != GroupAndHostCache.end()) {
                    return it->second;
                } else {
                    return nullptr;
                }
            }

            void SetForGroupAndHost(THostName group, THostName host, const TInstanceKeyImpl* impl) const noexcept {
                TLightWriteGuard guard(CacheMutex);
                GroupAndHostCache.emplace(std::make_pair(group, host), impl);
            }

            const TInstanceKeyImpl* GetGroupless() const noexcept {
                TLightReadGuard guard(CacheMutex);
                return GrouplessKeyCached;
            }

            void SetGroupless(const TInstanceKeyImpl* impl) const noexcept {
                TLightWriteGuard guard(CacheMutex);
                GrouplessKeyCached = impl;
            }

        private:
            const TString Named;
            const TStringBuf Itype;
            const bool HasMultipleValues_;

            mutable THashMap<TInternedTagNameSet, const TInstanceKeyImpl*> AggregatedByCache;
            mutable THashMap<std::pair<THostName, THostName>, const TInstanceKeyImpl*> GroupAndHostCache;
            mutable const TInstanceKeyImpl* GrouplessKeyCached;
            TLightRWLock CacheMutex;
        };

        template <size_t N>
        class TStaticInstanceKeyImpl final: public TInstanceKeyImpl {
        public:
            TStaticInstanceKeyImpl(TString named, TStringBuf itype, bool hasMultipleValues,
                                   const TSmallVec<std::pair<TStringBuf, TStringBuf>>& tags,
                                   const TSmallVec<TStringBuf>& aggregated)
                : TInstanceKeyImpl(named, itype, hasMultipleValues)
                , TagsLength(tags.size())
                , AggregatedLength(aggregated.size())
            {
                Y_VERIFY(tags.size() <= N);
                Y_VERIFY(aggregated.size() <= N);

                TVector<TTagName> tagNames(Reserve(TagsLength + AggregatedLength));
                for (const auto idx : xrange(TagsLength)) {
                    const auto& source(tags[idx]);
                    const TTagName internedName(source.first);
                    Tags[idx] = TInstanceKeyTag{
                        .Name = source.first,
                        .InternedName = internedName,
                        .Value = source.second,
                        .ValueHash = ComputeHash(source.second)
                    };
                    tagNames.emplace_back(internedName);

                    static const TTagName internedHostTag(HOST_TAG);
                    static const TTagName internedGroupTag(GROUP_TAG);
                    if (internedName == internedHostTag) {
                        HostName = THostName(source.second);
                    } else if (internedName == internedGroupTag) {
                        if (HostName.Empty()) {
                            HostName = THostName(source.second);
                        }
                        GroupName = THostName(source.second);
                    }
                }
                TagNameSet = TInternedTagNameSet(tagNames);

                for (const auto& aggregatedTag: aggregated) {
                    tagNames.emplace_back(TTagName(aggregatedTag));
                }
                TagsAndAggregatedNameSet = TInternedTagNameSet(tagNames);

                std::copy_n(aggregated.begin(), aggregated.size(), Aggregated.begin());
            }

            TArrayRef<const TInstanceKeyTag> GetTags() const noexcept override {
                return MakeArrayRef(Tags.data(), TagsLength);
            }

            TAggregated GetAggregated() const noexcept override {
                return MakeArrayRef(Aggregated.data(), AggregatedLength);
            }

            TInternedTagNameSet GetTagNameSet() const noexcept override {
                return TagNameSet;
            }

            TInternedTagNameSet GetTagsAndAggregatedNameSet() const noexcept override {
                return TagsAndAggregatedNameSet;
            }

            THostName GetHostName() const noexcept override {
                return HostName;
            }

            THostName GetGroupName() const noexcept override {
                return GroupName;
            }

        private:
            std::array<TInstanceKeyTag, N> Tags;
            std::array<TStringBuf, N> Aggregated;

            ui8 TagsLength;
            ui8 AggregatedLength;

            TInternedTagNameSet TagNameSet;
            TInternedTagNameSet TagsAndAggregatedNameSet;

            THostName HostName;
            THostName GroupName;
        };

        class TAccumulatingVisitor {
        public:
            TAccumulatingVisitor() {
            }

            TAccumulatingVisitor(TAccumulatingVisitor&& other)
                : Length(other.Length)
                , Itype(other.Itype)
                , Tags(std::move(other.Tags))
                , Aggregated(std::move(other.Aggregated))
            {
            }

            TAccumulatingVisitor& SetItype(TStringBuf itype) noexcept {
                Length += itype.size();
                Length += 1; // '|' after itype
                Itype = itype;
                return *this;
            }

            TAccumulatingVisitor& AddTag(TStringBuf key, TStringBuf value) noexcept {
                Length += key.size();
                Length += value.size();
                Length += 1 + (Tags.empty() ? 0 : 1); // '=' and ';' if needed
                Tags.emplace_back(key, value);
                return *this;
            }

            TAccumulatingVisitor& AddAggregated(TStringBuf tag) noexcept {
                Length += tag.size() + 1;
                Aggregated.emplace_back(tag);
                return *this;
            }

            bool Finish() noexcept {
                // TODO: remove sorting when all instance keys will be generated by cpp code
                Sort(Tags);
                Sort(Aggregated);
                return !!Length;
            }

            const TString& ToNamed() const noexcept {
                if (!Builder.empty()) {
                    return Builder;
                }

                Builder.reserve(Length);

                Builder << Itype;

                TStringBuf lastTag;
                bool putTagsSeparator = false;
                for (const auto& pair : Tags) {
                    if (Y_UNLIKELY(lastTag == pair.first)) {
                        Builder << ',' << pair.second;

                    } else {
                        Builder << (putTagsSeparator ? ';' : '|') << pair.first << '=' << pair.second;

                        lastTag = pair.first;
                        putTagsSeparator = true;
                    }
                }

                if (!putTagsSeparator) {
                    Builder << '|';
                }

                bool putAggregatedSeparator = false;
                for (const auto& tag : Aggregated) {
                    Builder << (putAggregatedSeparator ? ',' : '|') << tag;
                    putAggregatedSeparator = true;
                }

                // case with multiple tag values not supported
                Y_ASSERT(Builder.size() <= Length);

                return Builder;
            }

            THolder<TInstanceKeyImpl> ToImpl() const {
                // because string key can be changed parse it again
                // arcadia strings use COW so it's possible to use references to it's content
                const TString named(ToNamed());

                TAccumulatingVisitor visitor;
                ParseNamedStringUnsafe(named, visitor);

                // check that all tags have only one value
                bool hasMultipleValues = false;
                TStringBuf prevTag;
                for (const auto& pair : visitor.Tags) {
                    if (Y_UNLIKELY(pair.first == prevTag)) {
                        hasMultipleValues = true;
                        break;
                    }
                    prevTag = pair.first;
                }

                size_t size(Max(visitor.Tags.size(), visitor.Aggregated.size()));
                if (size <= 4) {
                    return MakeHolder<TStaticInstanceKeyImpl<4>>(
                        named, visitor.Itype, hasMultipleValues,
                        visitor.Tags, visitor.Aggregated);
                } else if (size <= 8) {
                    return MakeHolder<TStaticInstanceKeyImpl<8>>(
                        named, visitor.Itype, hasMultipleValues,
                        visitor.Tags, visitor.Aggregated);
                } else if (size <= 16) {
                    return MakeHolder<TStaticInstanceKeyImpl<16>>(
                        named, visitor.Itype, hasMultipleValues,
                        visitor.Tags, visitor.Aggregated);
                } else {
                    ythrow yexception() << "too many tags given";
                }
            }

        private:
            size_t Length = 0;
            TStringBuf Itype;
            // try to allocate temporary vectors on stack
            TSmallVec<std::pair<TStringBuf, TStringBuf>> Tags;
            TSmallVec<TStringBuf> Aggregated;
            // cached string in named format
            mutable TStringBuilder Builder;
        };

        enum class ENamedParserState {
            ITYPE,
            TAGS,
            AGGREGATED,
            END
        };

        bool ParseNamedStringUnsafe(TStringBuf incoming, TAccumulatingVisitor& visitor) {
            // not intended to use with invalid data!
            ENamedParserState parserState(ENamedParserState::ITYPE);
            for (const auto& it : StringSplitter(incoming).Split('|')) {
                switch (parserState) {
                    case ENamedParserState::ITYPE: {
                        if (it.Token().empty()) {
                            return false;
                        }
                        visitor.SetItype(it.Token());
                        parserState = ENamedParserState::TAGS;
                        break;
                    };
                    case ENamedParserState::TAGS: {
                        TKeyValueDelimStringIter tags(it.Token(), ";");
                        while (tags.Valid()) {
                            // allow empty tag with empty value, but don't ignore non-empty strings
                            if (tags.Value().empty() && !it.Token().empty()) {
                                return false;
                            }
                            for (const auto& value : StringSplitter(tags.Value()).Split(',')) {
                                if (tags.Key().empty() || value.Token().empty()) {
                                    return false;
                                }
                                visitor.AddTag(tags.Key(), value.Token());
                            }
                            ++tags;
                        }
                        parserState = ENamedParserState::AGGREGATED;
                        break;
                    };
                    case ENamedParserState::AGGREGATED: {
                        for (const auto& aggregated : StringSplitter(it.Token()).Split(',')) {
                            if (aggregated.Token().empty()) {
                                return false;
                            }
                            visitor.AddAggregated(aggregated.Token());
                        }
                        parserState = ENamedParserState::END;
                        break;
                    }
                    case ENamedParserState::END: {
                        return false;
                    }
                };
            }
            return visitor.Finish();
        }

        inline bool ParseUnderscoreStringUnsafe(TStringBuf incoming, TAccumulatingVisitor& visitor) {
            TSmallVec<TStringBuf> tags;
            tags.reserve(5);
            StringSplitter(incoming).Split('_').Collect(&tags);
            switch (tags.size()) {
                case 2: {
                    visitor.SetItype(tags[0]);
                    if (tags[1] != SELF) {
                        return false;
                    }
                    visitor.AddAggregated(CTYPE_TAG);
                    visitor.AddAggregated(GEO_TAG);
                    visitor.AddAggregated(PRJ_TAG);
                    visitor.AddAggregated(TIER_TAG);
                    break;
                }
                case 5: {
                    visitor.SetItype(tags[0]);
                    visitor.AddTag(CTYPE_TAG, tags[1]);
                    visitor.AddTag(PRJ_TAG, tags[2]);
                    visitor.AddTag(GEO_TAG, tags[3]);
                    if (tags[4] == SELF) {
                        visitor.AddAggregated(TIER_TAG);
                    } else {
                        visitor.AddTag(TIER_TAG, tags[4]);
                    }
                    break;
                }
                default: {
                    return false;
                }
            };
            return visitor.Finish();
        }

        inline bool ParseHistDbUnsafe(const TVector<std::pair<TStringBuf, TStringBuf>> input, TAccumulatingVisitor& visitor) {
            switch (input.size()) {
                case 2: {
                    for (const auto& pair : input) {
                        if (pair.first == ITYPE_TAG) {
                            visitor.SetItype(pair.second);
                        } else if (pair.first == TIER_TAG) {
                            if (pair.second == SELF) {
                                visitor.AddAggregated(TIER_TAG);
                            } else {
                                visitor.AddTag(TIER_TAG, pair.second);
                            }
                        } else {
                            return false;
                        }
                    }
                    visitor.AddAggregated(CTYPE_TAG);
                    visitor.AddAggregated(GEO_TAG);
                    visitor.AddAggregated(PRJ_TAG);
                    break;
                }
                case 5: {
                    for (const auto& pair : input) {
                        if (pair.first == ITYPE_TAG) {
                            visitor.SetItype(pair.second);
                        } else if (pair.first == TIER_TAG) {
                            if (pair.second == SELF) {
                                visitor.AddAggregated(TIER_TAG);
                            } else {
                                visitor.AddTag(TIER_TAG, pair.second);
                            }
                        } else {
                            visitor.AddTag(pair.first, pair.second);
                        }
                    }
                    break;
                }
                default: {
                    return false;
                }
            };
            return visitor.Finish();
        }

        inline bool ParseBsConfigStringSafe(TStringBuf incoming, TAccumulatingVisitor &visitor) {
            TSmallVec<TStringBuf> tokens;
            tokens.reserve(3);
            bool hasItype = false;
            for (const TStringBuf tag : StringSplitter(incoming).Split(' ')) {
                StringSplitter(tag).Split('_').Collect(&tokens);
                if (tokens.size() != 3 || (tokens[0] != "a" && tokens[0] != "itag")) {
                    // NOTE(rocco66): e.g. "a_itype_base", "itag_replica_0"
                    continue;
                }
                if (tokens[1] == ITYPE_TAG && IsCorrectInstanceTagValue(tokens[2])) {
                    hasItype = true;
                    visitor.SetItype(tokens[2]);
                } else if (IsCorrectTagName(tokens[1]) && IsCorrectInstanceTagValue(tokens[2])){
                    visitor.AddTag(tokens[1], tokens[2]);
                }
            }
            if (!hasItype) {
                ythrow yexception() << "there is no itype tag";
            }
            return visitor.Finish();
        }

        class TImplCache: public TNonCopyable {
        private:
            using TCacheMap = THashMap<TString, THolder<TInstanceKeyImpl>>;

            TLightRWLock Mutex;
            TCacheMap Cache;

            TImplCache() {
            }

        public:
            static TImplCache& Instance() {
                static TImplCache Self;
                return Self;
            }

            const TInstanceKeyImpl* Get(TStringBuf named) {
                TLightReadGuard rg(Mutex);
                const auto it = Cache.find(named);
                if (it != Cache.end()) {
                    return it->second.Get();
                } else {
                    return nullptr;
                }
            }

            const TInstanceKeyImpl* GetOrCreate(const TAccumulatingVisitor& visitor) {
                const TString named(visitor.ToNamed());

                {
                    TLightReadGuard rg(Mutex);
                    const auto it = Cache.find(named);
                    if (it != Cache.end()) {
                        return it->second.Get();
                    }
                }

                TLightWriteGuard wg(Mutex);

                TCacheMap::insert_ctx ctx;
                const auto it = Cache.find(named, ctx);
                if (it != Cache.end()) {
                    return it->second.Get();
                }

                THolder<TInstanceKeyImpl> impl(visitor.ToImpl());
                auto newIt = Cache.emplace_direct(ctx, named, std::move(impl));
                return newIt->second.Get();
            }
        };
    }

    TVector<TInstanceKey> TInstanceKey::CartesianProduct() const {
        if (Y_UNLIKELY(!Empty() && Impl->HasMultipleValues())) {
            // this branch is relatively rare because normally key hasn't multiple values

            NPrivate::TAccumulatingVisitor defaultVisitor;
            defaultVisitor.SetItype(GetItype());
            for (const auto& tag : GetAggregated()) {
                defaultVisitor.AddAggregated(tag);
            }

            TVector<NPrivate::TAccumulatingVisitor> variants;
            variants.emplace_back(std::move(defaultVisitor));

            // fill axes with possible tag values
            TMap<TStringBuf, TVector<TStringBuf>> axes;
            for (const auto& pair : GetTags()) {
                axes[pair.Name].emplace_back(pair.Value);
            }

            // O(N^3) is very bad but we didn't care about this branch
            for (const auto& axis : axes) {
                TVector<NPrivate::TAccumulatingVisitor> newVariants;
                for (auto& variant : variants) {
                    for (const auto& value : axis.second) {
                        newVariants.emplace_back(std::move(variant));
                        newVariants.back().AddTag(axis.first, value);
                    }
                }
                variants = std::move(newVariants);
            }

            TVector<TInstanceKey> result;
            result.reserve(variants.size());
            for (const auto& visitor : variants) {
                result.emplace_back(TInstanceKey(NPrivate::TImplCache::Instance().GetOrCreate(visitor)));
            }
            return result;
        } else {
            return {Impl};
        }
    }

    TInstanceKey TInstanceKey::FromNamed(TStringBuf input) {
        // maybe proper instance already exists
        const auto* impl(NPrivate::TImplCache::Instance().Get(input));
        if (impl != nullptr) {
            return {impl};
        }

        NPrivate::TAccumulatingVisitor visitor;
        if (!NPrivate::ParseNamedStringUnsafe(input, visitor)) {
            ythrow yexception() << "bad instance key given: " << input;
        }

        return {NPrivate::TImplCache::Instance().GetOrCreate(visitor)};
    }

    TInstanceKey TInstanceKey::FromUnderscore(TStringBuf input) {
        NPrivate::TAccumulatingVisitor visitor;
        if (!NPrivate::ParseUnderscoreStringUnsafe(input, visitor)) {
            ythrow yexception() << "bad instance key given: " << input;
        }
        return {NPrivate::TImplCache::Instance().GetOrCreate(visitor)};
    }

    TInstanceKey TInstanceKey::FromRtcEnviron() {
        TString tags {GetEnv("BSCONFIG_ITAGS", "a_itype_unknown")};
        TString host {GetEnv("HOSTNAME", "unknown")};
        return TInstanceKey::FromBsConfig(tags, host);
    }

    TInstanceKey TInstanceKey::FromBsConfig(TStringBuf input, TStringBuf host) {
        NPrivate::TAccumulatingVisitor visitor;
        if (IsCorrectInstanceTagValue(host)) {
            visitor.AddTag("host", host);
        } else {
            ythrow yexception() << "Invalid host value: " << host;
        }
        if (!NPrivate::ParseBsConfigStringSafe(input, visitor)) {
            ythrow yexception() << "bad instance key given: " << input;
        }
        return {NPrivate::TImplCache::Instance().GetOrCreate(visitor)};
    }

    TInstanceKey TInstanceKey::FromHistDb(const TVector<std::pair<TStringBuf, TStringBuf>>& input) {
        NPrivate::TAccumulatingVisitor visitor;
        if (!NPrivate::ParseHistDbUnsafe(input, visitor)) {
            ythrow yexception() << "bad instance key given";
        }
        return {NPrivate::TImplCache::Instance().GetOrCreate(visitor)};
    }

    TInstanceKey TInstanceKey::FromParts(TStringBuf itype, const TVector<std::pair<TStringBuf, TStringBuf>>& tags, const TVector<TStringBuf>& aggregated) {
        NPrivate::TAccumulatingVisitor visitor;
        visitor.SetItype(itype);
        for (const auto& [key, value] : tags) {
            visitor.AddTag(key, value);
        }
        for (const auto& tag : aggregated) {
            visitor.AddAggregated(tag);
        }
        visitor.Finish();
        return {NPrivate::TImplCache::Instance().GetOrCreate(visitor)};
    }

    TVector<TInstanceKey> TInstanceKey::CreateMany(std::span<const TString* const> names) {
        TVector<TInstanceKey> result(Reserve(names.size()));
        for (const auto& name : names) {
            result.emplace_back(FromNamed(*name));
        }
        return result;
    }

    TInstanceKey TInstanceKey::FromAgent(TStringBuf itype, TStringBuf tail) {
        const TString named(TStringBuilder() << itype << "|" << tail);
        try {
            return NTags::TInstanceKey::FromNamed(named);
        } catch(...) {
            const TString underscored(TStringBuilder() << itype << "_" << tail);
            try {
                return NTags::TInstanceKey::FromUnderscore(underscored);
            } catch(...) {
                return {};
            }
        }
    }

    const TString& TInstanceKey::ToNamed() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return NPrivate::EMPTY_STRING;
        } else {
            return Impl->GetNamed();
        }
    }

    TStringBuf TInstanceKey::GetItype() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return NPrivate::EMPTY_STRING;
        } else {
            return Impl->GetItype();
        }
    }

    TInternedTagNameSet TInstanceKey::GetTagNameSet() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return NPrivate::EMPTY_TAG_NAME_SET;
        } else {
            return Impl->GetTagNameSet();
        }
    }

    TInternedTagNameSet TInstanceKey::GetTagsAndAggregatedNameSet() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return NPrivate::EMPTY_TAG_NAME_SET;
        } else {
            return Impl->GetTagsAndAggregatedNameSet();
        }
    }

    NPrivate::TInstanceKeyTags TInstanceKey::GetTags() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return NPrivate::EMPTY_TAGS;
        } else {
            return Impl->GetTags();
        }
    }

    NPrivate::TTagValuePairs TInstanceKey::GetTagValuePairs() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return NPrivate::EMPTY_TAG_VALUE_PAIRS;
        } else {
            NPrivate::TTagValuePairs pairs;
            for (const auto& tag : GetTags()) {
                pairs.emplace_back(tag.Name, tag.Value);
            }
            return pairs;
        }
    }

    NPrivate::TAggregated TInstanceKey::GetAggregated() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return NPrivate::EMPTY_AGGREGATED;
        } else {
            return Impl->GetAggregated();
        }
    }

    THostName TInstanceKey::GetHostName() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return {};
        } else {
            return Impl->GetHostName();
        }
    }

    THostName TInstanceKey::GetGroupName() const noexcept {
        if (Y_UNLIKELY(Empty())) {
            return {};
        } else {
            return Impl->GetGroupName();
        }
    }

    TInstanceKey TInstanceKey::AggregateBy(const TVector<TStringBuf>& tagNames) const noexcept {
        return AggregateBy(TTagNameSet(tagNames));
    }

    TInstanceKey TInstanceKey::AggregateBy(TInternedTagNameSet tagNameSet) const noexcept {
        const auto* relatedImpl(Impl->FindAggregatedBy(tagNameSet));
        if (relatedImpl != nullptr) {
            return {relatedImpl};
        }

        NPrivate::TAccumulatingVisitor visitor;
        visitor.SetItype(GetItype());
        for (const auto& tag : GetTags()) {
            if (tagNameSet.Has(tag.InternedName)) {
                visitor.AddAggregated(tag.Name);
            } else {
                visitor.AddTag(tag.Name, tag.Value);
            }
        }
        for (const auto& tag : GetAggregated()) {
            visitor.AddAggregated(tag);
        }
        visitor.Finish();

        relatedImpl = NPrivate::TImplCache::Instance().GetOrCreate(visitor);
        Impl->SetAggregatedBy(tagNameSet, relatedImpl);
        return {relatedImpl};
    }

    TInstanceKey TInstanceKey::SetTag(TStringBuf tagName, TStringBuf value) const noexcept {
        NPrivate::TAccumulatingVisitor visitor;
        visitor.SetItype(GetItype());
        visitor.AddTag(tagName, value);
        for (const auto& tag : GetTags()) {
            if (tag.Name != tagName) {
                visitor.AddTag(tag.Name, tag.Value);
            }
        }
        for (const auto& tag : GetAggregated()) {
            if (tag != tagName) {
                visitor.AddAggregated(tag);
            }
        }
        visitor.Finish();

        return {NPrivate::TImplCache::Instance().GetOrCreate(visitor)};
    }

    TInstanceKey TInstanceKey::SetGroupAndHost(THostName group, THostName host) const noexcept {
        NPrivate::TAccumulatingVisitor visitor;

        visitor.SetItype(GetItype());
        if (!group.Empty()) {
            visitor.AddTag(NPrivate::GROUP_TAG, group.GetName());
        }
        if (host.Empty()) {
            visitor.AddAggregated(NPrivate::HOST_TAG);
        } else {
            visitor.AddTag(NPrivate::HOST_TAG, host.GetName());
        }
        for (const auto& tag : GetTags()) {
            if (tag.Name == NPrivate::GROUP_TAG || tag.Name == NPrivate::HOST_TAG) {
                continue;
            }
            visitor.AddTag(tag.Name, tag.Value);
        }
        for (const auto& tag : GetAggregated()) {
            if (tag == NPrivate::GROUP_TAG || tag == NPrivate::HOST_TAG) {
                continue;
            }
            visitor.AddAggregated(tag);
        }
        visitor.Finish();

        const auto* relatedImpl = NPrivate::TImplCache::Instance().GetOrCreate(visitor);
        return {relatedImpl};
    }

    TInstanceKey TInstanceKey::AddGroupAndHost(THostName group, THostName host) const noexcept {
        // Redefines host from YASM_CONTAINER_TAG if presents specified
        const auto* relatedImpl(Impl->FindByGroupAndHost(group, host));
        if (relatedImpl != nullptr) {
            return {relatedImpl};
        }

        NPrivate::TAccumulatingVisitor visitor;
        visitor.SetItype(GetItype());
        visitor.AddTag(NPrivate::GROUP_TAG, group.GetName());
        TStringBuf hostName = host.GetName();
        for (const auto& tag : GetTags()) {
            if (tag.Name != NPrivate::YASM_CONTAINER_TAG) {
                visitor.AddTag(tag.Name, tag.Value);
            } else {
                hostName = tag.Value;
            }
        }
        if (host.Empty()) {
            visitor.AddAggregated(NPrivate::HOST_TAG);
        } else {
            visitor.AddTag(NPrivate::HOST_TAG, hostName);
        }
        for (const auto& tag : GetAggregated()) {
            visitor.AddAggregated(tag);
        }
        visitor.Finish();

        relatedImpl = NPrivate::TImplCache::Instance().GetOrCreate(visitor);
        Impl->SetForGroupAndHost(group, host, relatedImpl);
        return {relatedImpl};
    }

    size_t TInstanceKey::GetValueHash(TInternedTagNameSet tagNameSet) const noexcept {
        size_t result = ComputeHash(GetItype());
        for (const auto& tag : GetTags()) {
            if (tagNameSet.Has(tag.InternedName)) {
                result = CombineHashes(result, tag.ValueHash);
            }
        }
        return result;
    }

    TInstanceKey TInstanceKey::RemoveGroup() const noexcept {
        const auto* relatedImpl(Impl->GetGroupless());
        if (relatedImpl != nullptr) {
            return {relatedImpl};
        }

        NPrivate::TAccumulatingVisitor visitor;
        visitor.SetItype(GetItype());
        for (const auto& tag : GetTags()) {
            if (tag.Name != NPrivate::GROUP_TAG) {
                visitor.AddTag(tag.Name, tag.Value);
            }
        }
        for (const auto& tag : GetAggregated()) {
            visitor.AddAggregated(tag);
        }
        visitor.Finish();

        relatedImpl = NPrivate::TImplCache::Instance().GetOrCreate(visitor);
        Impl->SetGroupless(relatedImpl);
        return {relatedImpl};
    }

    bool TInstanceKey::IsSmallAggregate() const {
        return GetAggregated() == NPrivate::TAggregated(&NPrivate::TIER_TAG, 1);
    }
}

template <>
void Out<NTags::TInstanceKey>(IOutputStream& stream,
                              TTypeTraits<NTags::TInstanceKey>::TFuncParam key) {
    stream.Write(key.ToNamed());
}
