#pragma once

#include <infra/yasm/common/labels/tags/tag_name.h>
#include <infra/yasm/common/labels/tags/interned_tag_name_set.h>

#include <infra/yasm/common/labels/host/host.h>

#include <library/cpp/containers/stack_vector/stack_vec.h>
#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/digest/multi.h>
#include <util/string/split.h>
#include <util/string/builder.h>
#include <library/cpp/deprecated/split/delim_string_iter.h>
#include <util/generic/strbuf.h>
#include <util/generic/algorithm.h>
#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/generic/array_ref.h>

#include <array>
#include <span>

namespace NTags {
    namespace NPrivate {
        struct TInstanceKeyTag {
            TStringBuf Name;
            TTagName InternedName;

            TStringBuf Value;
            size_t ValueHash;
        };
        using TInstanceKeyTags = TArrayRef<const TInstanceKeyTag>;

        using TAggregated = TArrayRef<const TStringBuf>;

        using TTagValuePairs = TVector<std::pair<TStringBuf, TStringBuf>>;

        class TInstanceKeyImpl;
    }

    class TInstanceKey {
    public:
        inline TInstanceKey() noexcept
            : TInstanceKey(nullptr)
        {
        }

        inline TInstanceKey(const TInstanceKey& other) noexcept
            : TInstanceKey(other.Impl)
        {
        }

        inline TInstanceKey(TInstanceKey&& other) noexcept
            : TInstanceKey(other.Impl)
        {
        }

        inline TInstanceKey& operator=(const TInstanceKey& other) noexcept {
            Impl = other.Impl;
            return *this;
        }

        static TInstanceKey FromNamed(TStringBuf input);
        static TInstanceKey FromUnderscore(TStringBuf input);
        static TInstanceKey FromRtcEnviron();
        static TInstanceKey FromBsConfig(TStringBuf input, TStringBuf host);
        static TInstanceKey FromAgent(TStringBuf itype, TStringBuf tail);
        static TInstanceKey FromHistDb(const TVector<std::pair<TStringBuf, TStringBuf>>& input);
        static TInstanceKey FromParts(TStringBuf itype, const TVector<std::pair<TStringBuf, TStringBuf>>& tags, const TVector<TStringBuf>& aggregated);

        static TVector<TInstanceKey> CreateMany(std::span<const TString* const> names);

        const TString& ToNamed() const noexcept;

        TStringBuf GetItype() const noexcept;
        NPrivate::TInstanceKeyTags GetTags() const noexcept;
        NPrivate::TTagValuePairs GetTagValuePairs() const noexcept;
        NPrivate::TAggregated GetAggregated() const noexcept;

        NZoom::NHost::THostName GetHostName() const noexcept;
        NZoom::NHost::THostName GetGroupName() const noexcept;

        TVector<TInstanceKey> CartesianProduct() const;

        TInternedTagNameSet GetTagNameSet() const noexcept;
        TInternedTagNameSet GetTagsAndAggregatedNameSet() const noexcept;

        TInstanceKey AggregateBy(const TVector<TStringBuf>& tagNames) const noexcept;
        TInstanceKey AggregateBy(TInternedTagNameSet tagNameSet) const noexcept;
        TInstanceKey SetTag(TStringBuf tagName, TStringBuf value) const noexcept;

        TInstanceKey SetGroupAndHost(NZoom::NHost::THostName group, NZoom::NHost::THostName host) const noexcept;
        TInstanceKey AddGroupAndHost(NZoom::NHost::THostName group, NZoom::NHost::THostName host) const noexcept;
        TInstanceKey RemoveGroup() const noexcept;

        size_t GetValueHash(TInternedTagNameSet tagNameSet) const noexcept;

        bool IsSmallAggregate() const;

        inline bool Empty() const noexcept {
            return Impl == nullptr;
        }

        size_t Hash() const noexcept {
            return reinterpret_cast<size_t>(Impl);
        }

        bool operator==(const TInstanceKey& other) const noexcept {
            return Impl == other.Impl;
        }

        bool operator!=(const TInstanceKey& other) const noexcept {
            return Impl != other.Impl;
        }

        bool operator<(const TInstanceKey& other) const noexcept {
            return Impl < other.Impl;
        }

    private:
        inline TInstanceKey(const NPrivate::TInstanceKeyImpl* impl)
            : Impl(impl)
        {
        }

        const NPrivate::TInstanceKeyImpl* Impl = nullptr;
    };
}

template <>
struct THash<NTags::TInstanceKey> {
    inline size_t operator()(const NTags::TInstanceKey& v) const noexcept {
        return v.Hash();
    }
};

template <>
struct THash<typename NTags::NPrivate::TAggregated> {
    inline size_t operator()(const NTags::NPrivate::TAggregated& aggregated) const noexcept {
        size_t hash = 0;
        for (const auto& aggr : aggregated) {
            hash = CombineHashes(THash<TStringBuf>()(aggr), hash);
        }
        return hash;
    }
};
