#include <infra/yasm/common/labels/tags/instance_key.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/stream/tokenizer.h>
#include <util/generic/set.h>
#include <util/generic/xrange.h>

using namespace NTags;
using namespace NZoom::NHost;

class TTagsTest: public TTestBase {
    UNIT_TEST_SUITE(TTagsTest);
    UNIT_TEST(TestEmpty)
    UNIT_TEST(TestHistDb)
    UNIT_TEST(TestBsConfig)
    UNIT_TEST(TestBsConfigInvalidTags)
    UNIT_TEST(TestOnRealData)
    UNIT_TEST(TestForEquality)
    UNIT_TEST(TestHasing)
    UNIT_TEST(TestAggregatedUnderscore)
    UNIT_TEST(TestNonAggregatedUnderscore)
    UNIT_TEST(TestNonAggregatedUnderscoreWithTier)
    UNIT_TEST(TestAggregatedAndNamed)
    UNIT_TEST(TestAggregatedAndNamedWithoutTags)
    UNIT_TEST(TestNonAggregatedAndNamed)
    UNIT_TEST(TestNamedWithMultipleValues)
    UNIT_TEST(TestTags)
    UNIT_TEST(TestMultipleTags)
    UNIT_TEST(TestAggregated)
    UNIT_TEST(TestCartesianProduct)
    UNIT_TEST(TestAgent)
    UNIT_TEST(TestAggregateBy)
    UNIT_TEST(TestGroupAndHost)
    UNIT_TEST(TestSetGroupAndHost)
    UNIT_TEST(TestGroupAndHostContainer)
    UNIT_TEST(TestGroupRemoval)
    UNIT_TEST(TestSmallAggregate)
    UNIT_TEST(TestCreateMany)
    UNIT_TEST_SUITE_END();

private:
    inline void TestEmpty() {
        UNIT_ASSERT(!TInstanceKey::FromNamed("common||ctype,geo,prj,tier").Empty());

        TInstanceKey key;
        UNIT_ASSERT(key.Empty());
        UNIT_ASSERT_STRINGS_EQUAL(key.ToNamed(), "");
        UNIT_ASSERT_STRINGS_EQUAL(key.GetItype(), "");
        UNIT_ASSERT(key.GetTags().empty());
        UNIT_ASSERT(key.GetAggregated().empty());
    }

    inline void TestHistDb() {
        UNIT_ASSERT_VALUES_EQUAL(
            TInstanceKey::FromHistDb({{"itype", "common"}, {"tier", "self"}}),
            TInstanceKey::FromNamed("common||ctype,geo,prj,tier"));
        UNIT_ASSERT_VALUES_EQUAL(
            TInstanceKey::FromHistDb({{"itype", "addrsupper"}, {"ctype", "prestable"}, {"geo", "sas"}, {"prj", "addrs"}, {"tier", "self"}}),
            TInstanceKey::FromNamed("addrsupper|ctype=prestable;geo=sas;prj=addrs|tier"));
    }

    inline void TestBsConfig() {
        // NOTE(rocco66): copy/pase from prod process environment variables
        TString HOSTNAME {"man1-0161.search.yandex.net"};
        TString BSCONFIG_ITAGS {
            "MAN_YASM_LINES_DUMPER a_ctype_prod a_dc_man a_geo_man a_itype_yasmdumper "
            "a_line_man2 a_metaprj_internal a_prj_yasm-lines a_tier_none cgset_memory_recharge_on_pgfault_1"
            " itag_replica_0 extended_loopscript production portopowered use_hq_spec enable_hq_report "
            "enable_hq_poll"
        };
        UNIT_ASSERT_VALUES_EQUAL(
            TInstanceKey::FromBsConfig(BSCONFIG_ITAGS, HOSTNAME),
            TInstanceKey::FromNamed(
                "yasmdumper|host=man1-0161.search.yandex.net;ctype=prod;dc=man;geo=man;"
                "line=man2;metaprj=internal;prj=yasm-lines;tier=none;replica=0"
            )
        );
    }

    inline void TestBsConfigInvalidTags() {
        TString HOSTNAME {"man1-0161.search.yandex.net"};
        TString BSCONFIG_ITAGS_INVALID_CHAR {"a_ctype_prod^ a_itype_yasmdumper"};
        UNIT_ASSERT_VALUES_EQUAL(
            TInstanceKey::FromBsConfig(BSCONFIG_ITAGS_INVALID_CHAR, HOSTNAME),
            TInstanceKey::FromNamed("yasmdumper|host=man1-0161.search.yandex.net")
        );

        TString BSCONFIG_ITAGS_ITYPELESS {"a_ctype_prod"};
        UNIT_ASSERT_EXCEPTION(
            TInstanceKey::FromBsConfig(BSCONFIG_ITAGS_ITYPELESS, HOSTNAME),
            yexception
        );
    }

    inline void TestOnRealData() {
        TFileInput input("test_instance_keys.txt");
        TStreamTokenizer<TEol> tokenizer{&input};
        for (const auto& line : tokenizer) {
            const TString input(line);
            AssertIsNormalizedNamed(input);
        }
    }

    inline void TestForEquality() {
        UNIT_ASSERT_EQUAL(
            TInstanceKey::FromNamed("common||ctype,geo,prj,tier"),
            TInstanceKey::FromUnderscore("common_self"));
        UNIT_ASSERT_EQUAL(
            TInstanceKey::FromNamed("addrsupper|ctype=prestable;geo=sas;prj=addrs|tier"),
            TInstanceKey::FromNamed("addrsupper|geo=sas;prj=addrs;ctype=prestable|tier"));
        UNIT_ASSERT_UNEQUAL(
            TInstanceKey::FromNamed("addrsupper|ctype=prestable;geo=sas;prj=addrs|tier"),
            TInstanceKey::FromNamed("addrsupper|ctype=prestable;geo=man;prj=addrs|tier"));
    }

    inline void TestHasing() {
        THashSet<TInstanceKey> keys;
        keys.emplace(TInstanceKey::FromNamed("addrsupper|ctype=prestable;geo=sas;prj=addrs|tier"));
        keys.emplace(TInstanceKey::FromNamed("addrsupper|ctype=prestable;geo=man;prj=addrs|tier"));
        keys.emplace(TInstanceKey::FromNamed("addrsupper|ctype=prestable;geo=sas;prj=addrs|tier"));
        UNIT_ASSERT_EQUAL(keys.size(), 2);
    }

    inline void TestAggregatedUnderscore() {
        TString input("common_self");
        TString output("common||ctype,geo,prj,tier");
        AssertIsNormalizedUnderscore(input, output);
    }

    inline void TestNonAggregatedUnderscore() {
        TString input("qloud_unknown_oort.oort-production-installations.oortqloudextproduction1345_myt_self");
        TString output("qloud|ctype=unknown;geo=myt;prj=oort.oort-production-installations.oortqloudextproduction1345|tier");
        AssertIsNormalizedUnderscore(input, output);
    }

    inline void TestNonAggregatedUnderscoreWithTier() {
        TString input("qloud_unknown_oort.oort-production-installations.oortqloudextproduction1345_myt_1");
        TString output("qloud|ctype=unknown;geo=myt;prj=oort.oort-production-installations.oortqloudextproduction1345;tier=1");
        AssertIsNormalizedUnderscore(input, output);
    }

    inline void TestAggregatedAndNamed() {
        AssertIsNormalizedNamed("addrsupper|ctype=prestable;geo=sas;prj=addrs|tier");
    }

    inline void TestAggregatedAndNamedWithoutTags() {
        AssertIsNormalizedNamed("addrsupper||ctype,geo,prj,tier");
    }

    inline void TestNonAggregatedAndNamed() {
        AssertIsNormalizedNamed("yasmhserver|ctype=prod;geo=sas;prj=yasm-lines;tier=none");
    }

    inline void TestNamedWithMultipleValues() {
        AssertIsNormalizedNamed("yasmhserver|ctype=prod;geo=sas;prj=yasm-kernel,yasm-lines;tier=none,something");
    }

    inline void TestTags() {
        auto first(TInstanceKey::FromNamed("common|ctype=prod;geo=sas;prj=yasm-lines"));
        UNIT_ASSERT_EQUAL(first.GetTagValuePairs(), NTags::NPrivate::TTagValuePairs({std::make_pair(TStringBuf("ctype"), TStringBuf("prod")),
                                                                                     std::make_pair(TStringBuf("geo"), TStringBuf("sas")),
                                                                                     std::make_pair(TStringBuf("prj"), TStringBuf("yasm-lines"))}));
        UNIT_ASSERT_EQUAL(first.GetAggregated(), TArrayRef<const TStringBuf>());

        auto second(TInstanceKey::FromNamed("common|ctype=prod;geo=sas;prj=yasm-lines;slow=first;yellow=second"));
        UNIT_ASSERT_EQUAL(second.GetTagValuePairs(), NTags::NPrivate::TTagValuePairs({std::make_pair(TStringBuf("ctype"), TStringBuf("prod")),
                                                                                      std::make_pair(TStringBuf("geo"), TStringBuf("sas")),
                                                                                      std::make_pair(TStringBuf("prj"), TStringBuf("yasm-lines")),
                                                                                      std::make_pair(TStringBuf("slow"), TStringBuf("first")),
                                                                                      std::make_pair(TStringBuf("yellow"), TStringBuf("second"))}));
        UNIT_ASSERT_EQUAL(second.GetAggregated(), TArrayRef<const TStringBuf>());
    }

    inline void TestMultipleTags() {
        TSet<TString> reference;
        for (auto idx : xrange(15)) {
            reference.emplace(Sprintf("%3d", idx));
        }

        TStringBuilder named;
        named << "common|";
        for (const auto& tag : reference) {
            named << tag << "=" << tag << ";";
        }
        named << "finish=finish";
        reference.emplace("finish");

        TSet<TString> found;
        for (const auto& pair : TInstanceKey::FromNamed(named).GetTags()) {
            found.emplace(pair.Name);
        }

        UNIT_ASSERT_EQUAL(reference, found);
    }

    inline void TestAggregated() {
        auto first(TInstanceKey::FromNamed("common||ctype,geo,prj"));
        UNIT_ASSERT_EQUAL(first.GetAggregated(), TArrayRef<const TStringBuf>({TStringBuf("ctype"),
                                                                              TStringBuf("geo"),
                                                                              TStringBuf("prj")}));
        UNIT_ASSERT_EQUAL(first.GetTagValuePairs(), NTags::NPrivate::TTagValuePairs{});

        auto second(TInstanceKey::FromNamed("common||ctype,geo,prj,slow,yellow"));
        UNIT_ASSERT_EQUAL(second.GetAggregated(), TArrayRef<const TStringBuf>({TStringBuf("ctype"),
                                                                               TStringBuf("geo"),
                                                                               TStringBuf("prj"),
                                                                               TStringBuf("slow"),
                                                                               TStringBuf("yellow")}));
        UNIT_ASSERT_EQUAL(second.GetTagValuePairs(), NTags::NPrivate::TTagValuePairs{});
    }

    inline void TestCartesianProduct() {
        auto first(TInstanceKey::FromNamed("common|ctype=prod;geo=sas;prj=yasm-lines"));
        UNIT_ASSERT_EQUAL(first.CartesianProduct(), TVector<TInstanceKey>({first}));

        auto second(TInstanceKey::FromNamed("common|ctype=prod;geo=sas,man;prj=yasm-lines,yasm-kernel"));
        UNIT_ASSERT_EQUAL(second.CartesianProduct(), TVector<TInstanceKey>({TInstanceKey::FromNamed("common|ctype=prod;geo=man;prj=yasm-kernel"),
                                                                            TInstanceKey::FromNamed("common|ctype=prod;geo=man;prj=yasm-lines"),
                                                                            TInstanceKey::FromNamed("common|ctype=prod;geo=sas;prj=yasm-kernel"),
                                                                            TInstanceKey::FromNamed("common|ctype=prod;geo=sas;prj=yasm-lines")}));
    }

    inline void TestAgent() {
        UNIT_ASSERT_VALUES_EQUAL(
            TInstanceKey::FromNamed("balancer|ctype=prestable;geo=sas;prj=users-playlist;tier=zagorod"),
            TInstanceKey::FromAgent("balancer", "ctype=prestable;geo=sas;prj=users-playlist;tier=zagorod"));
        UNIT_ASSERT_VALUES_EQUAL(
            TInstanceKey::FromNamed("balancer|ctype=prestable;geo=sas;prj=users-playlist|tier"),
            TInstanceKey::FromAgent("balancer", "ctype=prestable;geo=sas;prj=users-playlist|tier"));
        UNIT_ASSERT_VALUES_EQUAL(
            TInstanceKey::FromNamed("balancer||ctype,geo,prj,tier"),
            TInstanceKey::FromAgent("balancer", "|ctype,geo,prj,tier"));
        UNIT_ASSERT_VALUES_EQUAL(
            TInstanceKey::FromNamed("common||ctype,geo,prj,tier"),
            TInstanceKey::FromAgent("common", "self"));
    }

    inline void TestAggregateBy() {
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable;geo=sas;prj=users-playlist;tier=zagorod"));
            const auto aggregatedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;tier=zagorod|geo,prj"));
            UNIT_ASSERT_VALUES_EQUAL(rawKey.AggregateBy(TVector<TStringBuf>{"geo", "prj"}), aggregatedKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable;prj=users-playlist;tier=zagorod"));
            const auto aggregatedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;tier=zagorod|prj"));
            UNIT_ASSERT_VALUES_EQUAL(rawKey.AggregateBy(TVector<TStringBuf>{"geo", "prj"}), aggregatedKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable|prj"));
            UNIT_ASSERT_VALUES_EQUAL(rawKey.AggregateBy(TVector<TStringBuf>{"geo", "prj"}), rawKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer||geo,prj"));
            UNIT_ASSERT_VALUES_EQUAL(rawKey.AggregateBy(TVector<TStringBuf>{"geo", "prj"}), rawKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable;prj=users-playlist|tier"));
            const auto aggregatedKey(TInstanceKey::FromNamed("balancer|ctype=prestable|prj,tier"));
            UNIT_ASSERT_VALUES_EQUAL(rawKey.AggregateBy(TVector<TStringBuf>{"prj"}), aggregatedKey);
        }
    }

    inline void TestGroupAndHost() {
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable|tier"));
            UNIT_ASSERT(rawKey.GetHostName().Empty());

            const auto modifiedKey(rawKey.AddGroupAndHost(TStringBuf("SAS.000"), TStringBuf("sas1-1234")));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000;host=sas1-1234|tier"));
            UNIT_ASSERT_VALUES_EQUAL(expectedKey.GetHostName(), THostName(TStringBuf("sas1-1234")));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000;host=sas1-1234|tier"));
            const auto modifiedKey(rawKey.AggregateBy(TVector<TStringBuf>{"host"}));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000|host,tier"));
            UNIT_ASSERT_VALUES_EQUAL(expectedKey.GetHostName(), THostName(TStringBuf("SAS.000")));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable|tier"));

            const THostName emptyHost;
            const auto modifiedKey(rawKey.AddGroupAndHost(TStringBuf("SAS.000"), emptyHost));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000|host,tier"));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
    }

    inline void TestSetGroupAndHost() {
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable|tier"));
            const auto modifiedKey(rawKey.SetGroupAndHost(TStringBuf("SAS.000"), TStringBuf("sas1-1234")));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000;host=sas1-1234|tier"));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable|tier"));
            const auto modifiedKey(rawKey.SetGroupAndHost(TStringBuf("SAS.000"), THostName()));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000|host,tier"));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.001;host=sas1-1234|tier"));
            const auto modifiedKey(rawKey.SetGroupAndHost(TStringBuf("SAS.000"), THostName()));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000|host,tier"));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.001;host=sas1-1234|tier"));
            const auto modifiedKey(rawKey.SetGroupAndHost(THostName(), THostName()));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable|host,tier"));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
    }

    inline void TestGroupAndHostContainer() {
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable;yasm_container=bylcf2kcwkknnhq6.sas.yp-c.yandex.net|tier"));

            const auto modifiedKey(rawKey.AddGroupAndHost(TStringBuf("SAS.000"), TStringBuf("sas1-1234")));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000;host=bylcf2kcwkknnhq6.sas.yp-c.yandex.net|tier"));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
        {
            const auto rawKey(TInstanceKey::FromNamed("balancer|ctype=prestable;yasm_container=bylcf2kcwkknnhq6.sas.yp-c.yandex.net|tier"));

            const THostName emptyHost;
            const auto modifiedKey(rawKey.AddGroupAndHost(TStringBuf("SAS.000"), emptyHost));
            const auto expectedKey(TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000|host,tier"));
            UNIT_ASSERT_VALUES_EQUAL(modifiedKey, expectedKey);
        }
    }

    inline void TestGroupRemoval() {
        const auto key1 = TInstanceKey::FromNamed("balancer|ctype=prestable;group=SAS.000;host=sas1-1234|prj,tier")
            .RemoveGroup();
        const auto expectedKey = TInstanceKey::FromNamed("balancer|ctype=prestable;host=sas1-1234|prj,tier");
        UNIT_ASSERT_VALUES_EQUAL(key1, expectedKey);
        const auto key2 = key1.RemoveGroup();
        UNIT_ASSERT_VALUES_EQUAL(key2, expectedKey);
        const auto key3 = TInstanceKey::FromNamed("balancer|ctype=prestable|prj,tier")
            .AddGroupAndHost(TStringBuf("SAS.000"), TStringBuf("sas1-1234"))
            .RemoveGroup();
        UNIT_ASSERT_VALUES_EQUAL(key3, expectedKey);
    }

    inline void TestSmallAggregate() {
        const auto noAggregate(TInstanceKey::FromNamed("balancer|ctype=prestable;prj=users-playlist;tier=zagorod"));
        UNIT_ASSERT(!noAggregate.IsSmallAggregate());

        const auto hasSmallAggregate(TInstanceKey::FromNamed("balancer|ctype=prestable;prj=users-playlist|tier"));
        UNIT_ASSERT(hasSmallAggregate.IsSmallAggregate());

        const auto hasSomeAggregate(TInstanceKey::FromNamed("balancer||ctype,prj,tier"));
        UNIT_ASSERT(!hasSomeAggregate.IsSmallAggregate());
    }

    inline void TestCreateMany() {
        TVector<TString> names{"common||ctype,geo,prj,tier", "upper||ctype,geo,prj,tier"};
        TVector<TInstanceKey> keys = TInstanceKey::CreateMany(TVector{&names[0], &names[1]});
        UNIT_ASSERT_VALUES_EQUAL(keys.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(keys.at(0), TInstanceKey::FromNamed("common||ctype,geo,prj,tier"));
        UNIT_ASSERT_VALUES_EQUAL(keys.at(1), TInstanceKey::FromNamed("upper||ctype,geo,prj,tier"));
    }

    inline void AssertIsNormalizedNamed(const TString& input) {
        const auto instanceKey(TInstanceKey::FromNamed(input));
        UNIT_ASSERT_STRINGS_EQUAL(input, instanceKey.ToNamed());
    }

    inline void AssertIsNormalizedUnderscore(const TString& input, const TString& output) {
        const auto instanceKey(TInstanceKey::FromUnderscore(input));
        UNIT_ASSERT_STRINGS_EQUAL(output, instanceKey.ToNamed());
    }
};

UNIT_TEST_SUITE_REGISTRATION(TTagsTest);
