#include "request_key.h"
#include "verification.h"

#include <util/digest/multi.h>
#include <util/generic/hash.h>
#include <util/generic/xrange.h>
#include <util/string/join.h>
#include <util/string/strip.h>

#include <re2/re2.h>

using namespace NTags;

namespace {
    using namespace NTags::NPrivate;

    class TEqMatcher final : public TMatcher {
    public:
        TEqMatcher(TStringBuf raw)
            : TMatcher(raw)
        {
        }

        bool Match(TStringBuf value) const override {
            return value == Raw;
        }

        void Visit(IRequestKeyVisitor& visitor) const override {
            visitor.OnEqualValues(Raw);
        }
    };

    class TRegexpMatcher final : public TMatcher {
    public:
        TRegexpMatcher(TStringBuf raw)
            : TMatcher(raw)
            , Regexp(CreateRegexp(Raw))
        {
        }

        bool Match(TStringBuf value) const override {
            return RE2::FullMatch(re2::StringPiece(value), Regexp);
        }

        void Visit(IRequestKeyVisitor& visitor) const override {
            visitor.OnRegexpValues(Raw);
        }

    private:
        static TString CreateRegexp(TStringBuf base) {
            return JoinSeq(".*", StringSplitter(base).Split('*').ToList<TStringBuf>());
        }

        const re2::RE2 Regexp;
    };

    class TListMatcher final : public TMatcher {
    public:
        TListMatcher(TStringBuf raw, TVector<TMatcherPtr>&& list)
            : TMatcher(raw)
            , List(std::move(list))
        {
        }

        bool Match(TStringBuf value) const override {
            for (const TMatcherPtr& matcher : List) {
                if (matcher->Match(value)) {
                    return true;
                }
            }
            return false;
        }

        void Visit(IRequestKeyVisitor& visitor) const override {
            visitor.OnGroupStart(List.size());
            for (const auto& matcher : List) {
                matcher->Visit(visitor);
            }
            visitor.OnGroupEnd();
        }

    private:
        const TVector<TMatcherPtr> List;
    };

    TMatcherPtr CreateMatcher(const TStringBuf tagName, const TStringBuf tagValue) {
        if (tagValue == "*" || (tagName == TIER_TAG && tagValue == "self")) {
            return nullptr;
        }

        if (!IsCorrectRequestTagValue(tagValue)) {
            ythrow yexception() << "incorrect tag " << tagValue;
        }

        if (tagValue.Contains("*")) {
            return MakeHolder<TRegexpMatcher>(tagValue);
        }

        return MakeHolder<TEqMatcher>(tagValue);
    }

    TMatcherPtr CreateMatcher(const TStringBuf tagName, TSmallVec<TStringBuf> tagValues) {
        for (TStringBuf& tagValue : tagValues) {
            tagValue = StripString(tagValue);
        }

        Sort(tagValues);

        TVector<TMatcherPtr> matchers;
        for (TStringBuf tagValue : tagValues) {
            TMatcherPtr matcher = CreateMatcher(tagName, tagValue);
            if (!matcher) {
                return nullptr;
            }
            matchers.emplace_back(std::move(matcher));
        }
        if (matchers.size() == 1) {
            return std::move(matchers[0]);
        }
        return MakeHolder<TListMatcher>(JoinSeq(",", tagValues), std::move(matchers));
    }

    struct TDictBuilder final : public IRequestKeyVisitor {
        void OnKey(TStringBuf key) override {
            CurrentKey = key;
        }

        void OnEqualValues(TStringBuf value) override {
            Result[CurrentKey].emplace_back(value);
        }

        void OnRegexpValues(TStringBuf value) override {
            Result[CurrentKey].emplace_back(value);
        }

        TMap<TStringBuf, TSmallVec<TStringBuf>> Result;
        TStringBuf CurrentKey;
    };

    struct TEqualBuilder final : public IRequestKeyVisitor {
        void OnTagCount(size_t count) override {
            EqualTags.reserve(count);
            EqualValues.reserve(count);
        }

        void OnKey(TStringBuf key) override {
            CurrentKey = key;
        }

        void OnGroupStart(size_t) override {
            EqualMatcher = false;
            HasOnlyEqualMatcher = false;
        }

        void OnGroupEnd() override {
            EqualMatcher = true;
        }

        void OnEqualValues(TStringBuf value) override {
            if (EqualMatcher) {
                EqualTags.emplace_back(CurrentKey);
                EqualValues.emplace_back(value);
            }
        }

        void OnRegexpValues(TStringBuf) override {
            HasOnlyEqualMatcher = false;
        }

        TVector<TStringBuf> EqualTags;
        TVector<TStringBuf> EqualValues;
        TStringBuf CurrentKey;
        bool HasOnlyEqualMatcher = true;
        bool EqualMatcher = true;
    };

    struct TTagNameSetBuilder final : public IRequestKeyVisitor {
        void OnKey(TStringBuf key) override {
            CurrentKey = key;
        }

        void OnGroupStart(size_t) override {
            EqualMatcher = false;
        }

        void OnGroupEnd() override {
            EqualMatcher = true;
        }

        void OnEqualValues(TStringBuf) override {
            if (EqualMatcher) {
                TagNames.emplace_back(CurrentKey);
            }
        }

        void OnRegexpValues(TStringBuf) override {
        }

        TStringBuf CurrentKey;
        TVector<TTagName> TagNames;
        bool EqualMatcher = true;
    };

    struct TValueHashBuilder final : public IRequestKeyVisitor {
        TValueHashBuilder(TInternedTagNameSet tagNameSet)
            : TagNameSet(tagNameSet)
            , EqualMatcher(true)
            , ValueHash(0)
        {
        }

        void OnItype(TStringBuf itype) override {
            ValueHash = ComputeHash(itype);
        }

        void OnKey(TStringBuf key) override {
            CurrentKey = key;
        }

        void OnGroupStart(size_t) override {
            EqualMatcher = false;
        }

        void OnGroupEnd() override {
            EqualMatcher = true;
        }

        void OnEqualValues(TStringBuf value) override {
            if (EqualMatcher && TagNameSet.Has(CurrentKey)) {
                ValueHash = CombineHashes(ValueHash, ComputeHash(value));
            }
        }

        void OnRegexpValues(TStringBuf) override {
        }

        TInternedTagNameSet TagNameSet;
        TStringBuf CurrentKey;
        bool EqualMatcher;
        size_t ValueHash;
    };

    void CheckItype(TStringBuf itype) {
        if (itype.empty()) {
            ythrow yexception() << "no itype specified";
        }
        if (itype.Contains(",")) {
            ythrow yexception() << "multiple itype specified: " << itype;
        }
        if (!IsCorrectRequestTagValue(itype)) {
            ythrow yexception() << "incorrect itype specified: " << itype;
        }
    }

    TParsedRequestKey ParseDynamicString(const TStringBuf& str) {
        TParsedRequestKey result;

        THashSet<TStringBuf> tagNames;
        for (const auto& it : StringSplitter(str).Split(';')) {
            TStringBuf tagName, tagValue;

            Split(StripString(it.Token()), "=", tagName, tagValue);

            if (tagName.empty()) {
                ythrow yexception() << "empty tag name: " << it.Token();
            }
            if (!IsCorrectTagName(tagName)) {
                ythrow yexception() << "incorrect tag name: " << tagName;
            }

            if (tagName == ITYPE_TAG) {
                if (!result.Itype.empty()) {
                    ythrow yexception() << "itype defined twice";
                }
                result.Itype = StripString(tagValue);
                CheckItype(result.Itype);
            } else {
                result.TagsRaw.emplace_back(tagName, tagValue);
                if (tagNames.contains(tagName)) {
                    ythrow yexception() << "tags names are not unique: " << tagName;
                }
                tagNames.insert(tagName);
            }
        }
        if (result.Itype.empty()) {
            ythrow yexception() << "missing itype " << str;
        }

        return result;
    }

    TParsedRequestKey ParseUnderscoredString(const TStringBuf& rawStr) {
        TParsedRequestKey result;

        TVector<TStringBuf> tagsValues = StringSplitter(rawStr).Split('_').ToList<TStringBuf>();
        switch (tagsValues.size()) {
            case 2: {
                if (tagsValues[1] != SELF) {
                    ythrow yexception() << "bad request key given " << rawStr;
                }
                break;
            }
            case 5: {
                result.TagsRaw.emplace_back(CTYPE_TAG, tagsValues[1]);
                result.TagsRaw.emplace_back(PRJ_TAG, tagsValues[2]);
                result.TagsRaw.emplace_back(GEO_TAG, tagsValues[3]);
                result.TagsRaw.emplace_back(TIER_TAG, tagsValues[4]);
                break;
            }
            default: {
                ythrow yexception() << "bad request key given " << rawStr;
            }
        }
        CheckItype(TString{tagsValues[0]});
        result.Itype = tagsValues[0];

        return result;
    }

    TParsedRequestKey ParseRequestKey(const TStringBuf& str) {
        if (str.Contains("=")) {
            return ParseDynamicString(str);
        }
        return ParseUnderscoredString(str);
    }

    void FillNormalizedTags(const TSmallVec<std::pair<TStringBuf, TStringBuf>>& tagsRaw,
                            TSmallVec<TString>& tagsNames, TRequestTags& tags) {

        for (const auto& tag : tagsRaw) {
            TVector<TStringBuf> tagValues = StringSplitter(tag.second).Split(',').ToList<TStringBuf>();
            auto matcher = CreateMatcher(tag.first, tagValues);
            if (matcher) {
                tagsNames.emplace_back(tag.first);
                tags.emplace_back(tag.first, std::move(matcher));
            }
        }
        Sort(tags, [](const TRequestTagPair& x, const TRequestTagPair& y) { return x.first < y.first; });
        SortUnique(tagsNames);
    }

    TString MakeNormalizedString(const TStringBuf& itype, const TRequestTags& tags) {
        TSmallVec<TString> tagPairs;
        tagPairs.emplace_back(Join("=", ITYPE_TAG, itype));
        for (const auto& tag : tags) {
            tagPairs.emplace_back(Join("=", tag.first, tag.second->DynamicString()));
        }
        return JoinSeq(";", tagPairs);
    }

    using TGroupsIntegrityValue = THashSet<std::pair<TStringBuf, TStringBuf>>;
    using TGroupsIntegrity = THashMap<TAggregated, TGroupsIntegrityValue>;
    using TGroup = TVector<TInstanceKey>;
    using TGroups = THashMap<TAggregated, TGroup>;
    using TResolveValue = TVector<TGroup>;
    using TGroupedKey = std::pair<size_t, ssize_t>;
}

void IRequestKeyVisitor::OnItype(TStringBuf) {
}

void IRequestKeyVisitor::OnTagCount(size_t) {
}

void IRequestKeyVisitor::OnKey(TStringBuf) {
}

void IRequestKeyVisitor::OnGroupStart(size_t) {
}

void IRequestKeyVisitor::OnGroupEnd() {
}

TRequestKey TRequestKey::FromString(const TStringBuf& rawStr) {
    return CreateRequestKey(rawStr, ParseRequestKey(rawStr));
}

TRequestKey TRequestKey::NormalizedFromString(const TStringBuf& rawStr) {
    return CreateNormalizedRequestKey(ParseRequestKey(rawStr));
}

TRequestKey::TRequestKey(const TStringBuf& raw, const TStringBuf& itype,
                         TRequestTags&& tags, const TSmallVec<TString>& tagsNames)
    : Raw(raw)
    , Itype(itype)
    , Tags(std::move(tags))
    , TagsNames(tagsNames)
{
    TTagNameSetBuilder builder;
    Visit(builder);
    TagNameSet = TInternedTagNameSet(builder.TagNames);
}

TRequestKey TRequestKey::CreateNormalizedRequestKey(const TParsedRequestKey& parsedRequestKey) {
    TSmallVec<TString> tagsNames;
    TRequestTags tags;
    FillNormalizedTags(parsedRequestKey.TagsRaw, tagsNames, tags);

    TString normalizedName = MakeNormalizedString(parsedRequestKey.Itype, tags);
    return TRequestKey(normalizedName, parsedRequestKey.Itype, std::move(tags), tagsNames);
}

TRequestKey TRequestKey::CreateRequestKey(const TStringBuf& raw, const TParsedRequestKey& parsedRequestKey) {
    TSmallVec<TString> tagsNames;
    TRequestTags tags;
    FillNormalizedTags(parsedRequestKey.TagsRaw, tagsNames, tags);

    return TRequestKey(raw, parsedRequestKey.Itype, std::move(tags), tagsNames);
}

bool TRequestKey::Match(TInstanceKey instanceKey, bool exactMatch) const {
    if (Itype != instanceKey.GetItype()) {
        return false;
    }

    if (exactMatch && instanceKey.GetTags().size() != Tags.size()) {
       return false;
    }

    if (!instanceKey.GetTagNameSet().IsSuperSet(TagNameSet)) {
        return false;
    }

    auto instanceTags = instanceKey.GetTags();
    auto instanceIter = instanceTags.begin();
    for (const auto& requestTag : Tags) {
        bool matchTag = false;

        while (instanceIter != instanceTags.end() && instanceIter->Name < requestTag.first) {
            ++instanceIter;
        }

        if (exactMatch &&
            (instanceIter == instanceTags.end() ||
                instanceIter->Name != requestTag.first ||
                instanceIter->Value != requestTag.second->DynamicString())) {
            return false;
        }

        while (instanceIter != instanceTags.end() && instanceIter->Name == requestTag.first) {
            if (!matchTag && requestTag.second->Match(instanceIter->Value)) {
                matchTag = true;
            }
            ++instanceIter;
        }

        if (!matchTag) {
            return false;
        }
    }
    return true;
}

TString TRequestKey::GetDynamicString() const {
    return MakeNormalizedString(Itype, Tags);
}

TMap<TStringBuf, TSmallVec<TStringBuf>> TRequestKey::GetDict() const {
    TDictBuilder builder;
    builder.Result[NPrivate::ITYPE_TAG].emplace_back(Itype);
    Visit(builder);
    return std::move(builder.Result);
}

bool TRequestKey::HasOnlyEqualMatcher() const {
    TEqualBuilder builder;
    Visit(builder);
    return builder.HasOnlyEqualMatcher;
}

TVector<TStringBuf> TRequestKey::EqualTags() const {
    TEqualBuilder builder;
    Visit(builder);
    return std::move(builder.EqualTags);
}

TVector<TStringBuf> TRequestKey::EqualValues() const {
    TEqualBuilder builder;
    Visit(builder);
    return std::move(builder.EqualValues);
}

size_t TRequestKey::GetValueHash() const noexcept {
    TValueHashBuilder builder(GetTagNameSet());
    Visit(builder);
    return builder.ValueHash;
}

void TRequestKey::Visit(IRequestKeyVisitor& visitor) const {
    visitor.OnItype(GetItype());
    visitor.OnTagCount(Tags.size());
    for (const auto& [tagName, matcher] : Tags) {
        visitor.OnKey(tagName);
        matcher->Visit(visitor);
    }
}

bool TRequestKey::operator==(const TRequestKey& other) const noexcept {
    if (Itype != other.Itype || Tags.size() != other.Tags.size()) {
        return false;
    }
    for (size_t i : xrange(Tags.size())) {
        const auto& tag1 = Tags[i];
        const auto& tag2 = other.Tags[i];
        if (tag1.first != tag2.first || *tag1.second.Get() != *tag2.second.Get()) {
            return false;
        }
    }
    return true;
}

template <>
void Out<NTags::TRequestKey>(IOutputStream& stream,
                             TTypeTraits<NTags::TRequestKey>::TFuncParam key) {
    stream.Write(key.ToNamed());
}
