#pragma once

#include <infra/yasm/common/labels/tags/instance_key.h>

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>

namespace NTags {
    struct IRequestKeyVisitor;

    namespace NPrivate {
        class TMatcher {
        public:
            TMatcher(TStringBuf raw)
                : Raw(raw)
            {
            }

            virtual void Visit(IRequestKeyVisitor&) const = 0;
            virtual bool Match(TStringBuf) const = 0;
            virtual ~TMatcher() = default;

            const TString& DynamicString() const noexcept {
                return Raw;
            };

            bool operator==(const TMatcher& other) const noexcept {
                return Raw == other.Raw;
            }

            bool operator!=(const TMatcher& other) const noexcept {
                return Raw != other.Raw;
            }

        protected:
            const TString Raw;
        };

        using TMatcherPtr = THolder<TMatcher>;

        struct TParsedRequestKey {
            TStringBuf Itype;
            TSmallVec<std::pair<TStringBuf, TStringBuf>> TagsRaw;
        };
    }

    using TRequestTagPair = std::pair<TString, NPrivate::TMatcherPtr>;
    using TRequestTags = TVector<TRequestTagPair>;

    struct IRequestKeyVisitor {
        virtual ~IRequestKeyVisitor() = default;

        virtual void OnItype(TStringBuf itype);
        virtual void OnTagCount(size_t count);
        virtual void OnKey(TStringBuf key);
        virtual void OnGroupStart(size_t count);
        virtual void OnGroupEnd();
        virtual void OnEqualValues(TStringBuf value) = 0;
        virtual void OnRegexpValues(TStringBuf value) = 0;
    };

    class TRequestKey {
    public:
        TRequestKey(TRequestKey&& other) = default;

        // Create request key from it's raw string representation
        static TRequestKey FromString(const TStringBuf& rawStr);

        // Create normalized request key from it's raw string representation.
        // Normalized key returns normalized (dynamic) string in ToNamed().
        static TRequestKey NormalizedFromString(const TStringBuf& rawStr);

        bool Match(TInstanceKey instanceKey, bool exactMatch = false) const;

        const TString& ToNamed() const noexcept {
            return Raw;
        }
        const TString& GetItype() const noexcept {
            return Itype;
        }
        TInternedTagNameSet GetTagNameSet() const noexcept {
            return TagNameSet;
        }

        TString GetDynamicString() const;

        TMap<TStringBuf, TSmallVec<TStringBuf>> GetDict() const;

        bool HasOnlyEqualMatcher() const;
        TVector<TStringBuf> EqualTags() const;
        TVector<TStringBuf> EqualValues() const;

        size_t GetValueHash() const noexcept;

        void Visit(IRequestKeyVisitor& visitor) const;

        bool operator==(const TRequestKey& other) const noexcept;

    private:
        static TRequestKey CreateRequestKey(const TStringBuf& raw, const NPrivate::TParsedRequestKey& parsedRequestKey);
        static TRequestKey CreateNormalizedRequestKey(const NPrivate::TParsedRequestKey& parsedRequestKey);

        TRequestKey(const TStringBuf& raw, const TStringBuf& itype,
                    TRequestTags&& tags, const TSmallVec<TString>& tagsNames);

        const TString Raw;
        const TString Itype;
        TRequestTags Tags;
        TSmallVec<TString> TagsNames;
        TInternedTagNameSet TagNameSet;
    };
}

template<>
struct THash<NTags::TRequestKey> {
    inline size_t operator()(const NTags::TRequestKey& v) const noexcept {
        return ComputeHash(v.ToNamed());
    }
};
