#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/tags/request_key.h>

#include <library/cpp/testing/unittest/registar.h>

bool IsRequestValid(TStringBuf request) {
    try {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
    } catch (const yexception& e) {
        return false;
    }
    return true;
}

bool IsRequestInvalid(TStringBuf request) {
    return !IsRequestValid(request);
}

class TRequestTagsTest: public TTestBase {
    UNIT_TEST_SUITE(TRequestTagsTest);
    UNIT_TEST(RequestKeyTests)
    UNIT_TEST(EqualTagsTests)
    UNIT_TEST(ValueHashTests)
    UNIT_TEST_SUITE_END();

private:
    void CheckRequestKey(TString undescored, TString named,
                         const TVector<std::pair<TStringBuf, TStringBuf>>& histdb,
                         const TSmallVec<TString>& validRequests, const TSmallVec<TString>& invalidRequests) {
        NTags::TInstanceKey instanceKeyUndescored = NTags::TInstanceKey::FromUnderscore(undescored);
        NTags::TInstanceKey instanceKeyNamed = NTags::TInstanceKey::FromNamed(named);
        NTags::TInstanceKey instanceKeyHistDB = NTags::TInstanceKey::FromHistDb(histdb);

        UNIT_ASSERT_EQUAL(named, instanceKeyNamed.ToNamed());
        UNIT_ASSERT_EQUAL(named, instanceKeyUndescored.ToNamed());
        UNIT_ASSERT_EQUAL(named, instanceKeyHistDB.ToNamed());

        UNIT_ASSERT_EQUAL(instanceKeyNamed, instanceKeyUndescored);
        UNIT_ASSERT_EQUAL(instanceKeyNamed, instanceKeyHistDB);
        UNIT_ASSERT_EQUAL(instanceKeyUndescored, instanceKeyHistDB);

        for (const TString& request : validRequests) {
            NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
            UNIT_ASSERT(requestKey.Match(instanceKeyNamed));
            UNIT_ASSERT(requestKey.Match(instanceKeyUndescored));
            UNIT_ASSERT(requestKey.Match(instanceKeyHistDB));
        }
        for (const TString& request : invalidRequests) {
            NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
            UNIT_ASSERT(!requestKey.Match(instanceKeyNamed));
            UNIT_ASSERT(!requestKey.Match(instanceKeyUndescored));
            UNIT_ASSERT(!requestKey.Match(instanceKeyHistDB));
        }
    }

    void RequestKeyTests() {
        CheckRequestKey("mmeta_prod_web-main_ams_yandsearch",
                        "mmeta|ctype=prod;geo=ams;prj=web-main;tier=yandsearch",
                        {
                            {"itype", "mmeta"},
                            {"ctype", "prod"},
                            {"geo", "ams"},
                            {"prj", "web-main"},
                            {"tier", "yandsearch"},
                        },
                        {
                            //valid cases
                            "mmeta_prod_web-main_ams_yandsearch",
                            "mmeta_prod_web-main_ams_self",
                            "mmeta_self",
                            "itype=mmeta",
                            "itype=mmeta;ctype=prod",
                            "itype=mmeta;ctype=*",
                            "itype=mmeta;ctype=prod;prj=*",
                        },
                        {
                            //invalid cases
                            "mmeta_prod_web-main_ams_MSUserData0",
                            "mmeta_prod_web-main_vla_yandsearch",
                            "upper_prod_web-main_ams_yandsearch",
                            "upper_self",
                            "itype=upper",
                            "itype=upper;ctype=prod",
                            "itype=mmeta;ctype=prestable",

                        });

        CheckRequestKey("mmeta_self",
                        "mmeta||ctype,geo,prj,tier",
                        {{"itype", "mmeta"},
                         {"tier", "self"}},
                        {
                            //valid cases
                            "itype=mmeta",
                            "itype=mmeta;tier=*",
                            "itype=mmeta;ctype=*;tier=*",
                            "itype=mmeta;ctype=*;geo=*;tier=*",
                            "mmeta_self",
                        },
                        {
                            //invalid cases
                            "itype=mmeta;ctype=prod",
                            "itype=mmeta;ctype=prod;prj=*",
                            "mmeta_prod_web-main_ams_self",
                            "itype=upper",
                            "itype=upper;ctype=prod",
                            "itype=mmeta;ctype=prestable",
                        });

        // regexp test
        CheckRequestKey("mmeta_prod_web-main_ams_yandsearch",
                        "mmeta|ctype=prod;geo=ams;prj=web-main;tier=yandsearch",
                        {{"itype", "mmeta"},
                         {"ctype", "prod"},
                         {"geo", "ams"},
                         {"prj", "web-main"},
                         {"tier", "yandsearch"}},
                        {
                            //valid case
                            "itype=mmeta;ctype=prod;tier=*dsearch",
                            "itype=mmeta;ctype=prod;tier=yands*",
                            "itype=mmeta;ctype=prod;tier=*ds*",
                            "itype=mmeta;ctype=prod;prj=*-main",
                        },
                        {
                            //invalid cases
                            "itype=mmeta;ctype=prod;prj=*-mainsss",

                        });
    }

    void EqualTagsTests() {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString("itype=apphost;geo=msk,sas;ctype=prestable");
        UNIT_ASSERT(!requestKey.HasOnlyEqualMatcher());
        UNIT_ASSERT_VALUES_EQUAL(requestKey.EqualTags(), TVector<TStringBuf>{"ctype"});
        UNIT_ASSERT_VALUES_EQUAL(requestKey.EqualValues(), TVector<TStringBuf>{"prestable"});
    }

    void ValueHashTests() {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString("itype=apphost;geo=msk,sas;ctype=prestable;tier=none");
        NTags::TInstanceKey instanceKey = NTags::TInstanceKey::FromNamed("apphost|geo=msk;ctype=prestable;tier=none;prj=something");
        UNIT_ASSERT_VALUES_EQUAL(requestKey.GetValueHash(), instanceKey.GetValueHash(requestKey.GetTagNameSet()));
    }
};

class TRequestDynamicTest: public TTestBase {
    UNIT_TEST_SUITE(TRequestDynamicTest);
    UNIT_TEST(ReqestKeyFromDynamicTests)
    UNIT_TEST_SUITE_END();

private:
    void CheckDynamicRequest(TString request, const TSmallVec<TString>& validInstances,
                             const TSmallVec<TString>& invalidInstances) {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
        for (const TString& instance : validInstances) {
            NTags::TInstanceKey instanceKey = NTags::TInstanceKey::FromNamed(instance);
            UNIT_ASSERT(requestKey.Match(instanceKey));
        }
        for (const TString& instance : invalidInstances) {
            NTags::TInstanceKey instanceKey = NTags::TInstanceKey::FromNamed(instance);
            UNIT_ASSERT(!requestKey.Match(instanceKey));
        }
    }

    void ReqestKeyFromDynamicTests() {
        CheckDynamicRequest("itype=mmeta",
                            {
                                //valid cases
                                "mmeta",
                                "mmeta||tier",
                                "mmeta|ctype=prod;geo=ams;prj=web-main;tier=yandsearch",
                            },
                            {
                                //invalid cases
                                "upper",
                                "upper|ctype=prod;geo=ams;prj=web-main;tier=yandsearch",

                            });

        CheckDynamicRequest("itype=mmeta;ctype=p*;geo=msk",
                            {
                                //valid cases
                                "mmeta|ctype=prod;geo=msk",
                                "mmeta|ctype=prestable;geo=msk",
                                "mmeta|ctype=prestable,prod;geo=msk",
                                "mmeta|ctype=prestable,prod,testing;geo=msk",
                                "mmeta|ctype=prod,testing;geo=msk",
                                "mmeta|ctype=prestable,testing;geo=msk",
                            },
                            {
                                //invalid cases
                                "mmeta|ctype=testing;geo=msk",
                                "mmeta|ctype=prod",
                                "mmeta|ctype=prestable",
                                "mmeta|ctype=prestable,prod",
                                "mmeta|ctype=prestable,prod,testing",
                                "mmeta|ctype=prod;geo=sas",
                                "mmeta|ctype=prestable;geo=sas",
                                "mmeta|ctype=prestable,prod;geo=sas",
                                "mmeta|ctype=prestable,prod,testing;geo=sas",
                            });

        CheckDynamicRequest("itype=mmeta;ctype=prod,prestable;geo=msk",
                            {
                                //valid cases
                                "mmeta|ctype=prod;geo=msk",
                                "mmeta|ctype=prestable;geo=msk",
                                "mmeta|ctype=prestable,prod;geo=msk",
                                "mmeta|ctype=prestable,prod,testing;geo=msk",
                                "mmeta|ctype=prod,testing;geo=msk",
                                "mmeta|ctype=prestable,testing;geo=msk",
                            },
                            {
                                //invalid cases
                                "mmeta|ctype=testing;geo=msk",
                                "mmeta|ctype=prod",
                                "mmeta|ctype=prestable",
                                "mmeta|ctype=prestable,prod",
                                "mmeta|ctype=prestable,prod,testing",
                                "mmeta|ctype=prod;geo=sas",
                                "mmeta|ctype=prestable;geo=sas",
                                "mmeta|ctype=prestable,prod;geo=sas",
                                "mmeta|ctype=prestable,prod,testing;geo=sas",
                            });

        // Regex matcher and aggregated instances
        CheckDynamicRequest("itype=qloud;prj=cocaine.*",
                            {
                                //valid cases
                                "qloud|ctype=ctype;geo=geo;prj=cocaine.foo;tier=tier",
                                "qloud|ctype=ctype;geo=geo;prj=cocaine.foo|tier",
                            },
                            {
                                // invalid cases
                                "qloud||ctype,geo,prj,tier",
                            });
    }
};

class TUnificationTest: public TTestBase {
    UNIT_TEST_SUITE(TUnificationTest);
    UNIT_TEST(TestUnificationCases)
    UNIT_TEST_SUITE_END();

private:
    void CheckIsAllRequestEqual(const TSmallVec<TString>& requests) {
        NTags::TRequestKey unificationKey = NTags::TRequestKey::FromString(requests[0]);
        for (const TString& request : requests) {
            NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
            UNIT_ASSERT_EQUAL(requestKey.GetDynamicString(), unificationKey.GetDynamicString());
            UNIT_ASSERT_EQUAL(requestKey, unificationKey);
        }
    }

    void TestUnificationCases() {
        CheckIsAllRequestEqual({
            "mmeta_self",
            "itype=mmeta",
            "itype=mmeta;tier=*",
            "itype=mmeta;tier=self",
            "itype=mmeta;ctype=*",
            "itype=mmeta;ctype=*;geo=man,*",
        });

        CheckIsAllRequestEqual({
            "mmeta_prod_web_msk_self",
            "itype=mmeta;ctype=prod;prj=web;geo=msk",
            "itype=mmeta;geo=msk;prj=web;ctype=prod",
            "itype=mmeta;geo=msk;prj=web;ctype=prod;tier=self",
            "itype=mmeta;geo=msk;prj=web;ctype=prod;tier=*",
            "itype=mmeta;geo=msk;prj=web;ctype=prod;tier=*,yandsearch",
            " itype= mmeta ; geo= msk ; prj= web ; ctype= prod ;tier=*,yandsearch",
        });

        CheckIsAllRequestEqual({"itype=mmeta;geo=msk,man",
                                "itype=mmeta;geo=man,msk"});

        CheckIsAllRequestEqual({
            "mmeta_prod_web_msk_yandsearch",
            "itype=mmeta;ctype=prod;prj=web;geo=msk;tier=yandsearch",
            "itype=mmeta;tier=yandsearch;geo=msk;prj=web;ctype=prod",
        });

        CheckIsAllRequestEqual({
            "itype=mmeta;geo=msk;prj=web",
            "prj=web;geo=msk;itype=mmeta",              // переставленные ключи
            "itype=mmeta;geo=msk;ctype=*;prj=web",      // лишний wildcard
            "itype=mmeta;geo=msk;ctype=prod,*;prj=web", // лишний wildcard
        });

        CheckIsAllRequestEqual({"itype=mmeta;ctype=prestable,testing;geo=msk;prj=web",
                                "prj=web;ctype=testing,prestable;itype=mmeta;geo=msk"});

        CheckIsAllRequestEqual({"itype=mmeta;ctype=prod;geo=m*;prj=web",
                                "prj=web;ctype=prod;geo=m*;itype=mmeta"});

        CheckIsAllRequestEqual({
            "itype=mmeta;collection=images;ctype=prod;geo=msk;prj=web",
            "itype=mmeta;ctype=prod;prj=web;geo=msk;collection=images",
            "itype=mmeta;ctype=prod;prj=web;geo=msk;collection=images;tier=*",
            "itype=mmeta;ctype=prod;prj=web;geo=msk;collection=images;tier=self",
            "itype=mmeta;ctype=prod;prj=web;geo=msk;collection=images;tier=self,yandsearch",
            "itype=mmeta;ctype=prod;prj=web;geo=msk;collection=images;submodule=tass,*", // А здесь даже шесть тэгов
        });
    }
};

class TRequestToDictTest: public TTestBase {
    UNIT_TEST_SUITE(TRequestToDictTest);
    UNIT_TEST(RequestKeyDictTests)
    UNIT_TEST_SUITE_END();

private:
    using TDict = TMap<TStringBuf, TSmallVec<TStringBuf>>;

    void CheckRequestDict(const TString request, const TDict& dict) {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
        const TDict& requestDict = requestKey.GetDict();
        UNIT_ASSERT_EQUAL(requestDict, dict);
    }

    void RequestKeyDictTests() {
        CheckRequestDict("itype=mmeta", {{"itype", {"mmeta"}}});

        CheckRequestDict("mmeta_self", {{"itype", {"mmeta"}}});

        CheckRequestDict("mmeta_prod_web-main_ams_yandsearch",
                         {{"itype", {"mmeta"}},
                          {"ctype", {"prod"}},
                          {"geo", {"ams"}},
                          {"prj", {"web-main"}},
                          {"tier", {"yandsearch"}}});

        CheckRequestDict("mmeta_prod_web-main_ams_self",
                         {
                             {"itype", {"mmeta"}},
                             {"ctype", {"prod"}},
                             {"geo", {"ams"}},
                             {"prj", {"web-main"}},
                         });

        CheckRequestDict("itype=mmeta;ctype=prestable,prod;prj=web-main",
                         {
                             {"itype", {"mmeta"}},
                             {"ctype", {"prestable", "prod"}},
                             {"prj", {"web-main"}},
                         });

        CheckRequestDict("itype=mmeta;ctype=p*,t*;geo=*m*;tier=*",
                         {
                             {"itype", {"mmeta"}},
                             {"ctype", {"p*", "t*"}},
                             {"geo", {"*m*"}},
                         });
    }
};

class TValidUnderscoredRequestTest: public TTestBase {
    UNIT_TEST_SUITE(TValidUnderscoredRequestTest);
    UNIT_TEST(ValidUnderscoretTests)
    UNIT_TEST_SUITE_END();

    void CheckUnderscoredRequest(TStringBuf request) {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
        NTags::TInstanceKey instanceKey = NTags::TInstanceKey::FromUnderscore(request);
        UNIT_ASSERT(requestKey.Match(instanceKey));
    }

private:
    void ValidUnderscoretTests() {
        CheckUnderscoredRequest("mmeta_prod_web-main_ams_yandsearch");
        CheckUnderscoredRequest("mmeta_self");
    }
};

class TValidDynamicRequestTest: public TTestBase {
    UNIT_TEST_SUITE(TValidDynamicRequestTest);
    UNIT_TEST(CheckRequestKeyFromkDynamicRequest)
    UNIT_TEST_SUITE_END();

    void CheckDynamicString(TStringBuf request) {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
        NTags::TRequestKey requestKeyFromDynamicString = NTags::TRequestKey::FromString(requestKey.GetDynamicString());

        UNIT_ASSERT_EQUAL(requestKey.GetDynamicString(), requestKeyFromDynamicString.GetDynamicString());
        UNIT_ASSERT_EQUAL(requestKey, requestKeyFromDynamicString);
    }

private:
    void CheckRequestKeyFromkDynamicRequest() {
        CheckDynamicString("itype=mmeta");
        CheckDynamicString("itype=mmeta;ctype=prod;geo=ams;prj=web-main;tier=yandsearch");
        CheckDynamicString("itype=mmeta;ctype=prestable,prod;geo=ams,msk;prj=web-main;tier=yandsearch");
        CheckDynamicString("itype=mmeta;ctype=prestable,prod;tier=yandsearch");
        CheckDynamicString("itype=mmeta;ctype=prestable,prod");
        CheckDynamicString("itype=mmeta;geo=msk,ams;prj=web-main;tier=yandsearch");
        CheckDynamicString("itype=mmeta;ctype=prod,prestable;geo=ams,msk;prj=web-main;tier=yandsearch");
        CheckDynamicString("itype=mmeta;ctype=p*;geo=ams,msk;prj=web-main;tier=yandsearch");
        CheckDynamicString("itype=mmeta;ctype=p*;geo=*m*;prj=web-main");
        CheckDynamicString("itype=mmeta;ctype=*;geo=*m*;prj=web-main");
        CheckDynamicString("itype=mmeta;ctype=prestable, prod");
        CheckDynamicString("itype=mmeta; ctype=prestable,prod");
        CheckDynamicString("itype=mmeta; ctype=prestable, prod");
    }
};

class TValidAndSortedDynamicRequestTest: public TTestBase {
    UNIT_TEST_SUITE(TValidAndSortedDynamicRequestTest);
    UNIT_TEST(TestSortedDynamicRequests)
    UNIT_TEST_SUITE_END();

    void CheckSortDynamicRequest(TStringBuf request) {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString(request);
        UNIT_ASSERT_EQUAL(requestKey.GetDynamicString(), request);
    }

private:
    void TestSortedDynamicRequests() {
        CheckSortDynamicRequest("itype=mmeta");
        CheckSortDynamicRequest("itype=mmeta;geo=sas");
        CheckSortDynamicRequest("itype=mmeta;geo=ams,msk;prj=web-main;tier=yandsearch");
        CheckSortDynamicRequest("itype=mmeta;geo=ams,msk,sas;prj=web-main;tier=yandsearch");
    }
};

class TInvalidUnderscoredRequestTest: public TTestBase {
    UNIT_TEST_SUITE(TInvalidUnderscoredRequestTest);
    UNIT_TEST(TestInvalidUnderscoredCases)
    UNIT_TEST_SUITE_END();


private:
    void TestInvalidUnderscoredCases() {
        UNIT_ASSERT(IsRequestInvalid(""));
        UNIT_ASSERT(IsRequestInvalid("mmeta|"));
        UNIT_ASSERT(IsRequestInvalid("mmeta;"));
        UNIT_ASSERT(IsRequestInvalid("ммета"));
        UNIT_ASSERT(IsRequestInvalid("_self"));
        UNIT_ASSERT(IsRequestInvalid("mmeta____yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("mmeta_custom_self"));
        UNIT_ASSERT(IsRequestValid("mm-eta_self"));  //GOLOVAN-6414
        UNIT_ASSERT(IsRequestInvalid("mmeta_se|lf"));
        UNIT_ASSERT(IsRequestInvalid("mmeta_prod_web_main_asm_yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("mmeta_prod_self"));
        UNIT_ASSERT(IsRequestInvalid("mmeta_yandsearch"));
    }
};

class TInvalidDynamicRequestTest: public TTestBase {
    UNIT_TEST_SUITE(TInvalidDynamicRequestTest);
    UNIT_TEST(TestInvalidRequestCases)
    UNIT_TEST_SUITE_END();

private:
    void TestInvalidRequestCases() {
        UNIT_ASSERT(IsRequestInvalid(""));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta,upper"));
        UNIT_ASSERT(IsRequestValid("itype=mm-eta;ctype=prod;geo=ams;prj=web-main;tier=yandsearch"));  //GOLOVAN-6414
        UNIT_ASSERT(IsRequestInvalid("ctype=prod;geo=am_s;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("ctype=prod;geo=ams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta,upper;ctype=prod;geo=ams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=;ctype=prod;geo=ams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=ctype=prod;geo=ams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;ctype=;geo=ams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;=prod;geo=ams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;prod;geoams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;prod;geo ams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=m*;prod;geo ams;prj=web-main;tier=yandsearch"));
        UNIT_ASSERT(IsRequestInvalid("itype=m*;geo=ams =sas"));
        UNIT_ASSERT(IsRequestInvalid("itype=m*;geo=ams;geo=sas"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;itype=geo"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta=prod"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;geo=ams;geo=ams"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;geo=ams;geo=man"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;geo=ams;geo=*"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;itype=common;geo=ams"));
        UNIT_ASSERT(IsRequestInvalid("itype=mmeta;itype=mmeta;geo=ams"));
    }
};

class TExactMatchTest: public TTestBase {
    UNIT_TEST_SUITE(TExactMatchTest);
    UNIT_TEST(TestExactMatch)
    UNIT_TEST_SUITE_END();

private:
    void TestExactMatch() {
        NTags::TRequestKey requestKey = NTags::TRequestKey::FromString("itype=mmeta;geo=ams;prj=web-main;tier=yandsearch");
        NTags::TRequestKey requestKeyWithTemplate = NTags::TRequestKey::FromString("itype=mmeta;geo=ams;prj=web-main;tier=yandsearch*");
        NTags::TInstanceKey exactEqual = NTags::TInstanceKey::FromNamed("mmeta|geo=ams;prj=web-main;tier=yandsearch");
        NTags::TInstanceKey moreTags = NTags::TInstanceKey::FromNamed("mmeta|ctype=prod;geo=ams;prj=web-main;tier=yandsearch");

        UNIT_ASSERT(requestKey.Match(exactEqual, false));
        UNIT_ASSERT(requestKey.Match(exactEqual, true));

        UNIT_ASSERT(requestKey.Match(moreTags, false));
        UNIT_ASSERT(!requestKey.Match(moreTags, true));

        UNIT_ASSERT(requestKeyWithTemplate.Match(exactEqual, false));
        UNIT_ASSERT(!requestKeyWithTemplate.Match(exactEqual, true));
    }
};

UNIT_TEST_SUITE_REGISTRATION(TRequestTagsTest);
UNIT_TEST_SUITE_REGISTRATION(TRequestDynamicTest);
UNIT_TEST_SUITE_REGISTRATION(TUnificationTest);
UNIT_TEST_SUITE_REGISTRATION(TRequestToDictTest);
UNIT_TEST_SUITE_REGISTRATION(TValidUnderscoredRequestTest);
UNIT_TEST_SUITE_REGISTRATION(TValidDynamicRequestTest);
UNIT_TEST_SUITE_REGISTRATION(TValidAndSortedDynamicRequestTest);
UNIT_TEST_SUITE_REGISTRATION(TInvalidUnderscoredRequestTest);
UNIT_TEST_SUITE_REGISTRATION(TInvalidDynamicRequestTest);
UNIT_TEST_SUITE_REGISTRATION(TExactMatchTest);
