#include "accumulators.h"

#include <infra/yasm/common/points/accumulators/impl/average.h>
#include <infra/yasm/common/points/accumulators/impl/average.h>
#include <infra/yasm/common/points/accumulators/impl/hgram.h>
#include <infra/yasm/common/points/accumulators/impl/last.h>
#include <infra/yasm/common/points/accumulators/impl/max.h>
#include <infra/yasm/common/points/accumulators/impl/min.h>
#include <infra/yasm/common/points/accumulators/impl/oldavg.h>
#include <infra/yasm/common/points/accumulators/impl/summ.h>
#include <infra/yasm/common/points/accumulators/impl/summnone.h>
#include <infra/yasm/common/points/value/abstract/value.h>

#include <optional>

using namespace NZoom::NAccumulators;

namespace {

    /**
     * @brief Value type wrapper default-initialized as NoneValue()
     * 
     * TCompactAccumulatorsArray stores a _fixed_ number of points. Hence, every point is default-initialized
     * on an array's creation. Current point implementations default-initialize to a 0, thus there's no way to 
     * represent a null value. This type initializes a point as a NoneValue() and proxies every acces to a real
     * type if it was given
     * 
     * @tparam T a real type this wrapper proxies access to
     */
    template <typename T>
    class TOptionalValue: public TAbstractAccumulator, public NZoom::NValue::IValue {
    public:
        TOptionalValue()
            : Value_{}
        {}

        void Clean() override {
            if (Value_.has_value()) {
                Value_->Clean();
            }
        }

        const NZoom::NValue::IValue& GetValue() const override {
            if (Value_.has_value()) {
                return *Value_;
            } else {
                return NZoom::NValue::TNoneValue::GetSingleton();
            }
        }

        void MulNone() override {
        }

        void MulFloat(const double value) override {
            if (!Value_.has_value()) {
                Value_.emplace();
            }

            Value_->MulFloat(value);
        }

        void MulVec(const TVector<double>& value) override {
            if (!Value_.has_value()) {
                Value_.emplace();
            }

            Value_->MulVec(value);
        }

        void MulCountedSum(const double sum, const ui64 count) override {
            if (!Value_.has_value()) {
                Value_.emplace();
            }

            Value_->MulCountedSum(sum, count);
        }

        void MulHgram(const NZoom::NHgram::THgram& value) override {
            if (!Value_.has_value()) {
                Value_.emplace();
            }

            Value_->MulHgram(value);
        }

        NZoom::NValue::EValueType GetType() const noexcept override {
            if (Value_.has_value()) {
                return Value_->GetType();
            } else {
                return NZoom::NValue::TNoneValue::GetSingleton().GetType();
            }
        }

        bool IsDefault() const noexcept override {
            return Value_.has_value() ? Value_->IsDefault() : true;
        }

        void Update(IUpdatable& accumulator) const override {
            if (Value_.has_value()) {
                Value_->Update(accumulator);
            } else {
                accumulator.MulNone();
            }
        }

    private:
        std::optional<T> Value_;
    };

    template <ENullPolicy>
    struct TIsNullable: public std::false_type {};
    template <>
    struct TIsNullable<ENullPolicy::Nullable>: public std::true_type {};

    template <ENullPolicy policy, typename T>
    using TMaybeNullable = std::conditional_t<TIsNullable<policy>::value, TOptionalValue<T>, T>;

    /// Create new empty accumulator with appropriate type.

    THolder<TAbstractAccumulator> CreateAccumulator(const EAccumulatorType type, bool allowLegacyTypes) {
        switch(type) {
        case EAccumulatorType::Average:
            return MakeHolder<TAverage>();
        case EAccumulatorType::Counter:
            ythrow yexception() << "Not implemented";
        case EAccumulatorType::Hgram:
            return MakeHolder<THgram>(allowLegacyTypes);
        case EAccumulatorType::Last:
            return MakeHolder<TLast>();
        case EAccumulatorType::List:
        case EAccumulatorType::Avg:
            return MakeHolder<TOldAvg>();
        case EAccumulatorType::Max:
            return MakeHolder<TMax>();
        case EAccumulatorType::Min:
            return MakeHolder<TMin>();
        case EAccumulatorType::Summ:
            return MakeHolder<TSumm>();
        case EAccumulatorType::SummNone:
            return MakeHolder<TSummNone>();
        }
    }

    template <ENullPolicy nullPolicy = ENullPolicy::NonNullable>
    THolder<IAbstractAccumulatorArray> CreateAccumulatorArray(const EAccumulatorType type, size_t size) {
        switch (type) {
        case EAccumulatorType::Average:
            return MakeHolder<TTypedAccumulatorArray<TMaybeNullable<nullPolicy, TAverage>>>(size);
        case EAccumulatorType::Counter:
          ythrow yexception() << "Not implemented";
        case EAccumulatorType::Hgram:
            return MakeHolder<TTypedAccumulatorArray<TMaybeNullable<nullPolicy, THgram>>>(size);
        case EAccumulatorType::Last:
            return MakeHolder<TTypedAccumulatorArray<TMaybeNullable<nullPolicy, TLast>>>(size);
        case EAccumulatorType::List:
        case EAccumulatorType::Avg:
            return MakeHolder<TTypedAccumulatorArray<TMaybeNullable<nullPolicy, TOldAvg>>>(size);
        case EAccumulatorType::Max:
            return MakeHolder<TTypedAccumulatorArray<TMaybeNullable<nullPolicy, TMax>>>(size);
        case EAccumulatorType::Min:
            return MakeHolder<TTypedAccumulatorArray<TMaybeNullable<nullPolicy, TMin>>>(size);
        case EAccumulatorType::Summ:
            return MakeHolder<TTypedAccumulatorArray<TMaybeNullable<nullPolicy, TSumm>>>(size);
        case EAccumulatorType::SummNone:
            return MakeHolder<TTypedAccumulatorArray<TMaybeNullable<nullPolicy, TSummNone>>>(size);
        }
    }
}

TAccumulator::TAccumulator(EAccumulatorType type, bool allowLegacyTypes)
    : Accumulator(CreateAccumulator(type, allowLegacyTypes))
{
}

TAccumulator* TAccumulator::FromYasmConfName(const TStringBuf yasmConfName) {
    EAccumulatorType type;
    if (!TryAccumulatorTypeFromYasmConfName(yasmConfName, type)) {
        return nullptr;
    }
    return new TAccumulator(type);
}

TAccumulatorsArray::TAccumulatorsArray(const EAccumulatorType type, const size_t size) {
    Inner.reserve(size);
    for (size_t i = 0; i < size; ++i) {
        Inner.emplace_back(type);
    }
}

void TAccumulatorsArray::Mul(const TVector<NZoom::NValue::TValue>& values, const ssize_t offset) {
    const size_t leftOffset = (offset >= 0) ? offset : 0;
    const size_t rightOffset = (offset < 0) ? -offset : 0;
    if (leftOffset > Inner.size() || rightOffset > values.size()) {
        return;
    }

    auto innerIt = Inner.begin() + leftOffset;
    auto valuesIt = values.cbegin() + rightOffset;

    while (innerIt != Inner.end() && valuesIt != values.cend()) {
        innerIt->Mul(*valuesIt);
        ++innerIt;
        ++valuesIt;
    }
}

TCompactAccumulatorsArray::TCompactAccumulatorsArray(const EAccumulatorType type, size_t size, ENullPolicy nullPolicy)
    : Type(type)
    , Size(size)
    , NullPolicy_{nullPolicy}
{
}

TCompactAccumulatorsArray::TCompactAccumulatorsArray(const TCompactAccumulatorsArray& other)
    : Type(other.Type)
    , Size(other.Size)
    , NullPolicy_{other.NullPolicy_}
{
    for (const auto offset : xrange(other.Len())) {
        Mul(other.GetValue(offset), offset);
    }
}

void TCompactAccumulatorsArray::MaybeInit() {
    if (!Inner) {
        switch (NullPolicy_) {
            case ENullPolicy::NonNullable:
                Inner = CreateAccumulatorArray<ENullPolicy::NonNullable>(Type, Size);
                break;
            case ENullPolicy::Nullable:
                Inner = CreateAccumulatorArray<ENullPolicy::Nullable>(Type, Size);
                break;
        }
    }
}

void TCompactAccumulatorsArray::Mul(const NZoom::NValue::TValue& value, size_t offset) {
    Mul(value.GetValue(), offset);
}

void TCompactAccumulatorsArray::Mul(const NZoom::NValue::TValueRef& value, size_t offset) {
    MaybeInit();
    value.Update(Inner->Get(offset));
}

void TCompactAccumulatorsArray::Set(const NZoom::NValue::TValue& value, size_t offset) {
    Set(value.GetValue(), offset);
}

void TCompactAccumulatorsArray::Set(const NZoom::NValue::TValueRef& value, size_t offset) {
    MaybeInit();
    auto& accumulator(Inner->Get(offset));
    accumulator.Clean();
    value.Update(accumulator);
}

void TCompactAccumulatorsArray::Merge(const NZoom::NValue::TValue& value, size_t offset) {
    Merge(value.GetValue(), offset);
}

void TCompactAccumulatorsArray::Merge(const NZoom::NValue::TValueRef& value, size_t offset) {
    MaybeInit();
    auto& accumulator = Inner->Get(offset);
    if (accumulator.GetValue().IsDefault()) {
        value.Update(accumulator);
    }
}

NZoom::NValue::TValueRef TCompactAccumulatorsArray::GetValue(size_t offset) const {
    if (Inner) {
        return Inner->Get(offset).GetValue();
    } else {
        ythrow yexception() << "accumulators not initialized";
    }
}

void TCompactAccumulatorsArray::Clean() {
    Inner.Reset();
}

size_t TCompactAccumulatorsArray::Len() const {
    if (Inner) {
        return Inner->Len();
    } else {
        return 0;
    }
}

bool TCompactAccumulatorsArray::Empty() const {
    return !Inner;
}

EAccumulatorType TCompactAccumulatorsArray::GetType() const {
    return Type;
}
