#pragma once

#include <infra/yasm/common/points/accumulators/abstract/accumulator.h>
#include <infra/yasm/common/points/accumulators/types/accumulators_types.h>
#include <infra/yasm/common/points/value/types.h>

#include <util/generic/string.h>

namespace NZoom {
    namespace NAccumulators {

        class TAccumulator {
        private:
            THolder<TAbstractAccumulator> Accumulator;
        public:
            TAccumulator(EAccumulatorType type, bool allowLegacyTypes = true);

            inline void Mul(const NZoom::NValue::TValueRef& value) {
                value.Update(*Accumulator.Get());
            }

            inline void Mul(const NZoom::NValue::TValue& value) {
                value.Update(*Accumulator.Get());
            }

            inline void Clean() {
                Accumulator->Clean();
            }

            inline NZoom::NValue::TValueRef GetValue() const {
                return Accumulator->GetValue();
            }

            static TAccumulator* FromYasmConfName(const TStringBuf yasmConfName);
        };

        class TAccumulatorsArray {
        private:
            TVector<TAccumulator> Inner;

        public:
            TAccumulatorsArray(const EAccumulatorType type, const size_t size);

            inline TAccumulatorsArray(TAccumulatorsArray&& other)
                : Inner(std::move(other.Inner))
            {
            }

            inline TVector<TAccumulator>::iterator begin() {
                return Inner.begin();
            }

            inline  TVector<TAccumulator>::iterator end() {
                return Inner.end();
            }

            inline TVector<TAccumulator>::const_iterator begin() const {
                return Inner.begin();
            }

            inline TVector<TAccumulator>::const_iterator end() const {
                return Inner.end();
            }

            inline size_t Len() const noexcept {
                return Inner.size();
            }

            inline bool Empty() const noexcept {
                return Inner.empty();
            }

            inline TAccumulator& operator[](size_t i) {
                return Inner[i];
            }

            inline const TAccumulator& operator[](size_t i) const {
                return Inner[i];
            }

            void Mul(const TVector<NZoom::NValue::TValue>& values, const ssize_t offset);
        };

        enum class ENullPolicy {
            Nullable,
            NonNullable,
        };

        class TCompactAccumulatorsArray {
        private:
            EAccumulatorType Type;
            size_t Size;
            THolder<IAbstractAccumulatorArray> Inner;
            ENullPolicy NullPolicy_{ENullPolicy::NonNullable};

        public:
            TCompactAccumulatorsArray(const EAccumulatorType type, size_t size, ENullPolicy nullPolicy = ENullPolicy::NonNullable);
            TCompactAccumulatorsArray(const TCompactAccumulatorsArray& other);

            void MaybeInit();

            void Mul(const NZoom::NValue::TValue& value, size_t offset);
            void Mul(const NZoom::NValue::TValueRef& value, size_t offset);

            void Set(const NZoom::NValue::TValue& value, size_t offset);
            void Set(const NZoom::NValue::TValueRef& value, size_t offset);

            void Merge(const NZoom::NValue::TValue& value, size_t offset);
            void Merge(const NZoom::NValue::TValueRef& value, size_t offset);

            NZoom::NValue::TValueRef GetValue(size_t offset) const;

            void Clean();
            size_t Len() const;
            bool Empty() const;
            EAccumulatorType GetType() const;
        };
    }
}

