#include "accumulators_types.h"

#include <util/system/unaligned_mem.h>
#include <util/generic/strbuf.h>
#include <util/string/cast.h>
#include <util/generic/vector.h>

#include <array>

using namespace NZoom::NAccumulators;

namespace {

    constexpr TStringBuf ACC_TYPE_LETTERS = "vchtxnme";
    constexpr std::array<EAccumulatorType, 8> ACC_TYPE_ARRAY = {{
        EAccumulatorType::Average,
        EAccumulatorType::Counter,
        EAccumulatorType::Hgram,
        EAccumulatorType::Last,
        EAccumulatorType::Max,
        EAccumulatorType::Min,
        EAccumulatorType::Summ,
        EAccumulatorType::SummNone
    }}; //https://bugs.llvm.org/show_bug.cgi?id=21629

    constexpr std::array<EAggregationMethod, 4> AGGREGATION_METHOD_ARRAY = {{
        EAggregationMethod::Agent,
        EAggregationMethod::Group,
        EAggregationMethod::MetaGroup,
        EAggregationMethod::Rollup,
    }};

    bool TryAccumulatorTypeFromLetter(const char letter, EAccumulatorType& type) noexcept {
        // Will be unrolled :)
        for (ui8 pos = 0; pos < ACC_TYPE_LETTERS.length(); ++pos) {
            if (ACC_TYPE_LETTERS[pos] == letter) {
                type = ACC_TYPE_ARRAY[pos];
                return true;
            }
        }
        return false;
    }

#define MP(name, value) std::pair<TStringBuf, EAccumulatorType>(name, EAccumulatorType::value)
    constexpr std::array<std::pair<TStringBuf, EAccumulatorType>, 11> ACC_TYPE_YC_MAP = {{
        MP("aver", Average),
        MP("counter", Counter),
        MP("hgram", Hgram),
        MP("lvalue", Last),
        MP("trnsp", Last),
        MP("avg", Avg),
        MP("list", List),
        MP("max", Max),
        MP("min", Min),
        MP("summ", Summ),
        MP("summnone", SummNone)
    }};
#undef MP

    inline size_t GetNameDelimiterPos(const TStringBuf name) noexcept {
        const auto length = name.length();
        if (length >= 4 && name[length - 4] == '_') { //_max
            return length - 4;
        }
        if (length >= 5 && name[length - 5] == '_') { //_summ, _tvvx
            return length - 5;
        }
        if (length >= 6 && name[length - 6] == '_') { //_hgram
            return length - 6;
        }
        return TStringBuf::npos;
    }

}

bool NZoom::NAccumulators::TryAccumulatorTypeFromYasmConfName(const TStringBuf name, EAccumulatorType& type) noexcept {
    for (const auto& p: ACC_TYPE_YC_MAP) {
        if (p.first == name) {
            type = p.second;
            return true;
        }
    }
    return false;
}

constexpr TAggregationRules::TAggregationRules(EAccumulatorType agent, EAccumulatorType group, EAccumulatorType metagroup, EAccumulatorType rollup,
    bool isDiffAggregation
)
    : Agent(agent)
    , Group(group)
    , Metagroup(metagroup)
    , Rollup(rollup)
    , DiffAggregation(isDiffAggregation)
{
}

EAccumulatorType TAggregationRules::GetAccumulatorType(const EAggregationMethod aggregationMethod) const noexcept {
    switch (aggregationMethod) {
    case EAggregationMethod::Agent:
        return Agent;
    case EAggregationMethod::Group:
        return Group;
    case EAggregationMethod::MetaGroup:
        return Metagroup;
    case EAggregationMethod::Rollup:
        return Rollup;
    }
}

bool TAggregationRules::IsDiffAggregation() const noexcept {
    return DiffAggregation;
}

constexpr TAggregationRules TAggregationRules::FillSame(const EAccumulatorType fill, bool isDiffAggregation) noexcept {
    return TAggregationRules(fill, fill, fill, fill, isDiffAggregation);
}

TMaybe<TAggregationRules> TAggregationRules::FromFullName(const TStringBuf name) noexcept {

    const auto pos = GetNameDelimiterPos(name);
    if (pos != TStringBuf::npos) {
        return NZoom::NAccumulators::TAggregationRules::FromName(TStringBuf(name, pos + 1));
    }
    return Nothing();
}

TMaybe<TAggregationRules> TAggregationRules::FromName(TStringBuf name) noexcept {
    if (Y_UNLIKELY(name.length() != 4)) {
        if (name == "hgram")
            return FillSame(EAccumulatorType::Hgram, true);
        if (name == "max")
            return FillSame(EAccumulatorType::Max, false);
        return Nothing();
    }
    if (Y_UNLIKELY(ReadUnaligned<ui32>(name.data()) == ReadUnaligned<ui32>("summ")))
        return FillSame(EAccumulatorType::Summ, true);

    //Exception-based error handling?
    EAccumulatorType groupType;
    if (!TryAccumulatorTypeFromLetter(name[1], groupType))
        return Nothing();

    EAccumulatorType agentType;
    const char agentLetter = name[0];

    const bool isDiffAggregation = (agentLetter == 'd');

    if (agentLetter == 'a' || agentLetter == 'd')
        agentType = groupType;
    else if (!TryAccumulatorTypeFromLetter(agentLetter, agentType))
        return Nothing();

    EAccumulatorType metagroupType;
    if (!TryAccumulatorTypeFromLetter(name[2], metagroupType))
        return Nothing();

    EAccumulatorType rollupType;
    if (!TryAccumulatorTypeFromLetter(name[3], rollupType))
        return Nothing();

    return TAggregationRules(agentType, groupType, metagroupType, rollupType, isDiffAggregation);
}

TMaybe<TAggregationRules> TAggregationRules::FromYasmConfName(const TStringBuf name) noexcept {
    EAccumulatorType fill;
    if (!TryAccumulatorTypeFromYasmConfName(name, fill))
        return Nothing();
    return FillSame(fill, false);
}

TVector<std::pair<TString, TString>> TAggregationRules::GetFormattedAccumulatorTypes() const {
    TVector<std::pair<TString, TString>> result;
    result.reserve(AGGREGATION_METHOD_ARRAY.size());
    for (const auto method : AGGREGATION_METHOD_ARRAY) {
        result.emplace_back(to_lower(ToString(method)), to_lower(ToString(GetAccumulatorType(method))));
    }
    return result;
}
