#include "accumulators_types.h"

#include <library/cpp/testing/unittest/registar.h>

#include <array>

using namespace NZoom::NAccumulators;

Y_UNIT_TEST_SUITE(TZoomTAggregationRulesFactoryTest) {

    const std::array<EAggregationMethod, 4> METHODS_ARRAY = {{
        EAggregationMethod::Agent,
        EAggregationMethod::Group,
        EAggregationMethod::MetaGroup,
        EAggregationMethod::Rollup
    }};

    void DoTestFromStringName(const TString& name, const EAccumulatorType targetFill) {
        TMaybe<TAggregationRules> rulesMaybe = TAggregationRules::FromName(name);
        UNIT_ASSERT(rulesMaybe.Defined());
        for (const auto method: METHODS_ARRAY) {
            UNIT_ASSERT_VALUES_EQUAL(rulesMaybe.GetRef().GetAccumulatorType(method), targetFill);
        }
    }

    Y_UNIT_TEST(AccumulatorsFromName) {
        DoTestFromStringName("hgram", EAccumulatorType::Hgram);
        DoTestFromStringName("summ", EAccumulatorType::Summ);
        DoTestFromStringName("max", EAccumulatorType::Max);

        {
            UNIT_ASSERT(!TAggregationRules::FromName("zzz").Defined());
            UNIT_ASSERT(!TAggregationRules::FromName("xnmz").Defined());
        }

        {
            TMaybe<TAggregationRules> rulesMaybe = TAggregationRules::FromName("vcht");
            UNIT_ASSERT(rulesMaybe.Defined());
            const TAggregationRules& rules = rulesMaybe.GetRef();
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Agent), EAccumulatorType::Average);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Group), EAccumulatorType::Counter);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::MetaGroup), EAccumulatorType::Hgram);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Rollup), EAccumulatorType::Last);
        }
        {
            TMaybe<TAggregationRules> rulesMaybe = TAggregationRules::FromName("xnme");
            UNIT_ASSERT(rulesMaybe.Defined());
            const TAggregationRules& rules = rulesMaybe.GetRef();
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Agent), EAccumulatorType::Max);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Group), EAccumulatorType::Min);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::MetaGroup), EAccumulatorType::Summ);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Rollup), EAccumulatorType::SummNone);
        }
        {
            TMaybe<TAggregationRules> rulesMaybe = TAggregationRules::FromName("avvv");
            UNIT_ASSERT(rulesMaybe.Defined());
            const TAggregationRules& rules = rulesMaybe.GetRef();
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Agent), EAccumulatorType::Average);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Group), EAccumulatorType::Average);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::MetaGroup), EAccumulatorType::Average);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Rollup), EAccumulatorType::Average);
        }
    }

    Y_UNIT_TEST(AccumulatorsFromYasmConfName) {
        UNIT_ASSERT(!TAggregationRules::FromYasmConfName("zzz").Defined());
        {
            TMaybe<TAggregationRules> rulesMaybe = TAggregationRules::FromYasmConfName("aver");
            UNIT_ASSERT(rulesMaybe.Defined());
            const TAggregationRules& rules = rulesMaybe.GetRef();
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Agent), EAccumulatorType::Average);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Group), EAccumulatorType::Average);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::MetaGroup), EAccumulatorType::Average);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Rollup), EAccumulatorType::Average);
        }
        {
            TMaybe<TAggregationRules> rulesMaybe = TAggregationRules::FromYasmConfName("avg");
            UNIT_ASSERT(rulesMaybe.Defined());
            const TAggregationRules& rules = rulesMaybe.GetRef();
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Agent), EAccumulatorType::Avg);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Group), EAccumulatorType::Avg);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::MetaGroup), EAccumulatorType::Avg);
            UNIT_ASSERT_VALUES_EQUAL(rules.GetAccumulatorType(EAggregationMethod::Rollup), EAccumulatorType::Avg);
        }
    }

    Y_UNIT_TEST(AccumulatorsFromFullName) {
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_").Empty());
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_1").Empty());
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_12").Empty());
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_123").Empty());
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_1234").Empty());
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_12345").Empty());
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_123456").Empty());

        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_max").Defined());
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_hgram").Defined());
        UNIT_ASSERT(TAggregationRules::FromFullName("some-random-signal-name_ammv").Defined());
    }
}
