#include "hgram.h"
#include "ugram_freezer.h"

#include <cmath>

using namespace NZoom::NHgram;


THgram::THgram(THolder<IHgramImpl>&& impl)
    : Impl(std::move(impl))
{
}

THgram THgram::Default(bool small) {
    if (small) {
        return THgram::Small(TVector<double>(), 0);
    } else {
        return THgram::Normal(TVector<double>(), 0, 0);
    }
}

THgram THgram::Small(TVector<double>&& values, const size_t zeros) {
    return THgram(MakeHolder<THgramSmall>(std::move(values), zeros));
}

THgram THgram::Normal(TVector<double>&& buckets, const size_t zeros, const i16 startPower) {
    return THgram(MakeHolder<THgramNormal>(std::move(buckets), zeros, startPower));
}

THgram THgram::EmptyUgram() {
    return THgram(MakeHolder<THgramUgram>());
}

// throws TUgramError
THgram THgram::Ugram(TUgramBuckets&& buckets) {
    return THgram(MakeHolder<THgramUgram>(std::move(buckets)));
}

THgram THgram::UgramNoCheck(TUgramBuckets&& buckets) {
    return THgram(MakeHolder<THgramUgram>(std::move(buckets), false));
}

void THgram::Clean() {
    if (IsUgram()) {
        Impl->Clean();
    } else {
        Impl.Reset(new THgramSmall(TVector<double>(), 0));
    }
}

size_t THgram::Len() const {
    return Impl->Len();
}

bool THgram::IsUgram() const noexcept {
    return Impl->IsUgram();
}

void THgram::Mul(THgramMultiplicand&& input) {
    input.Update(this);
}

void THgram::MulFloat(const double value) {
    auto newImpl = Impl->MulFloat(value);
    if (newImpl) {
        Impl = std::move(newImpl);
    }
}

void THgram::MulSlice(const TVector<double>& values) {
    auto newImpl = Impl->MulSlice(values);
    if (newImpl) {
        Impl = std::move(newImpl);
    }
}

void THgram::MulHgram(const THgram& value) {
    value.Impl->Update(this);
}

void THgram::MulSmallHgram(const TVector<double>& values, const size_t zeros) {
    auto newImpl = Impl->MulSmallHgram(values, zeros);
    if (newImpl) {
        Impl = std::move(newImpl);
    }
}

void THgram::MulNormalHgram(const TNormal& value) {
    auto newImpl = Impl->MulNormalHgram(value);
    if (newImpl) {
        Impl = std::move(newImpl);
    }
}

void THgram::MulUgramHgram(const TUgram& value) {
    auto newImpl = Impl->MulUgramHgram(value);
    if (newImpl) {
        Impl = std::move(newImpl);
    }

}

bool THgram::operator ==(const THgram& other) const {
    return *Impl == *other.Impl;
}

void THgram::Store(IHgramStorageCallback& callback) const {
    Impl->Store(callback);
}

THgramSmall::THgramSmall(TVector<double>&& values, const size_t zeros)
    : Values(std::move(values))
    , Zeros(zeros)
{
}

THgramSmall::THgramSmall(const THgramSmall& other)
    : Values(other.Values)
    , Zeros(other.Zeros)
{
}

size_t THgramSmall::Len() const {
    return Values.size();
}

bool THgramSmall::IsUgram() const noexcept {
    return false;
}

void THgramSmall::Update(THgram* other) {
    other->MulSmallHgram(std::move(Values), Zeros);
}

void THgramSmall::Store(IHgramStorageCallback& callback) const {
    callback.OnStoreSmall(Values, Zeros);
}

void THgramSmall::Clean() noexcept {
    Values.clear();
    Zeros = 0;
}

THolder<IHgramImpl> THgramSmall::MulFloat(const double value) {
   if (value <= 0) {
       ++Zeros;
   } else {
       Values.push_back(value);
   }
   if (CheckForUpdateToNormal()) {
       return THgramNormal::FromSmall(std::move(Values), Zeros);
   } else {
       return nullptr;
   }
}

THolder<IHgramImpl> THgramSmall::MulSlice(const TVector<double>& values) {
    if (!values.empty()) {
        for (const auto value: values) {
            if (value <= 0) {
                ++Zeros;
            } else {
                Values.push_back(value);
            }
        }
        if (CheckForUpdateToNormal()) {
            return THgramNormal::FromSmall(std::move(Values), Zeros);
        } else {
            return nullptr;
        }
    }
    return nullptr;
}

THolder<IHgramImpl> THgramSmall::MulSmallHgram(const TVector<double>& values, const size_t zeros) {
    Zeros += zeros;
    if (values.empty()) {
        return nullptr;
    }
    Values.insert(Values.end(), values.begin(), values.end());
    if (CheckForUpdateToNormal()) {
        return THgramNormal::FromSmall(std::move(Values), Zeros);
    } else {
        return nullptr;
    }
}

THolder<IHgramImpl> THgramSmall::MulNormalHgram(const TNormal& value) {
    THolder<IHgramImpl> newImpl = MakeHolder<THgramNormal>(value);
    THolder<IHgramImpl> updatedImpl = newImpl->MulSmallHgram(Values, Zeros);
    return updatedImpl == nullptr ? std::move(newImpl) : std::move(updatedImpl);
}

THolder<IHgramImpl> THgramSmall::MulUgramHgram(const TUgram& value) {
    THolder<IHgramImpl> newImpl = MakeHolder<THgramUgram>(TUgramBuckets(value.GetBuckets()));
    THolder<IHgramImpl> updatedImpl = newImpl->MulSmallHgram(std::move(Values), Zeros);
    return updatedImpl == nullptr ? std::move(newImpl) : std::move(updatedImpl);
}

bool THgramSmall::CheckForUpdateToNormal() const noexcept {
    return Values.size() > MAX_SMALL_HGRAM_SIZE;
}

bool THgramSmall::operator ==(const IHgramImpl& other) const {
    const THgramSmall* impl = dynamic_cast<const THgramSmall*>(&other);
    if (impl == nullptr) {
        return false;
    }
    return Zeros == impl->Zeros && Values == impl->Values;
}


THgramNormal::THgramNormal(TVector<double>&& values, const size_t zeros, const i16 startPower)
    : Value(std::move(values), zeros, startPower)
{
}

THgramNormal::THgramNormal(TNormal&& value)
    : Value(std::move(value))
{
}

THgramNormal::THgramNormal(const TNormal& value)
    :Value(value)
{
}

size_t THgramNormal::Len() const {
    return Value.Len();
}

bool THgramNormal::IsUgram() const noexcept {
    return false;
}

void THgramNormal::Update(THgram* other) {
    other->MulNormalHgram(std::move(Value));
}

void THgramNormal::Store(IHgramStorageCallback& callback) const {
    callback.OnStoreNormal(Value.GetBuckets(), Value.GetZerosCount(), Value.GetStartPower());
}

void THgramNormal::Clean() noexcept {
    Y_FAIL("HgramNormal should not be cleared, it should be replaced by SmallHgram");
}

THolder<IHgramImpl> THgramNormal::MulFloat(const double value) {
    Value.MulFloat(value);
    return nullptr;
}

THolder<IHgramImpl> THgramNormal::MulSlice(const TVector<double>& values) {
    Value.MulSlice(values);
    return nullptr;
}

THolder<IHgramImpl> THgramNormal::MulSmallHgram(const TVector<double>& values, const size_t zeros) {
    Value.MulSmallHgram(values, zeros);
    return nullptr;
}

THolder<IHgramImpl> THgramNormal::MulNormalHgram(const TNormal& value) {
    Value.MulNormalHgram(value);
    return nullptr;
}

THolder<IHgramImpl> THgramNormal::MulUgramHgram(const TUgram& value) {
    THolder<IHgramImpl> newImpl = MakeHolder<THgramUgram>(value);
    THolder<IHgramImpl> updatedImpl = newImpl->MulNormalHgram(std::move(Value));
    return updatedImpl == nullptr ? std::move(newImpl) : std::move(updatedImpl);
}

THolder<IHgramImpl> THgramNormal::FromUgram(const TUgramBuckets& buckets) {
    double weight = 0;
    double maxBound = 0;
    double minBound = 0;
    bool lastIsPoint = false;
    for (const auto& bucket : buckets) {
        if (bucket.Weight <= 0) {
            continue;
        }
        weight += bucket.Weight;
        maxBound = bucket.UpperBound;
        lastIsPoint = bucket.UpperBound == bucket.LowerBound;
        if (bucket.UpperBound > 0 && minBound == 0){
            if (bucket.LowerBound > 0){
                minBound = bucket.LowerBound;
            } else {
                minBound = bucket.UpperBound * LOWER_BORDER_FOR_BUCKET_WITH_ZERO;
            }
        }
    }

    if (maxBound <= 0) {
        return THolder<IHgramImpl>(new THgramNormal({}, weight, -1));
    }

    i16 maxPower;
//    If last bucket is point and it is degree of HGRAM_BASE we need additional half open interval
    if (lastIsPoint) {
        maxPower = 1 + std::floor(std::log(maxBound) / std::log(HGRAM_BASE));
    } else {
        maxPower = std::ceil(std::log(maxBound) / std::log(HGRAM_BASE));
    }
    i16 minPower = std::max(double(maxPower - MAX_NORMAL_HGRAM_SIZE), std::floor(std::log(minBound) / std::log(HGRAM_BASE)));

    TUgramBuckets frozenBuckets(Reserve(maxPower - minPower));
    for (auto id : xrange(minPower, maxPower)) {
        frozenBuckets.emplace_back(std::pow(HGRAM_BASE, id), std::pow(HGRAM_BASE, id + 1), 0);
    }

    minBound = frozenBuckets.front().LowerBound;

    TUgramBuckets bucketsForFreeze(Reserve(buckets.size()));
    for (const auto& bucket: buckets) {
        if (bucket.UpperBound < minBound){
            continue;
        }
        bucketsForFreeze.emplace_back(bucket);
        if (bucketsForFreeze.back().LowerBound < minBound) {
            bucketsForFreeze.back().LowerBound = minBound;
        }
    }

    TUgramBucketsFreezer freezer(frozenBuckets);
    TVector<double> normalBuckets(Reserve(maxPower - minPower));
    for (const auto& bucket : freezer.Freeze(bucketsForFreeze)) {
        if (normalBuckets.empty() && bucket.Weight == 0) {
            minPower += 1;
            continue;
        }
        normalBuckets.push_back(bucket.Weight);
        weight -= bucket.Weight;
    }

    return MakeHolder<THgramNormal>(std::move(normalBuckets), weight, minPower);
}

THolder<IHgramImpl> THgramNormal::FromSmall(const TVector<double>& values, const size_t zeros) {
    THolder<IHgramImpl> res = MakeHolder<THgramNormal>(TVector<double>(), 0, -1);
    THolder<IHgramImpl> updatedImpl = res->MulSmallHgram(values, zeros);
    return updatedImpl == nullptr ? std::move(res) : std::move(updatedImpl);
}

bool THgramNormal::operator ==(const IHgramImpl& other) const {
    const THgramNormal* impl = dynamic_cast<const THgramNormal*>(&other);
    if (impl == nullptr) {
        return false;
    }
    return Value == impl->Value;
}


THgramUgram::THgramUgram() {
}

THgramUgram::THgramUgram(TUgram&& value)
    : Value(std::move(value))
{
}


THgramUgram::THgramUgram(const TUgram& value)
    : Value(value)
{
}

THgramUgram::THgramUgram(TUgramBuckets&& buckets, bool check)
    : Value(check ? TUgramBuilder(std::move(buckets)).Build() : std::move(buckets))
{
}

size_t THgramUgram::Len() const {
    return Value.Len();
}

bool THgramUgram::IsUgram() const noexcept {
    return true;
}

void THgramUgram::Update(THgram* other) {
    other->MulUgramHgram(std::move(Value));
}

void THgramUgram::Store(IHgramStorageCallback& callback) const {
    callback.OnStoreUgram(Value.GetBuckets());
}

THolder<IHgramImpl> THgramUgram::MulFloat(const double value) {
    Value.MulFloat(value);
    return nullptr;
}

THolder<IHgramImpl> THgramUgram::MulSlice(const TVector<double>& values) {
    Value.MulSlice(values);
    return nullptr;
}

THolder<IHgramImpl> THgramUgram::MulSmallHgram(const TVector<double>& values, const size_t zeros) {
    Value.MulSmallHgram(values, zeros);
    return nullptr;
}

THolder<IHgramImpl> THgramUgram::MulNormalHgram(const TNormal& value) {
    Value.MulNormalHgram(value);
    return nullptr;
}

THolder<IHgramImpl> THgramUgram::MulUgramHgram(const TUgram& value) {
    Value.MulUgramHgram(value);
    return nullptr;
}

bool THgramUgram::operator ==(const IHgramImpl& other) const {
    const THgramUgram* impl = dynamic_cast<const THgramUgram*>(&other);
    if (impl == nullptr) {
        return false;
    }
    return Value == impl->Value;
}

void THgramUgram::Clean() noexcept {
    Value.Clean();
};

THolder<IHgramImpl> THgramUgram::FromSmall(const TVector<double>& values, const size_t zeros) {
    THolder<IHgramImpl> res = MakeHolder<THgramUgram>();
    THolder<IHgramImpl> updatedImpl = res->MulSmallHgram(values, zeros);
    return updatedImpl == nullptr ? std::move(res) : std::move(updatedImpl);
}

THolder<IHgramImpl> THgramUgram::FromNormal(const TVector<double>& values, const size_t zeros, const i16 startPower) {
    THolder<IHgramImpl> res = MakeHolder<THgramUgram>();
    TVector<double> newValues = values;
    TNormal normalHgram (std::move(newValues), zeros, startPower);
    THolder<IHgramImpl> updatedImpl = res->MulNormalHgram(normalHgram);
    return updatedImpl == nullptr ? std::move(res) : std::move(updatedImpl);
}

THgramMultiplicand::THgramMultiplicand(const double value)
    : Impl(new THgramMultiplicandFloat(value))
{
}

THgramMultiplicand::THgramMultiplicand(const TVector<double>& value)
    : Impl(new THgramMultiplicandSlice(value))
{
}

THgramMultiplicand::THgramMultiplicand(TVector<double>&& value)
    : Impl(new THgramMultiplicandSlice(std::move(value)))
{
}

THgramMultiplicand::THgramMultiplicand(THgram&& value)
    : Impl(new THgramMultiplicandHgram(std::move(value)))
{
}

void THgramMultiplicand::Update(THgram* hgramPtr) {
    Impl->Update(hgramPtr);
}


THgramMultiplicandFloat::THgramMultiplicandFloat(const double value)
    : Value(value)
{
}

void THgramMultiplicandFloat::Update(THgram* hgramPtr) {
    hgramPtr->MulFloat(Value);
}

THgramMultiplicandSlice::THgramMultiplicandSlice(const TVector<double>& value)
    : Value(value)
{
}

THgramMultiplicandSlice::THgramMultiplicandSlice(TVector<double>&& value)
    : Value(std::move(value))
{
}

void THgramMultiplicandSlice::Update(THgram* hgramPtr) {
    hgramPtr->MulSlice(std::move(Value));
}

THgramMultiplicandHgram::THgramMultiplicandHgram(THgram&& value)
    :Value(std::move(value))
{
}

void THgramMultiplicandHgram::Update(THgram* hgramPtr) {
    hgramPtr->MulHgram(std::move(Value));
}

