#pragma once

#include <infra/yasm/common/points/hgram/normal/normal.h>
#include <infra/yasm/common/points/hgram/ugram/ugram.h>

#include <util/generic/ptr.h>
#include <util/generic/vector.h>
#include <util/generic/array_ref.h>

namespace NZoom {
    namespace NHgram {
        const size_t MAX_SMALL_HGRAM_SIZE = 100;

        enum class EHgramType {
            SMALL,
            NORMAL,
            USER,
        };

        struct IHgramStorageCallback {
            virtual ~IHgramStorageCallback() = default;

            virtual void OnStoreSmall(const TVector<double>& values, const size_t zeroes) = 0;
            virtual void OnStoreSmall(const TArrayRef<const double>& values, size_t zeroes) {
                OnStoreSmall(TVector<double>(values.begin(), values.end()), zeroes);
            }
            virtual void OnStoreNormal(const TVector<double>& values, const size_t zeroes, const i16 startPower) = 0;
            virtual void OnStoreNormal(const TArrayRef<const double>& values, size_t zeroes, i16 startPower) {
                OnStoreNormal(TVector<double>(values.begin(), values.end()), zeroes, startPower);
            }
            virtual void OnStoreUgram(const TUgramBuckets& buckets) = 0;
        };

        class THgram;

        class IHgramImpl {
        public:
            virtual ~IHgramImpl() = default;
            virtual size_t Len() const = 0;

            virtual bool IsUgram() const noexcept = 0; // hack for metric manager.

            virtual THolder<IHgramImpl> MulFloat(const double value) = 0;
            virtual THolder<IHgramImpl> MulSlice(const TVector<double>& values) = 0;
            virtual THolder<IHgramImpl> MulSmallHgram(const TVector<double>& alues, const size_t zeros) = 0;
            virtual THolder<IHgramImpl> MulNormalHgram(const TNormal& value) = 0;
            virtual THolder<IHgramImpl> MulUgramHgram(const TUgram& value) = 0;

            virtual void Update(THgram* other) = 0;
            virtual void Store(IHgramStorageCallback& callback) const = 0;

            virtual void Clean() = 0;

            virtual EHgramType GetType() const noexcept = 0;
            virtual bool IsDefault() const noexcept = 0;

            virtual bool operator ==(const IHgramImpl& other) const = 0;

        };

        class IHgramMultiplicandImpl {
        public:
            virtual ~IHgramMultiplicandImpl() = default;

            virtual void Update(THgram* hgramPtr) = 0;
        };

        class THgramMultiplicand {
        private:
            THolder<IHgramMultiplicandImpl> Impl;

        public:
            explicit THgramMultiplicand(const double value);
            explicit THgramMultiplicand(const TVector<double>& value);
            explicit THgramMultiplicand(TVector<double>&& value);
            explicit THgramMultiplicand(THgram&& value);

            void Update(THgram* hgramPtr);
        };


        class THgramMultiplicandFloat : public IHgramMultiplicandImpl {
        private:
            double Value = 0.0;

        public:
            THgramMultiplicandFloat(const double value);

            virtual void Update(THgram* hgramPtr) override final;
        };

        class THgramMultiplicandSlice : public IHgramMultiplicandImpl {
        private:
            TVector<double> Value;

        public:
            THgramMultiplicandSlice(const TVector<double>& value);
            THgramMultiplicandSlice(TVector<double>&& value);

            virtual void Update(THgram* hgramPtr) override final;
        };

        class THgram {
        private:
            THolder<IHgramImpl> Impl;

            THgram(THolder<IHgramImpl>&& impl);

        public:
            static THgram Default(bool small = true);
            static THgram Small(TVector<double>&& values, const size_t zeros);
            static THgram Normal(TVector<double>&& buckets, const size_t zeros, const i16 startPower);
            static THgram EmptyUgram();
            static THgram Ugram(TUgramBuckets&& buckets); // throws TUgramError;
            static THgram UgramNoCheck(TUgramBuckets&& buckets);

            void Clean();
            size_t Len() const;
            bool IsUgram() const noexcept;
            void Mul(THgramMultiplicand&& input);
            void MulFloat(const double value);
            void MulSlice(const TVector<double>& values);
            void MulHgram(const THgram& value);
            void MulSmallHgram(const TVector<double>& values, const size_t zeros);
            void MulNormalHgram(const TNormal& value);
            void MulUgramHgram(const TUgram& value);

            void Store(IHgramStorageCallback& callback) const;

            EHgramType GetType() const noexcept {
                return Impl->GetType();
            }

            bool IsDefault() const noexcept {
                return Impl->IsDefault();
            }

            bool operator ==(const THgram& other) const;
        };

        class THgramMultiplicandHgram : public IHgramMultiplicandImpl {
        private:
            THgram Value;

        public:
            //Going to avoid HGram copying as far as possible or think about shared pointers here
            THgramMultiplicandHgram(THgram&& value);

            virtual void Update(THgram* hgramPtr) override final;
        };

        //FIXME move to impl folder since is used in tests and python bindings only
        class THgramSmall : public IHgramImpl {
        private:
            TVector<double> Values;
            size_t Zeros = 0;

        public:
            THgramSmall(TVector<double>&& values, const size_t zeros);
            THgramSmall(const THgramSmall& other);

            size_t Len() const override final;
            bool IsUgram() const noexcept override final;

            THolder<IHgramImpl> MulFloat(const double value) override final;
            THolder<IHgramImpl> MulSlice(const TVector<double>& values) override final;
            THolder<IHgramImpl> MulSmallHgram(const TVector<double>& values, const size_t zeros) override final;
            THolder<IHgramImpl> MulNormalHgram(const TNormal& value) override final;
            THolder<IHgramImpl> MulUgramHgram(const TUgram& value) override final;

            void Update(THgram* other) override final;
            void Store(IHgramStorageCallback& callback) const override final;
            void Clean() noexcept override final;

            EHgramType GetType() const noexcept override final {
                return EHgramType::SMALL;
            }

            bool IsDefault() const noexcept override final {
                return Values.empty() && !Zeros;
            }

            bool operator ==(const IHgramImpl& other) const override final;

        private:
            bool CheckForUpdateToNormal() const noexcept;
        };

        class THgramNormal : public IHgramImpl {
        private:
            TNormal Value;
        public:
            THgramNormal(TVector<double>&& values, const size_t zeros, const i16 startPower);
            THgramNormal(TNormal&& value);
            THgramNormal(const TNormal& value);

            size_t Len() const override final;
            bool IsUgram() const noexcept override final;

            THolder<IHgramImpl> MulFloat(const double value) override final;
            THolder<IHgramImpl> MulSlice(const TVector<double>& values) override final;
            THolder<IHgramImpl> MulSmallHgram(const TVector<double>& values, const size_t zeros) override final;
            THolder<IHgramImpl> MulNormalHgram(const TNormal& value) override final;
            THolder<IHgramImpl> MulUgramHgram(const TUgram& value) override final;

            void Update(THgram* other) override final;
            void Store(IHgramStorageCallback& callback) const override final;
            void Clean() noexcept override final;

            EHgramType GetType() const noexcept override final {
                return EHgramType::NORMAL;
            }

            bool IsDefault() const noexcept override final {
                return Value.IsDefault();
            }

            bool operator ==(const IHgramImpl& other) const override final;

            static THolder<IHgramImpl> FromSmall(const TVector<double>& values, const size_t zeros);
            static THolder<IHgramImpl> FromUgram(const TUgramBuckets& buckets);
        };

        class THgramUgram : public IHgramImpl {
        private:
            TUgram Value;

        public:
            THgramUgram();
            THgramUgram(TUgram&& value);
            THgramUgram(const TUgram& value);
            THgramUgram(TUgramBuckets&& buckets, bool check = true);

            size_t Len() const override final;
            bool IsUgram() const noexcept override final;

            THolder<IHgramImpl> MulFloat(const double value) override final;
            THolder<IHgramImpl> MulSlice(const TVector<double>& values) override final;
            THolder<IHgramImpl> MulSmallHgram(const TVector<double>& values, const size_t zeros) override final;
            THolder<IHgramImpl> MulNormalHgram(const TNormal& value) override final;
            THolder<IHgramImpl> MulUgramHgram(const TUgram& value) override final;

            void Update(THgram* other) override final;
            void Store(IHgramStorageCallback& callback) const override final;
            void Clean() noexcept override final;

            EHgramType GetType() const noexcept override final {
                return EHgramType::USER;
            }

            bool IsDefault() const noexcept override final {
                return Value.IsDefault();
            }

            bool operator ==(const IHgramImpl& other) const override final;

            static THolder<IHgramImpl> FromSmall(const TVector<double>& values, const size_t zeros);
            static THolder<IHgramImpl> FromNormal(const TVector<double>& values, const size_t zeros, const i16 startPower);
        };
    }
}

