#include "hgram.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NHgram;

Y_UNIT_TEST_SUITE(TZoomTHgramGenericTest) {

    TVector<double> FillRange(const ui32 from, const ui32 to) {
        TVector<double> values;
        if (to < from) {
            return values;
        }
        values.reserve(to - from);
        for (ui32 i = from; i < to; ++i) {
            values.push_back(i);
        }
        return values;
    }

    Y_UNIT_TEST(AddSingleValue) {
        THgram hgram = THgram::Default();
        hgram.Mul(THgramMultiplicand(0.0));
        hgram.Mul(THgramMultiplicand(1.0));

        THgram target = THgram::Small({1.0}, 1);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(Convert) {
        TVector<double> values = FillRange(1, 101);
        THgram hgram = THgram::Small(std::move(values), 0);
        UNIT_ASSERT_EQUAL(hgram.Len(), 100);
        hgram.Mul(THgramMultiplicand(101.0));

        THgram target = THgram::Normal({0.0, 1.0, 1.0, 1.0, 2.0, 2.0, 4.0, 6.0, 8.0, 13.0, 19.0, 29.0, 15.0}, 0, -1);
        UNIT_ASSERT_EQUAL(hgram, target);

        hgram.Clean();
        THgram target2 = THgram::Small({}, 0);
        UNIT_ASSERT_EQUAL(hgram, target2);
    }

    Y_UNIT_TEST(Clean) {
        THgram hgram = THgram::Default();
        hgram.Mul(THgramMultiplicand(0.0));
        hgram.Clean();
        UNIT_ASSERT_EQUAL(hgram, THgram::Default());
    }

    Y_UNIT_TEST(HgramDefaults) {
        THgram hgram = THgram::Default();
        THgram hgramSmall = THgram::Default(true);
        THgram hgramNotSmall = THgram::Default(false);
        UNIT_ASSERT(hgram.IsDefault());
        UNIT_ASSERT_EQUAL(hgram.GetType(), EHgramType::SMALL);
        UNIT_ASSERT(hgramSmall.IsDefault());
        UNIT_ASSERT_EQUAL(hgramSmall.GetType(), EHgramType::SMALL);
        UNIT_ASSERT(hgramNotSmall.IsDefault());
        UNIT_ASSERT_EQUAL(hgramNotSmall.GetType(), EHgramType::NORMAL);
    }

    Y_UNIT_TEST(SmallMulSmallWithoutOverflow) {
        TVector<double> firstValues = FillRange(1, 10);

        THgram hgram = THgram::Small(TVector<double>(firstValues), 3);
        TVector<double> secondValues(2);
        TVector<double> secondValuesNonZero = FillRange(5, 15);;

        secondValues.insert(secondValues.end(), secondValuesNonZero.begin(), secondValuesNonZero.end());
        hgram.Mul(THgramMultiplicand(std::move(secondValues)));

        TVector<double> oracleValues(firstValues);
        oracleValues.insert(oracleValues.end(), secondValuesNonZero.begin(), secondValuesNonZero.end());

        THgram target = THgram::Small(std::move(oracleValues), 5);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(SmallMulSmallWithOverflow) {
        THgram hgram = THgram::Small({1.0, 2.0, 3.0}, 0);
        hgram.Mul(THgramMultiplicand(THgram::Small(TVector<double>(100, 50.0), 0)));
        THgram target = THgram::Normal({0.0, 1.0, 1.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 100.0}, 0, -1);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(SmallMulSliceWithOverflow) {
        THgram hgram = THgram::Small({1.0, 2.0, 3.0}, 0);
        hgram.MulSlice(TVector<double>(100, 50.0));
        THgram target = THgram::Normal({0.0, 1.0, 1.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 100.0}, 0, -1);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(SmallMulSliceEmpty) {
        THgram hgram = THgram::Small({1.0, 2.0, 3.0}, 0);
        hgram.MulSlice(TVector<double>());
        THgram target = THgram::Small({1.0, 2.0, 3.0}, 0);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(SmallMulNormal) {
        THgram hgram = THgram::Small({1.0, 2.0, 3.0}, 0);
        hgram.Mul(THgramMultiplicand(THgram::Normal({0.0, 50.0, 1.0, 1.0, 50.0}, 0, -1)));
        THgram target = THgram::Normal({0.0, 51.0, 2.0, 2.0, 50.0}, 0, -1);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(NormalMulFloat) {
        THgram hgram = THgram::Normal({0.0, 50.0, 1.0, 1.0, 50.0}, 0, -1);
        hgram.MulFloat(1.0);
        THgram target = THgram::Normal({0.0, 51.0, 1.0, 1.0, 50.0}, 0, -1);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(BrokenHgram1) {
        THgram hgram = THgram::Normal({9.077580642823873, 12.033941532861252, 26.77311115154033, 63.40929557420627, 49.799350244462055,
            79.97461967144834, 57.646856578174784, 31.816302540707984, 30.81549837456226, 104.04340738344665, 58.63471168013447,
            10.037013061552372, 26.014616938253546, 30.923694625825824, 17.0, 3.0, 1.0}, 373, -14);
        hgram.Mul(THgramMultiplicand(THgram::Normal({1415.5352314903478, 1370.4787515265152, 1231.7071139616319, 1281.2303752220696,
            1020.7562833189043, 613.7663449946037, 410.596195572875, 224.1424453139891, 102.0565698612205, 23.849660712354577,
            20.918041087040375, 19.98519477537905, 12.987638017800725, 3.990154145267849, 3.0, 1.0, 1.0, 1.0, 1.0}, 1764, -14)));
        THgram target = THgram::Normal({1424.6128121331717, 1382.5126930593765, 1258.4802251131723, 1344.6396707962758, 1070.5556335633662,
            693.740964666052, 468.24305215104977, 255.9587478546971, 132.87206823578276, 127.89306809580123, 79.55275276717485,
            30.022207836931422, 39.00225495605427, 34.913848771093676, 20.0, 4.0, 2.0, 1.0, 1.0}, 2137, -14);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(BrokenHgram2_1) {
        THgram hgram = THgram::Normal({0.0, 3.1947013598032044, 45.5450382361079, 178.27330005228413, 40.993956287746826,
            8.993004064057967}, 4, -14);
        hgram.Mul(THgramMultiplicand(THgram::Default()));
        THgram target = THgram::Normal({0.0, 3.1947013598032044, 45.5450382361079, 178.27330005228413, 40.993956287746826,
                    8.993004064057967}, 4, -14);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(BrokenHgram2_2) {
        THgram hgram = THgram::Default();
        hgram.Mul(THgramMultiplicand(THgram::Normal({0.0, 3.1947013598032044, 45.5450382361079, 178.27330005228413, 40.993956287746826,
            8.993004064057967}, 4, -14)));
        THgram target = THgram::Normal({0.0, 3.1947013598032044, 45.5450382361079, 178.27330005228413, 40.993956287746826,
            8.993004064057967}, 4, -14);
        UNIT_ASSERT_EQUAL(hgram, target);
    }

    Y_UNIT_TEST(BrokenHgram3) {
        THgram hgram = THgram::Normal({0.0, 4.0, 9.0, 2.0, 1.0, 13.0, 58.0, 61.0, 9.0, 2.0}, 0, 21);
        hgram.MulSlice({577.0});
    }

    Y_UNIT_TEST(BrokenHgram4) {
        THgram hgram = THgram::Normal({0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.0, 9.0, 2.0, 1.0, 13.0, 58.0, 61.0, 9.0, 2.0}, 0, 14);
        hgram.MulSlice({414913.0, 64748.0, 316185.0, 444669.0, 692951.0, 370211.0, 21834.0, 353053.0, 269856.0, 405392.0,
            24192.0,421708.0, 436515.0, 316802.0, 291846.0, 309037.0, 368988.0, 577.0, 313729.0, 316107.0});
    }

    Y_UNIT_TEST(HgramShouldNotPanicWhileMul) {
        THgram hgram = THgram::Small({10.0, 1.0, 1.0, 2.0, 3.0, 1.0, 1.0, 44.0, 1.0, 3.0, 1.8446744073709552e+19}, 1428);
        hgram.Mul(THgramMultiplicand(THgram::Normal({0.0, 119.0, 51.0, 26.0, 15.0, 1.0, 1.0, 0.0, 1.0}, 1226, -1)));
        hgram.Mul(THgramMultiplicand(THgram::Normal({0.0, 63.0, 22.0, 8.0, 3.0, 4.0, 1.0}, 1337, -1)));
    }

    Y_UNIT_TEST(Ugram2Normal) {
        auto ugram = THgramNormal::FromUgram({
            TUgramBucket(0, 0, 5),
            TUgramBucket(0, 1.5, 0),
            TUgramBucket(1.5, 3.375, 25.0)
        });

        auto target = THgramNormal({10, 15}, 5, 1);
        UNIT_ASSERT_EQUAL(*ugram, target);
    }

    Y_UNIT_TEST(Ugram2NormalWithZero) {
        auto ugram = THgramNormal::FromUgram({TUgramBucket(0, 15, 9 * 32),});

        auto target = THgramNormal({16, 24, 36, 54, 81, 77}, 0, 1);

        UNIT_ASSERT_EQUAL(*ugram, target);
    }


    Y_UNIT_TEST(UgramFirst2Zerro) {
        auto ugram = THgramNormal::FromUgram({
            TUgramBucket(pow(1.5, -100), pow(1.5, -99), 15),
            TUgramBucket(1.5, pow(1.5, 3), 25),
            });

        auto target = THgramNormal({10, 15}, 15, 1);

        UNIT_ASSERT_EQUAL(*ugram, target);
    }


    Y_UNIT_TEST(UgramOnePoint) {
        auto ugram = THgramNormal::FromUgram({
            TUgramBucket(1.5, 1.5, 10),
            });

        auto target = THgramNormal({10}, 0, 1);

        UNIT_ASSERT_EQUAL(*ugram, target);
    }

    Y_UNIT_TEST(Ugram100Buckets) {
        auto ugram = THgramNormal::FromUgram({
            TUgramBucket(pow(1.5, -98) * 1.1, pow(1.5, -97) * 0.9, 15),
            TUgramBucket(pow(1.5, -97) * 1.1, pow(1.5, -96) * 0.9, 15),
            TUgramBucket(1.5, pow(1.5, 3), 25),
            });
        TVector<double> buckets(100, 0.);
        buckets[0] = 15;
        buckets[98] = 10;
        buckets[99] = 15;

        auto target = THgramNormal(std::move(buckets), 15, -97);

        UNIT_ASSERT_EQUAL(*ugram, target);
    }

    Y_UNIT_TEST(Ugram2NormalZeros) {
        auto ugram = THgramNormal::FromUgram({
            TUgramBucket(0, 0, 5),
            TUgramBucket(0, 5, 0),
        });

        auto target = THgramNormal({}, 5, -1);

        UNIT_ASSERT_EQUAL(*ugram, target);
    }
}

