#pragma once

#include <util/generic/algorithm.h>
#include <util/generic/vector.h>
#include <util/generic/xrange.h>

namespace NZoom {
    namespace NHgram {

        class TUgramBucket {
        public:
            double LowerBound = 0.0;
            double UpperBound = 0.0;
            double Weight = 0.0;

            TUgramBucket();
            TUgramBucket(const double lowerBound, const double upperBound, const double weight);

            bool operator ==(const TUgramBucket& other) const noexcept;
            bool operator !=(const TUgramBucket& other) const noexcept;
            bool operator <(const TUgramBucket& other) const noexcept;
            bool operator <(const double other) const noexcept;

            double Size() const noexcept;
            bool IsPoint() const noexcept;

            static TUgramBucket Point(const double value, const double weight);
        };

        using TUgramBuckets = TVector<TUgramBucket>;

        inline bool HaveDifferentBounds(const TUgramBucket& first, const TUgramBucket& second) noexcept {
            return first.LowerBound != second.LowerBound || first.UpperBound != second.UpperBound;
        }

        inline bool HaveSameBounds(const TUgramBucket& first, const TUgramBucket& second) noexcept {
            return !HaveDifferentBounds(first, second);
        }

        inline bool HaveDifferentBounds(const TUgramBuckets& first, const TUgramBuckets& second) noexcept {
            if (first.size() == second.size()) {
                return AnyOf(xrange(first.size()), [&first, &second](int index) {
                    return HaveDifferentBounds(first[index], second[index]);
                });
            } else {
                return true;
            }
        }

        inline bool HaveSameBounds(const TUgramBuckets& first, const TUgramBuckets& second) noexcept {
            return !HaveDifferentBounds(first, second);
        }
    }
}
