#include "compress.h"

#include <util/generic/ylimits.h>
#include <util/system/tls.h>

using namespace NZoom::NHgram;

TUgramCompressor::TUgramCompressor() {
    Buffer.reserve(UGRAM_LIMIT);
    Result.reserve(UGRAM_LIMIT);
}

namespace {
    constexpr double POINT_FROM_SMALLEST_BUCKET_COEFFICIENT = 0.1;
    constexpr double POINT_FROM_TOTAL_INTERVAL_COEFFICIENT = 0.01;

    Y_STATIC_THREAD(TUgramCompressor) COMPRESSOR;
    Y_STATIC_THREAD(TLazyUgramCompressor) LAZY_COMPRESSOR;
}

// Remove zero series and calculate min width
double InitialCompress(const TUgramBuckets& buckets, TUgramBuckets& result) {
    double minWidth = Max<double>();
    for (const auto& bucket: buckets) {
       const double size = bucket.Size();
       if (size != 0.0 && size < minWidth) {
            minWidth = size;
       }

       if (bucket.Weight == 0.0) {
           continue;
        }
        result.push_back(bucket);
    }

    if (minWidth == Max<double>()) {
        // NOTE(rocco66): all buckets are points case
        if (!buckets.empty()) {
           const double allBucketsSize = buckets.back().UpperBound - buckets.front().LowerBound;
           return allBucketsSize * POINT_FROM_TOTAL_INTERVAL_COEFFICIENT;
        } else {
            return 1.0;  // NOTE(rocco66): it is impossible anyway, we have at least UGRAM_LIMIT buckets
        }
    } else {
        return minWidth * POINT_FROM_SMALLEST_BUCKET_COEFFICIENT;
    }
}


const TUgramBuckets& TUgramCompressor::Compress(const TUgramBuckets& ugramBuckets, const size_t bucketBordersLimit) {
    Result.clear();
    const double minWidth = InitialCompress(ugramBuckets, Result);
    if (Result.size() > FAST_COMPRESS_LIMIT) {
        Buffer.clear();
        Buffer.swap(Result);
        Fast.Compress(minWidth, Buffer, Result);
    }

    TRawBucketLimitCounter rawBucketCounter(Result, bucketBordersLimit);
    if (Result.size() > SLOW_COMPRESS_LIMIT || rawBucketCounter.ShouldWeCompress()) {
        Buffer.clear();
        Buffer.swap(Result);
        Slow.Compress(minWidth, Buffer, Result, rawBucketCounter);
    }
    return Result;
}

TUgramCompressor& TUgramCompressor::GetInstance() {
    return COMPRESSOR.Get();
}

TLazyUgramCompressor::TLazyUgramCompressor()
    : BaseCompressor()
{
}

const TUgramBuckets& TLazyUgramCompressor::Compress(const TUgramBuckets& ugramBuckets, size_t bordersLimit) {
    auto curBordersCount = CountUniqueBucketBorders(ugramBuckets);
    if (curBordersCount <= LAZY_UGRAM_COMPRESSION_THRESHOLD && (bordersLimit == 0 || curBordersCount <= bordersLimit)) {
        return ugramBuckets;
    } else {
        return BaseCompressor.Compress(ugramBuckets, bordersLimit);
    }
}

TLazyUgramCompressor& TLazyUgramCompressor::GetInstance() {
    return LAZY_COMPRESSOR.Get();
}
