#include "fast.h"

#include <util/generic/algorithm.h>
#include <util/generic/maybe.h>

using namespace NZoom::NHgram;

namespace {
    constexpr double EPSILON = 0.01;
    constexpr size_t WINDOW_SIZE = 4; // This value can't be < 2
}


TMergedBuckets::TMergedBuckets(const double minWidth, const TUgramBucket& bucket) {
    const double x = Max(bucket.Weight, EPSILON);
    const double y = Max(minWidth, bucket.Size());
    const double dens = x / y;

    LowerBound = bucket.LowerBound;
    UpperBound = bucket.UpperBound;
    Weight = bucket.Weight;
    MinDens = dens;
    MaxDens = dens;
    Damage = 0.0;
}

TMergedBuckets::TMergedBuckets(const double minWidth, const TMergedBuckets& left, const TMergedBuckets& right) {
    LowerBound = left.LowerBound;
    UpperBound = right.UpperBound;
    Weight = left.Weight + right.Weight;

    const double midBound = right.LowerBound;
    const double leftDensity = Max(left.Weight, EPSILON) / Max(minWidth, midBound - left.LowerBound);
    const double rightDensity = Max(right.Weight, EPSILON) / Max(minWidth, right.UpperBound - midBound);

    std::tie(MinDens, MaxDens) = MinMax(leftDensity, rightDensity);

    const double commonDensity = Weight / (right.UpperBound - left.LowerBound);
    Damage = Max(MaxDens / commonDensity, commonDensity / MinDens);
}

TFastCompressor::TFastCompressor() {
    Buffer.reserve(FAST_COMPRESS_LIMIT);
    Result.reserve(FAST_COMPRESS_LIMIT);
}

std::pair<TFastCompressor::TWindowIt, TMergedBuckets> TFastCompressor::FindBest(const double minWidth, const TList<TMergedBuckets>& window) {
    // [a, b, c, d] => [(a, b), (b, c), (c, d)]

    auto prevIt = window.begin();
    TMaybe<std::pair<TWindowIt, TMergedBuckets>> bestMaybe;
    auto currentIt = prevIt;
    ++currentIt;
    for (; currentIt != window.end(); ++currentIt) {
        TMergedBuckets current(minWidth, *prevIt, *currentIt);
        if (bestMaybe.Empty() || bestMaybe.GetRef().second.Damage > current.Damage) {
            bestMaybe = std::make_pair(prevIt, current);
        }
        prevIt = currentIt;
    }
    return bestMaybe.GetRef();
}

bool TFastCompressor::Iteration(const double minWidth, TFastCompressor::TItPair& src, TList<TMergedBuckets>& window) {
    // Window should be full
    while (window.size() < WINDOW_SIZE && src.first != src.second) {
        window.push_back(*src.first);
        ++src.first;
    }
    // Yield leftover item if window is not full after prevoius step,
    if (window.size() <= 1) {
        if (!window.empty()) {
            Result.push_back(window.front());
            window.pop_front();
        }
        return false;
    }
    // Window is full for that moment
    std::pair<TWindowIt, TMergedBuckets> best = FindBest(minWidth, window);
    // Yield values before merged
    std::copy(window.cbegin(), best.first, std::back_inserter(Result));
    window.erase(window.cbegin(), best.first);

    // Forget about merged values
    window.pop_front();
    window.pop_front();

    // Yield merged value
    Result.push_back(best.second);

    //FIXME can be removed since code at the beginning does the same thing?
    while (window.size() < WINDOW_SIZE && src.first != src.second) {
        window.push_back(*src.first);
        ++src.first;
    }
    return true;
}

void TFastCompressor::CompressLoop(const double minWidth, TFastCompressor::TItPair& src) {
    TList<TMergedBuckets> window;
    while (Iteration(minWidth, src, window));
}

void TFastCompressor::Compress(const double minWidth, const TUgramBuckets& buckets, TUgramBuckets& result) {
    if (buckets.empty()) {
        result.clear();
        return;
    }

    auto mbLast = buckets.back();

    Result.clear();
    std::transform(buckets.cbegin(), buckets.cend() - 1, std::back_inserter(Result),
        [minWidth](const TUgramBucket& bucket) -> TMergedBuckets {return TMergedBuckets(minWidth, bucket);});

   do {
        Buffer.clear();
        Result.swap(Buffer);

        TItPair src = std::make_pair(Buffer.cbegin(), Buffer.cend());
        CompressLoop(minWidth, src);
    } while (Result.size() - 1 > FAST_COMPRESS_LIMIT);

    std::transform(Result.cbegin(), Result.cend(), std::back_inserter(result),
        [](const TMergedBuckets& bucket) -> TUgramBucket {
            return TUgramBucket(bucket.LowerBound, bucket.UpperBound, bucket.Weight);
        });

    result.push_back(mbLast);
}
