#include "fast.h"
#include "compress.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/xrange.h>

using namespace NZoom::NHgram;

double InitialCompress(const TUgramBuckets& buckets, TUgramBuckets& result);


Y_UNIT_TEST_SUITE(TZoomTHgramUgramFastCompressorTest) {
    TUgramBucket ub(const double lowerBound, const double upperBound, const double weight) {
        return TUgramBucket(lowerBound, upperBound, weight);
    }

    void FastTest(const TUgramBuckets& test, const TUgramBuckets& target) {

        TUgramBuckets buckets;
        const double minWidth = InitialCompress(test, buckets);

        TFastCompressor compressor;
        TUgramBuckets result;
        compressor.Compress(minWidth, test, result);
        UNIT_ASSERT_EQUAL(result, target);
    }

    Y_UNIT_TEST(SimpleCompressSingleItetation) {
        FastTest({
                ub(1.0, 2.0, 1.0),
                ub(2.0, 3.0, 1.0),
                ub(3.0, 4.0, 1.0),
                ub(4.0, 5.0, 1.0),
                ub(5.0, 6.0, 1.0),
                ub(6.0, 7.0, 1.0),
                ub(7.0, 8.0, 1.0),
                ub(8.0, 9.0, 1.0)
            },
            {
                ub(1.0, 3.0, 1.0 + 1.0),
                ub(3.0, 5.0, 1.0 + 1.0),
                ub(5.0, 7.0, 1.0 + 1.0),
                ub(7.0, 8.0, 1.0),
                ub(8.0, 9.0, 1.0),
            });
    }

    Y_UNIT_TEST(AnotherCompressSingleItetation) {
        FastTest({
                ub(1.0, 2.0, 1.0),
                ub(2.0, 3.0, 2.0),
                ub(3.0, 4.0, 3.0), // merge this
                ub(4.0, 5.0, 4.0), // and this
                ub(5.0, 6.0, 5.0),
                ub(6.0, 7.0, 6.0),
                ub(7.0, 8.0, 7.0),  // this
                ub(8.0, 9.0, 8.0),  // and this
                ub(9.0, 10.0, 9.0)
            },
            {
                ub(1.0, 2.0, 1.0),
                ub(2.0, 3.0, 2.0),
                ub(3.0, 5.0, 3.0 + 4.0),
                ub(5.0, 6.0, 5.0),
                ub(6.0, 7.0, 6.0),
                ub(7.0, 9.0, 7.0 + 8.0),
                ub(9.0, 10.0, 9.0)
            });

        FastTest({
                ub(5.0, 10.0, 1.0),
                ub(10.0, 12.0, 3.0),  // merge this
                ub(15.0, 20.0, 3.0),  // and this
                ub(30.0, 30.0, 2.0),
                ub(30.0, 40.0, 3.0)
            },
            {
                ub(5.0, 10.0, 1.0),
                ub(10.0, 20.0, 3.0 + 3.0),
                ub(30.0, 30.0, 2.0),
                ub(30.0, 40.0, 3.0)
            });
    }

    Y_UNIT_TEST(TwoPointsCompressSingleItetation) {
        // merge two points
        FastTest({
            ub(1.0, 1.0, 1.0),
            ub(50.0, 50.0, 1.0),  // merge this
            ub(51.0, 51.0, 1.0),  // and this
            ub(100.0, 100.0, 1.0),
            ub(200.0, 200.0, 1.0)
        },
        {
            ub(1.0, 1.0, 1.0),
            ub(50.0, 51.0, 2.0),
            ub(100.0, 100.0, 1.0),
            ub(200.0, 200.0, 1.0)
        });
    }

    TUgramBuckets GenBuckets(size_t start, size_t stop) {
        TUgramBuckets res;
        for (const size_t i: xrange(start, stop)) {
            res.emplace_back(i, i + 1, 1.0);
        }
        return res;
    }

    double WeightSum(const TUgramBuckets& buckets) {
        double sum = 0;
        for (const auto& b: buckets) {
            sum += b.Weight;
        }
        return sum;
    }

    void AssertBucketsEq(const TUgramBucket& a, const TUgramBucket& b) {
        UNIT_ASSERT_EQUAL(a, b);
        UNIT_ASSERT_EQUAL(a.Weight, b.Weight);
    }

    Y_UNIT_TEST(CompressBucketsWithSameWeight) {
        auto origBuckets = GenBuckets(1, 200);
        UNIT_ASSERT_EQUAL(origBuckets.size(), 199);

        TUgramBuckets buckets;
        TFastCompressor compressor;
        compressor.Compress(0.1, origBuckets, buckets);

        UNIT_ASSERT_EQUAL(WeightSum(origBuckets), WeightSum(buckets));
        UNIT_ASSERT(buckets.size() <= FAST_COMPRESS_LIMIT);

        // NOTE(rocco66): compress buckets on start, then at the end
        UNIT_ASSERT_EQUAL(buckets[0].Size(), buckets[0].Weight);
        UNIT_ASSERT_EQUAL(buckets[25].Size(), buckets[25].Weight);

        const auto& lf4 = buckets[buckets.size() - 4];
        UNIT_ASSERT_EQUAL(lf4.Size(), lf4.Weight);

        // NOTE(rocco66): this is uncompressed "tail"
        AssertBucketsEq(buckets.back(), TUgramBucket(199.0, 200.0, 1.0));
    }

    Y_UNIT_TEST(CompressBucketsWithOnePike) {
        TUgramBuckets origBuckets;

        origBuckets.emplace_back(1.0, 2.0, 1.0);
        origBuckets.emplace_back(2.0, 3.0, 1000.0);
        const auto moreBuckets = GenBuckets(3, 300);
        origBuckets.insert(origBuckets.end(), moreBuckets.begin(), moreBuckets.end());


        TUgramBuckets buckets;
        TFastCompressor compressor;
        compressor.Compress(0.1, origBuckets, buckets);

        UNIT_ASSERT_EQUAL(WeightSum(origBuckets), WeightSum(buckets));
        UNIT_ASSERT(buckets.size() <= FAST_COMPRESS_LIMIT);

        // NOTE(rocco66): second bucket "protects" first from merge
        AssertBucketsEq(buckets[0], ub(1.0, 2.0, 1.0));

        // NOTE(rocco66): too heavy for merge
        AssertBucketsEq(buckets[1], ub(2.0, 3.0, 1000.0));

        // NOTE(rocco66): simple bucket, was merged many times
        AssertBucketsEq(buckets[2], ub(3.0, 7.0, 4.0));
    }
}

