#include "slow.h"

#include <util/generic/algorithm.h>
#include <util/generic/list.h>

using namespace NZoom::NHgram;

namespace {
    constexpr size_t CLOSE_ENDED_BUCKET_RAW_SIZE = 1;
    constexpr size_t OPEN_ENDED_BUCKET_RAW_SIZE = 2;

    double GetBucketDensity(const TUgramBucket& bucket, const double pointSize) {
        const double bucketSize = bucket.Size();
        return bucket.Weight / ((bucketSize == 0.0) ? pointSize : bucketSize);
    }

    void MergeNeighbor(TUgramBucket& self, const TUgramBucket& other) {
        // TODO(rocco66): note
        self.LowerBound = Min(self.LowerBound, other.LowerBound);
        self.UpperBound = Max(self.UpperBound, other.UpperBound);
        self.Weight += other.Weight;
    }

    void MergeBucketSet(
        TList<TCompressBucketsSet>& sets,
        const TList<TCompressBucketsSet>::iterator& mergePoint,
        TRawBucketLimitCounter& rawBucketCounter
    ) {
        if (mergePoint->GetLeft().UpperBound == mergePoint->GetRight().LowerBound) {
            rawBucketCounter.CloseEndedBucketWasMerged();
        } else {
            // NOTE(rocco66): one more empty bucket was needed for sparse intervals
            rawBucketCounter.OpenEndedBucketWasMerged();
        }

        if (mergePoint != sets.cbegin()) {
            auto prev = mergePoint;
            --prev;
            prev->RightMerge(*mergePoint);
        }
        auto next = mergePoint;
        ++next;
        if (next != sets.end()) {
           next->LeftMerge(*mergePoint);
        }
        sets.erase(mergePoint);
    }
}

namespace NZoom::NHgram {
    size_t CountUniqueBucketBorders(const TUgramBuckets& buckets) {
        // NOTE(rocco66): raw bucket is pair of doubles: 'bound' (upper in stockpile case) and 'weight'
        // NOTE(rocco66): we will need two raw buckets if interval just after bucket is empty
        size_t counter = 0;
        for (auto it = buckets.begin(); it != buckets.end(); it++) {
            auto nextBucket = it + 1;
            if (nextBucket == buckets.end() || it->UpperBound != nextBucket->LowerBound) {
                counter += OPEN_ENDED_BUCKET_RAW_SIZE;
            } else {
                counter += CLOSE_ENDED_BUCKET_RAW_SIZE;
            }
        }
        return counter;
    }
}

void CompressIteration(TList<TCompressBucketsSet>& sets, TRawBucketLimitCounter& rawBucketCounter) {
    const auto it = MinElement(sets.begin(), sets.end(),
        [](const auto& a, const auto& b) {
            return a.GetDamage() < b.GetDamage();
        }
    );
    if (it != sets.end()) {
        MergeBucketSet(sets, it, rawBucketCounter);
    }
}


void SetsToBuckets(const TList<TCompressBucketsSet>& sets, TUgramBuckets& output) {
    if (sets.empty()) {
        return;
    }
    output.reserve(output.size() + sets.size() + 1);
    auto it = sets.cbegin();
    output.push_back(it->GetLeft());
    output.push_back(it->GetRight());
    ++it;
    std::transform(it, sets.cend(), std::back_inserter(output), [](const auto& s) { return s.GetRight(); });
}


TList<TCompressBucketsSet> BucketsToSets(const TUgramBuckets& buckets, const double minWidth) {
    TList<TCompressBucketsSet> sets;
    if (buckets.size() > 1) {
        for (auto it = buckets.cbegin() + 1; it != buckets.cend(); ++it) {
            sets.emplace_back(*(it - 1), *it, minWidth);
        }
    }
    return sets;
}

TCompressBucketsSet::TCompressBucketsSet() {
}

TCompressBucketsSet::TCompressBucketsSet(const TUgramBucket& left, const TUgramBucket& right, const double pointSize)
    : Left(left)
    , Right(right)
    , TotalWeight(left.Weight + right.Weight)
    , TotalSize(right.UpperBound - left.LowerBound)
    , MaxBucketDensity(Max(GetBucketDensity(left, pointSize), GetBucketDensity(right, pointSize)))
    , Damage(MaxBucketDensity / (TotalWeight / TotalSize))
{
}

void TCompressBucketsSet::LeftMerge(TCompressBucketsSet& other) {
    if (other.MergedUgrams.Defined()) {
        Left = other.MergedUgrams.GetRef();
    } else {
        MergeNeighbor(Left, other.Left);
        other.MergedUgrams = Left;
    }
    MergeCommon(other);
}

void TCompressBucketsSet::RightMerge(TCompressBucketsSet& other) {
    if (other.MergedUgrams.Defined()) {
        Right = other.MergedUgrams.GetRef();
    } else {
        MergeNeighbor(Right, other.Right);
        other.MergedUgrams = Right;
    }
    MergeCommon(other);
}

void TCompressBucketsSet::MergeCommon(const TCompressBucketsSet& other) {
    TotalWeight = Left.Weight + Right.Weight;
    TotalSize = Right.UpperBound - Left.LowerBound;
    MaxBucketDensity = Max(MaxBucketDensity, other.MaxBucketDensity);
    Damage = MaxBucketDensity / (TotalWeight / TotalSize);
}

double TCompressBucketsSet::GetDamage() const noexcept {
    return Damage;
}

const TUgramBucket& TCompressBucketsSet::GetLeft() const noexcept {
    return Left;
}

const TUgramBucket& TCompressBucketsSet::GetRight() const noexcept {
    return Right;
}

TRawBucketLimitCounter::TRawBucketLimitCounter(const TUgramBuckets& buckets, const size_t rawBucketLimit)
    : Limit(rawBucketLimit)
    , Counter{}
{
    if (Limit != 0) {
        Counter = CountUniqueBucketBorders(buckets);
    }
}

bool TRawBucketLimitCounter::ShouldWeCompress() const {
    return Limit != 0 && Counter > Limit;
}

void TRawBucketLimitCounter::OpenEndedBucketWasMerged() {
    if (Limit != 0) {
        Y_ASSERT(Counter > OPEN_ENDED_BUCKET_RAW_SIZE);
        Counter -= OPEN_ENDED_BUCKET_RAW_SIZE;
    };
}

void TRawBucketLimitCounter::CloseEndedBucketWasMerged() {
    if (Limit != 0) {
        Y_ASSERT(Counter > CLOSE_ENDED_BUCKET_RAW_SIZE);
        Counter -= CLOSE_ENDED_BUCKET_RAW_SIZE;
    };
}

size_t TRawBucketLimitCounter::GetCounterValue() const {
    return Counter;
}

void TSlowCompressor::Compress(
        const double minWidth,
        const TUgramBuckets& buffer,
        TUgramBuckets& result,
        TRawBucketLimitCounter& rawBucketCounter
) {
    if (buffer.size() < 2) {
        result.insert(result.end(), buffer.cbegin(), buffer.cend());
        return;
    }

    auto sets = BucketsToSets(buffer, minWidth);

    while (sets.size() > SLOW_COMPRESS_LIMIT - 1 or rawBucketCounter.ShouldWeCompress()) {
        CompressIteration(sets, rawBucketCounter);
    }

    SetsToBuckets(sets, result);
}
