#include "compress.h"
#include "slow.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NHgram;

double InitialCompress(const TUgramBuckets& buckets, TUgramBuckets& result);
void SetsToBuckets(const TList<TCompressBucketsSet>& sets, TUgramBuckets& output);
TList<TCompressBucketsSet> BucketsToSets(const TUgramBuckets& buckets, const double minWidth);
void CompressIteration(TList<TCompressBucketsSet>& sets, TRawBucketLimitCounter& rawBucketCounter);

TUgramBucket ub(const double lowerBound, const double upperBound, const double weight) {
    return TUgramBucket(lowerBound, upperBound, weight);
}

Y_UNIT_TEST_SUITE(TZoomTHgramUgramSlowCompressorTest) {

    void SlowsTest(const TUgramBuckets& test, const TUgramBuckets& target, const size_t rawBucketLimit = 0) {

        TUgramBuckets buckets;
        const double minWidth = InitialCompress(test, buckets);
        auto sets = BucketsToSets(test, minWidth);
        TRawBucketLimitCounter state {test, rawBucketLimit};
        CompressIteration(sets, state);
        TUgramBuckets result;
        SetsToBuckets(sets, result);

        UNIT_ASSERT_VALUES_EQUAL(result, target);
    }

    Y_UNIT_TEST(SingleIterationFirstPair) {
        SlowsTest({
                ub(1.0, 2.0, 1.0), // this
                ub(2.0, 3.0, 1.0), // and this
                ub(5.5, 6.0, 1.0),
                ub(6.0, 6.1, 1.0),
                ub(8.0, 9.0, 1.0)
            },
            {
                ub(1.0, 3.0, 1.0 + 1.0),
                ub(5.5, 6.0, 1.0),
                ub(6.0, 6.1, 1.0),
                ub(8.0, 9.0, 1.0)
            }
        );
    }

    Y_UNIT_TEST(SingleIterationLastPair) {
        SlowsTest({
                ub(1.0, 2.0, 1.0),
                ub(2.5, 3.0, 1.0),
                ub(3.0, 3.1, 1.0),
                ub(4.0, 5.0, 1.0), // this
                ub(5.0, 6.0, 1.0) // and this
            },
            {
                ub(1.0, 2.0, 1.0),
                ub(2.5, 3.0, 1.0),
                ub(3.0, 3.1, 1.0),
                ub(4.0, 6.0, 1.0 + 1.0)
            }
        );
    }
}

Y_UNIT_TEST_SUITE(TZoomUgramCompressWithRawBucketsLimitTest) {

    void Test(const TUgramBuckets& test, const TUgramBuckets& target, const size_t rawBucketLimit = 0) {
        auto& compressor = TUgramCompressor::GetInstance();
        UNIT_ASSERT_VALUES_EQUAL(compressor.Compress(test, rawBucketLimit), target);
    }

    Y_UNIT_TEST(NoCompress) {
        Test(
            {
                ub(1.0, 2.0, 1.0),
                ub(2.0, 3.0, 1.0),
                ub(3.0, 4.0, 1.0)
            },
            {
                ub(1.0, 2.0, 1.0),
                ub(2.0, 3.0, 1.0),
                ub(3.0, 4.0, 1.0)
            },
            4
        );
    }

    Y_UNIT_TEST(Compress) {
        Test(
            {
                ub(1.0, 2.0, 1.0),
                ub(2.0, 3.0, 1.0),
                ub(3.0, 4.0, 1.0)
            },
            {
                ub(1.0, 3.0, 2.0),
                ub(3.0, 4.0, 1.0)
            },
            3
        );
    }
}

Y_UNIT_TEST_SUITE(TRawBucketLimitCounterTest) {

    Y_UNIT_TEST(OneByOneIntervals) {
        TUgramBuckets buckets {
            ub(1.0, 2.0, 1.0),
            ub(2.0, 3.0, 1.0),
            ub(3.0, 4.0, 1.0)
        };
        size_t limit {3};
        TRawBucketLimitCounter counter {buckets, limit};
        UNIT_ASSERT(counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 4);
        counter.CloseEndedBucketWasMerged();
        UNIT_ASSERT(!counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 3);
    }

    Y_UNIT_TEST(SparseIntervals) {
        TUgramBuckets buckets {
            ub(1.0, 2.0, 1.0),
            ub(3.0, 4.0, 1.0)
        };
        size_t limit {3};
        TRawBucketLimitCounter counter {buckets, limit};
        UNIT_ASSERT(counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 4);
        counter.OpenEndedBucketWasMerged();
        UNIT_ASSERT(!counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 2);
    }

    Y_UNIT_TEST(Points) {
        TUgramBuckets buckets {
            ub(1.0, 1.0, 1.0),
            ub(2.0, 2.0, 1.0)
        };
        size_t limit {3};
        TRawBucketLimitCounter counter {buckets, limit};
        UNIT_ASSERT(counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 4);
        counter.OpenEndedBucketWasMerged();
        UNIT_ASSERT(!counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 2);
    }

    Y_UNIT_TEST(PointOnLowerBound) {
        TUgramBuckets buckets {
                ub(1.0, 1.0, 1.0),
                ub(1.0, 2.0, 1.0),
                ub(2.0, 3.0, 1.0)
        };
        size_t limit {3};
        TRawBucketLimitCounter counter {buckets, limit};
        UNIT_ASSERT(counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 4);
        counter.CloseEndedBucketWasMerged();
        UNIT_ASSERT(!counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 3);
    }

    Y_UNIT_TEST(PointOnUpperBound) {
        TUgramBuckets buckets {
                ub(1.0, 2.0, 1.0),
                ub(2.0, 3.0, 1.0),
                ub(3.0, 3.0, 1.0)
        };
        size_t limit {3};
        TRawBucketLimitCounter counter {buckets, limit};
        UNIT_ASSERT(counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 4);
        counter.CloseEndedBucketWasMerged();
        UNIT_ASSERT(!counter.ShouldWeCompress());
        UNIT_ASSERT_VALUES_EQUAL(counter.GetCounterValue(), 3);
    }
}
