#include "ugram.h"

#include <util/string/cast.h>

#include <util/generic/algorithm.h>
#include <util/generic/ymath.h>

using namespace NZoom::NHgram;

namespace {

    inline double CheckBucket(const TUgramBucket& bucket, const double currentBound) {
        const double lower = bucket.LowerBound;
        const double upper = bucket.UpperBound;
        if (!(IsFinite(lower) && IsFinite(upper))) {
            ythrow TUgramError() << "WrongBucketBound";
        }
        if (lower > upper) {
            ythrow TUgramError() << "WrongBucketBound";
        }
        if (lower < 0.0) {
            ythrow TUgramError() << "WrongBucketBound";
        }
        if (bucket.Weight < 0.0 || !IsFinite(bucket.Weight)) {
            ythrow TUgramError() << "WrongWeight: " << ToString(bucket.Weight);
        }
        if (lower < currentBound) {
            ythrow TUgramError() << "WrongBucketBound: " << ToString(lower) << " vs " << ToString(currentBound);
        }
        return upper;
    }

    TUgramBuckets MergeUgrams(const TUgramBuckets& mine, const TUgramBuckets& other) {

        struct TUgramMerger {
            TUgramBuckets Buckets;
            using value_type = TUgramBucket; //compatibility

            TUgramMerger(size_t reserveSize) {
                Buckets.reserve(reserveSize);
            }

            void push_back(const TUgramBucket& bucket) {
                if (Buckets.empty()) {
                    Buckets.push_back(bucket);
                    return;
                }
                //10 .. 15 & 10 .. 20 => 10 .. 15 & 15 .. 20
                //10 .. 20 & 10 .. 30 => 10 .. 20 & 20 .. 30
                //10 .. 20 & 15 .. 20 => 10 .. 15 & 15 .. 20
                //10 .. 20 & 15 .. 18 => 10 .. 15 & 15 .. 18 & 18 .. 20
                //10 .. 20 & 15 .. 25 => 10 .. 15 & 15 .. 20 & 20 .. 25
                //
                //10 .. 10 & 10 .. 20 => 10 .. 10 & 10 .. 20
                //10 .. 20 & 15 .. 15 => 10 .. 15 & 15 .. 15 & 15 .. 20

                auto& prev = Buckets.back();
                if (HaveSameBounds(bucket, prev)) {
                    prev.Weight += bucket.Weight;
                    return;
                }
                if (prev.UpperBound <= bucket.LowerBound) {
                    Buckets.push_back(bucket);
                    return;
                }

                const double prevWeight = prev.Weight;
                const double prevUpper = prev.UpperBound;
                const double prevSize = prevUpper - prev.LowerBound;

                const double bucketSize = bucket.UpperBound - bucket.LowerBound;

                if (prev.LowerBound == bucket.LowerBound) {
                    Buckets.pop_back();
                } else {
                    prev.UpperBound = bucket.LowerBound;
                    prev.Weight = prevWeight * (bucket.LowerBound - prev.LowerBound) / prevSize;
                }
                if (prevUpper >= bucket.UpperBound) { // nesting case
                    Buckets.push_back(bucket);
                    if (bucket.LowerBound != bucket.UpperBound) { // bucket is not point - share weight with it
                        Buckets.back().Weight += prevWeight * bucketSize / prevSize;
                    }
                    if (bucket.UpperBound != prevUpper) {
                        Buckets.emplace_back(bucket.UpperBound, prevUpper, prevWeight * (prevUpper - bucket.UpperBound) / prevSize);
                    }
                } else { //intersection case
                    Buckets.emplace_back(bucket.LowerBound, prevUpper, prevWeight * (prevUpper - bucket.LowerBound) / prevSize +
                            bucket.Weight * (prevUpper - bucket.LowerBound) / bucketSize);
                    if (prevUpper != bucket.UpperBound) {
                        Buckets.emplace_back(prevUpper, bucket.UpperBound, bucket.Weight * (bucket.UpperBound - prevUpper) / bucketSize);
                    }
                }
            }
        };

        TUgramMerger merger(mine.size() + other.size());
        std::merge(mine.cbegin(), mine.cend(), other.cbegin(), other.cend(), std::back_inserter(merger));
        return std::move(merger.Buckets);
    }
}


TUgramBuilder::TUgramBuilder(const size_t capacity) {
    Buckets.reserve(capacity);
}

TUgramBuilder::TUgramBuilder(TUgramBuckets&& buckets) {
    Buckets.clear();
    Buckets.reserve(buckets.size());
    for (const auto& bucket: buckets) {
        try {
            CurrentBound = CheckBucket(bucket, CurrentBound);
            Buckets.push_back(bucket);
        } catch(...) {
        }
    }
//    Buckets = std::move(buckets);
}

void TUgramBuilder::Add(const TUgramBucket& bucket) {
    try {
        CurrentBound = CheckBucket(bucket, CurrentBound);
        Buckets.push_back(bucket);
    } catch(...) {
    }
}

TUgram TUgramBuilder::Build() {
    return TUgram(std::move(Buckets));
}


const TStringBuf TUgram::MARKER = TStringBuf("ugram");

TUgram::TUgram() {
}

TUgram::TUgram(TUgramBuckets&& buckets)
    : Buckets(std::move(buckets))
{
}

const TUgramBuckets& TUgram::GetBuckets() const {
    return Buckets;
}

size_t TUgram::Len() const noexcept {
    return Buckets.size();
}

void TUgram::MulFloat(const double value) {
    MulFloatInternal(value, 1.0);
}

void TUgram::MulSlice(const TVector<double>& values) {
    for (const double value: values) {
        MulFloat(value);
    }
}

void TUgram::MulSmallHgram(const TVector<double>& values, const size_t zeros) {
    MulZeros(zeros);
    MulSlice(values);
}

void TUgram::MulNormalHgram(const TNormal& other) {
    MulNormalHgram(other.GetBuckets(), other.GetZerosCount(), other.GetStartPower());
}

void TUgram::MulNormalHgram(const TVector<double>& buckets, const size_t zeros, const i16 startPower) {
    MulZeros(zeros);
    //FIXME Can do better by merging without temporary allocation
    TUgramBuilder builder(buckets.size());
    i32 power = startPower;
    for (const double weight: buckets) {
        TUgramBucket res(TNormal::GetHgramPower(power), TNormal::GetHgramPower(power + 1), weight);
        builder.Add(res);
        ++power;
    }
    MulUgramHgram(builder.Build());
}

void TUgram::MulUgramHgram(const TUgram& other) {
    Buckets = MergeUgrams(Buckets, other.Buckets);
}

void TUgram::MulBuckets(const TUgramBuckets& buckets) {
    Buckets = MergeUgrams(Buckets, buckets);
}

bool TUgram::operator ==(const TUgram& other) const noexcept {
    return Buckets == other.Buckets;
}

void TUgram::Clean() noexcept {
    Buckets.clear();
}

void TUgram::MulZeros(const size_t count) {
    if (!count) {
        return;
    }
    if (!Buckets.empty()) {
        TUgramBucket& bucket = Buckets.front();;
        if (bucket.LowerBound == 0.0 && bucket.UpperBound == 0.0) {
            bucket.Weight += count;
            return;
        }
    }
    Buckets.insert(Buckets.begin(), TUgramBucket::Point(0.0, count));
}

void TUgram::MulFloatInternal(const double value, const double weight) {
    const double nonNegativePoint = Max(value, 0.0);
    const auto it = LowerBound(Buckets.begin(), Buckets.end(), nonNegativePoint);
    if (it == Buckets.end()) {
        Buckets.push_back(TUgramBucket::Point(nonNegativePoint, weight));
        return;
    }
    if (it->UpperBound == nonNegativePoint) {
        if (it->LowerBound == nonNegativePoint) {
            it->Weight += weight;
            return;
        }
        const auto next = it + 1;
        if (next != Buckets.end() && next->UpperBound == nonNegativePoint && next->LowerBound == nonNegativePoint) {
            next->Weight += weight;
            return;
        }
    }
    if (it->LowerBound >= nonNegativePoint) {
        Buckets.insert(it, TUgramBucket::Point(nonNegativePoint, weight));
        return;
    }
    const double prevLowerBound = it->LowerBound;
    const double prevSize = it->UpperBound - prevLowerBound;
    const double prevWeight = it->Weight;

    const auto insertedBucketIt = Buckets.insert(it, 2, TUgramBucket());
    *(insertedBucketIt) = TUgramBucket(prevLowerBound, nonNegativePoint,
            prevWeight * (nonNegativePoint - prevLowerBound) / prevSize);

    *(insertedBucketIt + 1) = TUgramBucket::Point(nonNegativePoint, weight);

    auto& rightBucket = *(insertedBucketIt + 2);
    rightBucket.LowerBound = nonNegativePoint;
    rightBucket.Weight = prevWeight * (rightBucket.UpperBound - nonNegativePoint) / prevSize;
}
