#include "ugram_freezer.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <util/generic/algorithm.h>

using namespace NZoom::NHgram;

namespace {
    struct THgramVisitor : public IHgramStorageCallback {
    public:
        THgramVisitor(TUgramMerger& merger)
            : Merger(merger)
        {}

        void OnStoreSmall(const TVector<double> &values, const size_t zeros) override {
            TUgram ugram{};
            ugram.MulSmallHgram(values, zeros);
            Merger.CheckAndMulBuckets(ugram.GetBuckets());
        }

        void OnStoreNormal(const TVector<double> &values, const size_t zeros, const i16 startPower) override {
            TUgram ugram{};
            ugram.MulNormalHgram(values, zeros, startPower);
            Merger.CheckAndMulBuckets(ugram.GetBuckets());
        }

        void OnStoreUgram(const TUgramBuckets &buckets) override {
            Merger.CheckAndMulBuckets(buckets);
        }

    private:
        TUgramMerger& Merger;
    };

    inline void ParialSpread(const TUgramBucket* from, TUgramBucket* to) {
        if (to->Size() == 0.0) {
            //NOTE(rocco66): point target is not PartialSpread case
            return;
        } else {
            double overlapPart;
            if (from->Size() == 0.0) {
                overlapPart = 1;
            } else {
                double overlapSize = (
                    Min(from->UpperBound, to->UpperBound) - Max(from->LowerBound, to->LowerBound)
                );
                overlapPart = overlapSize / from->Size();
            }
            to->Weight += from->Weight * overlapPart;
        }
    }
}

TUgramBucketsFreezer::TUgramBucketsFreezer(TUgramBuckets frozenBuckets)
    : BucketsBuffer(std::move(frozenBuckets))
{
};

const TUgramBuckets& TUgramBucketsFreezer::Freeze(const TUgramBuckets& bucketsForFreeze) {
    if (bucketsForFreeze.empty()) {
        for (auto& frozenBucket: BucketsBuffer) {
            frozenBucket.Weight = 0.0;
        }
        return BucketsBuffer;
    }
    // NOTE(rocco66): sum of all frozen bucket intervals are equal or bigger than sum of buckets for freeze
    Y_ASSERT(BucketsBuffer);
    auto frozen = BucketsBuffer.begin();
    frozen->Weight = 0.0;
    auto forFreeze = bucketsForFreeze.begin();
    bool getNextFrozenBucket {false};

    do {
        if (getNextFrozenBucket) {
            frozen++;
            if (frozen == BucketsBuffer.end()) {
                break;
            } else {
                frozen->Weight = 0.0;
            }
        }
        if (HaveSameBounds(*forFreeze, *frozen)) {
            frozen->Weight += forFreeze->Weight;
            forFreeze++;
            getNextFrozenBucket = true;
        } else {
            // NOTE(rocco66): case forFreeze < frozen does not exist
            if (forFreeze->LowerBound >= frozen->UpperBound) {
                auto nextFrozen = frozen + 1;
                bool isLastFrozenBucket = nextFrozen == BucketsBuffer.end();
                bool isPointAtTheEnd = (
                    forFreeze->IsPoint() &&
                    forFreeze->LowerBound == frozen->UpperBound
                );
                if (isLastFrozenBucket || (isPointAtTheEnd && HaveDifferentBounds(*forFreeze, *nextFrozen))) {
                    frozen->Weight += forFreeze->Weight;
                    forFreeze++;
                }
                getNextFrozenBucket = true;
            } else {
                // NOTE(rocco66): partail forFreeze weight spread
                ParialSpread(forFreeze, frozen);
                getNextFrozenBucket = forFreeze->UpperBound >= frozen->UpperBound;
                if (forFreeze->UpperBound <= frozen->UpperBound) {
                    forFreeze++;
                }
            }
        }
    } while (forFreeze != bucketsForFreeze.end());

    if (frozen != BucketsBuffer.end()) {
        frozen++;
    }
    for (;frozen != BucketsBuffer.end(); frozen++) {
        frozen->Weight = 0.0;
    }
    return BucketsBuffer;
}

void TUgramMerger::MulHgram(const THgram& value) {
    THgramVisitor visitor(*this);
    value.Store(visitor);
}

void TUgramMerger::CheckAndMulBuckets(const TUgramBuckets& buckets) {
    if (!buckets.empty() && HaveDifferentBounds(UgramMergeResult.GetBuckets(), buckets)) {
        UgramMergeResult.MulBuckets(buckets);
        DifferentBucketsCounter++;
    }
}

bool TUgramMerger::Compress(size_t limit) {
    const auto& originBuckets = UgramMergeResult.GetBuckets();
    auto compressedBuckets = TUgramCompressor::GetInstance().Compress(originBuckets, limit);
    if (compressedBuckets.empty() && !originBuckets.empty()) {
        if (AllOf(originBuckets, [](const auto& bucket) {return bucket.Weight == 0.0;})) {
            // NOTE(rocco66): empty ugram and zero buckets ugram are different things
            // so use big zero-weight bucket
            compressedBuckets = TUgramBuckets{TUgramBucket{
                originBuckets.front().LowerBound,
                originBuckets.back().UpperBound,
                0.0
            }};
        }
    }
    bool frozenUgramWasCompressed = HaveDifferentBounds(originBuckets, compressedBuckets);
    UgramMergeResult = TUgram(std::move(compressedBuckets));
    return frozenUgramWasCompressed;
}

TMaybe<TUgramBucketsFreezer> TUgramMerger::GetFreezer(size_t limit) {
    bool frozenUgramWasCompressed = Compress(limit);
    auto& frozenBuckets = UgramMergeResult.GetBuckets();
    bool someDifferentUgramsWereMerged = DifferentBucketsCounter > 1;
    if (someDifferentUgramsWereMerged || frozenUgramWasCompressed) {
        return {TUgramBucketsFreezer{frozenBuckets}};
    } else {
        return Nothing();
    }
}
