#include "ugram_freezer.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NHgram;

Y_UNIT_TEST_SUITE(TZoomUgramFreezeTest) {

    void FreezeTest(TUgramBuckets frozenBuckets, TUgramBuckets bucketsForFreeze, TUgramBuckets result) {
        auto freezer = TUgramBucketsFreezer{frozenBuckets};
        auto fixedBuckets = freezer.Freeze(bucketsForFreeze);
        UNIT_ASSERT_VALUES_EQUAL(fixedBuckets, result);
    }

    Y_UNIT_TEST(SimpleFreeze) {
        FreezeTest(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 2.0),
            },
            {
                TUgramBucket(1.0, 2.0, 5.0),
                TUgramBucket(2.0, 3.0, 9.0),
            },
            {
                TUgramBucket(1.0, 2.0, 5.0),
                TUgramBucket(2.0, 3.0, 9.0),
            }
        );
    }

    Y_UNIT_TEST(ExtraFrozenBuckets) {
        FreezeTest(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 1.0),
                TUgramBucket(3.0, 4.0, 1.0),
                TUgramBucket(5.0, 6.0, 1.0),
            },
            {
                TUgramBucket(2.0, 3.0, 10.0),
            },
            {
                TUgramBucket(1.0, 2.0, 0.0),
                TUgramBucket(2.0, 3.0, 10.0),
                TUgramBucket(3.0, 4.0, 0.0),
                TUgramBucket(5.0, 6.0, 0.0),
            }
        );
    }

    Y_UNIT_TEST(ShiftedBuckets) {
        FreezeTest(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 1.0),
                TUgramBucket(3.0, 4.0, 1.0),
            },
            {
                TUgramBucket(1.5, 2.5, 10.0),
                TUgramBucket(2.5, 3.5, 20.0),
            },
            {
                TUgramBucket(1.0, 2.0, 5.0),
                TUgramBucket(2.0, 3.0, 15.0),
                TUgramBucket(3.0, 4.0, 10.0),
            }
        );
    }

    Y_UNIT_TEST(PointsBuckets) {
        FreezeTest(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.5, 1.0),
            },
            {
                TUgramBucket(1.5, 2.5, 20.0),
                TUgramBucket(2.5, 2.5, 3.0),
                TUgramBucket(3.0, 3.0, 5.0),
            },
            {
                TUgramBucket(1.0, 2.0, 10.0),
                TUgramBucket(2.0, 2.0, 0.0),
                TUgramBucket(2.0, 3.5, 18.0),
            }
        );
    }

    Y_UNIT_TEST(PointsOnBorder) {
        FreezeTest(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 1.0),
                TUgramBucket(4.0, 5.0, 1.0),
            },
            {
                TUgramBucket(1.0, 1.0, 5.0),
                TUgramBucket(2.0, 2.0, 5.0),
                TUgramBucket(3.0, 3.0, 5.0),
                TUgramBucket(4.0, 4.0, 5.0),
                TUgramBucket(5.0, 5.0, 5.0),
            },
            {
                TUgramBucket(1.0, 2.0, 5.0),
                TUgramBucket(2.0, 2.0, 5.0),
                TUgramBucket(2.0, 3.0, 5.0),
                TUgramBucket(4.0, 5.0, 10.0),
            }
        );
    }

    Y_UNIT_TEST(SumTwoOriginalBucketsWhileFreeze) {
        FreezeTest(
            {
                TUgramBucket{1.0, 3.0, 0},
                TUgramBucket{4.0, 5.0, 0},
            },
            {
                TUgramBucket{1.0, 2.0, 10},
                TUgramBucket{2.0, 3.0, 20},
                TUgramBucket{4.0, 4.0, 35},
                TUgramBucket{4.0, 5.0, 45},
            },
            {
                TUgramBucket{1.0, 3.0, 30},
                TUgramBucket{4.0, 5.0, 80},
            }
        );
    }

    Y_UNIT_TEST(EmptyBucketsAndEmptyTemplate) {
        FreezeTest(
            {
            },
            {
            },
            {
            }
        );
    }

    Y_UNIT_TEST(EmptyBucketsAndNonEmptyTemplate) {
        FreezeTest(
            {
                {1, 2, 1},
                {2, 3, 1}
            },
            {
            },
            {
                {1, 2, 0},
                {2, 3, 0}
            }
        );
    }

    Y_UNIT_TEST(WrongWeightSpreadFromProd) {
        // GOLOVAN-6608
        FreezeTest(
            {
                {38.44335938, 57.66503906, 0},
                {57.66503906, 86.49755859, 0},
                {86.49755859, 129.7463379, 0},
                {129.7463379, 194.6195068, 0},
                {194.6195068, 291.9292603, 0},
                {291.9292603, 437.8938904, 0}
            },
            {
                {194.6195068, 291.9292603, 25384}
            },
            {
                {38.44335938, 57.66503906, 0},
                {57.66503906, 86.49755859, 0},
                {86.49755859, 129.7463379, 0},
                {129.7463379, 194.6195068, 0},
                {194.6195068, 291.9292603, 25384},
                {291.9292603, 437.8938904, 0}
            }
        );
    }

    Y_UNIT_TEST(SameLastBucketsCrashCase) {
        FreezeTest(
            {
                {0, 26, 2},
                {26, 35, 0.06716417914477612},
                {35, 55, 0.14925373134328357},
                {55, 55, 0},
                {55, 162, 0.91691542288557215},
                {162, 162, 0},
                {162, 165, 0.20000000000000001},
                {165, 165, 0},
                {165, 176, 0.67697594501718206},
                {176, 186, 0.10309278350515463},
                {186, 191, 0.051546391752577317},
                {191, 191, 0},
                {191, 199, 0.08247422680412371},
                {199, 199, 0},
                {199, 205, 0.061855670103092786},
                {205, 205, 0},
                {205, 215, 0.10309278350515463},
                {215, 215, 0},
                {215, 231, 0.16494845360824742},
                {231, 231, 0},
                {231, 247, 0.16494845360824742},
                {247, 247, 0},
                {247, 266, 0.19587628865979381},
                {266, 266, 0},
                {266, 271, 0.051546391752577317},
                {271, 271, 0},
                {271, 281, 0.088570147915732866},
                {281, 281, 0},
                {281, 296, 0.13043478260869565},
                {296, 296, 0},
                {296, 325, 0.25217391304347825},
                {325, 376, 0.44347826086956521},
                {376, 376, 0},
                {376, 386, 0.086956521739130432},
                {386, 386, 0},
                {386, 390, 0.19619565217391305},
                {390, 390, 0},
                {390, 395, 0.3125},
                {395, 395, 0},
                {395, 416, 0.67105263157894735},
                {416, 416, 0},
                {416, 436, 0.26315789473684209},
                {436, 436, 0},
                {436, 470, 0.44736842105263158},
                {470, 470, 0},
                {470, 486, 0.4366028708133971},
                {486, 486, 0},
                {486, 500, 0.63636363636363635},
                {500, 501, 0.045454545454545456}
            },
            {
                {15, 24, 1},
                {24, 26, 1},
                {26, 160, 1},
                {160, 175, 1},
                {175, 272, 1},
                {272, 387, 1},
                {387, 403, 1},
                {403, 479, 1},
                {479, 501, 1},
                {501, 501, 1}
            },
            {
                {0, 26, 2},
                {26, 35, 0.067164179104477612},
                {35, 55, 0.14925373134328357},
                {55, 55, 0},
                {55, 162, 0.91691542288557215},
                {162, 162, 0},
                {162, 165, 0.2},
                {165, 165, 0},
                {165, 176, 0.67697594501718206},
                {176, 186, 0.10309278350515463},
                {186, 191, 0.051546391752577317},
                {191, 191, 0},
                {191, 199, 0.08247422680412371},
                {199, 199, 0},
                {199, 205, 0.061855670103092786},
                {205, 205, 0},
                {205, 215, 0.10309278350515463},
                {215, 215, 0},
                {215, 231, 0.16494845360824742},
                {231, 231, 0},
                {231, 247, 0.16494845360824742},
                {247, 247, 0},
                {247, 266, 0.19587628865979381},
                {266, 266, 0},
                {266, 271, 0.051546391752577317},
                {271, 271, 0},
                {271, 281, 0.088570147915732866},
                {281, 281, 0},
                {281, 296, 0.13043478260869565},
                {296, 296, 0},
                {296, 325, 0.25217391304347825},
                {325, 376, 0.44347826086956521},
                {376, 376, 0},
                {376, 386, 0.086956521739130432},
                {386, 386, 0},
                {386, 390, 0.19619565217391305},
                {390, 390, 0},
                {390, 395, 0.3125},
                {395, 395, 0},
                {395, 416, 0.67105263157894735},
                {416, 416, 0},
                {416, 436, 0.26315789473684209},
                {436, 436, 0},
                {436, 470, 0.44736842105263158},
                {470, 470, 0},
                {470, 486, 0.4366028708133971},
                {486, 486, 0},
                {486, 500, 0.63636363636363635},
                {500, 501, 0.045454545454545456}
            }
        );
    }

    // TODO fuzzy tests
}

Y_UNIT_TEST_SUITE(TZoomUgramMerger) {

    Y_UNIT_TEST(SingleUgramShouldNotBeCompressed) {
        auto merger = TUgramMerger{};
        NZoom::NValue::TValue value {THgram::Ugram(TUgramBuckets{
                TUgramBucket{1.0, 2.0, 10},
                TUgramBucket{2.0, 3.0, 20},
                TUgramBucket{3.0, 4.0, 30},
                TUgramBucket{4.0, 5.0, 40},
        })};

        value.Update(merger);
        auto freezer = merger.GetFreezer(10);
        UNIT_ASSERT(!freezer.Defined());
    }

    Y_UNIT_TEST(UgramsWithSameBoundsShouldNotBeCompressedIfItInLimit) {
        auto merger = TUgramMerger{};
        NZoom::NValue::TValue value {THgram::Ugram(TUgramBuckets{
            TUgramBucket{1.0, 2.0, 10},
            TUgramBucket{2.0, 3.0, 20},
            TUgramBucket{3.0, 4.0, 30},
            TUgramBucket{4.0, 5.0, 40},
        })};

        value.Update(merger);
        value.Update(merger);
        auto freezer = merger.GetFreezer(10);
        UNIT_ASSERT(!freezer.Defined());

        auto freezerWithLimit = merger.GetFreezer(4);
        UNIT_ASSERT(freezerWithLimit.Defined());
        auto frozenBuckets = freezerWithLimit->GetBuckets();
        TUgramBuckets oracleFrozenBuckets {
            TUgramBucket{1.0, 2.0, 10},
            TUgramBucket{2.0, 3.0, 20},
            TUgramBucket{3.0, 5.0, 70}
        };
        UNIT_ASSERT_VALUES_EQUAL(frozenBuckets, oracleFrozenBuckets);

    }

    void MergerTest(
        TUgramBuckets firstUgramBuckets,
        TUgramBuckets secondUgramBuckets,
        ui64 limit,
        const TUgramBuckets& oracleBucketsForFreeze
    ) {
        auto merger = TUgramMerger{};
        NZoom::NValue::TValue value1 {THgram::Ugram(std::move(firstUgramBuckets))};
        NZoom::NValue::TValue value2 {THgram::Ugram(std::move(secondUgramBuckets))};
        value1.Update(merger);
        value2.Update(merger);
        auto freezer = merger.GetFreezer(limit);
        UNIT_ASSERT(freezer.Defined());
        auto frozenBuckets = freezer->GetBuckets();
        UNIT_ASSERT_VALUES_EQUAL(frozenBuckets, oracleBucketsForFreeze);
    }

    Y_UNIT_TEST(SimpleUsage) {
        MergerTest(
            {
                TUgramBucket{1.0, 2.0, 1},
                TUgramBucket{2.0, 3.0, 1}
            },
            {
                TUgramBucket{3.0, 4.0, 1},
                TUgramBucket{4.0, 5.0, 1}
            },
            3,
            {
                TUgramBucket{1.0, 4.0, 3},
                TUgramBucket{4.0, 5.0, 1},
            }
        );
    }

    Y_UNIT_TEST(ZeroWeightSeries) {
        MergerTest(
            {
                TUgramBucket{1.0, 2.0, 0},
                TUgramBucket{2.0, 3.0, 0}
            },
            {
                TUgramBucket{3.0, 4.0, 0},
                TUgramBucket{4.0, 5.0, 0}
            },
            3,
            {
                TUgramBucket{1.0, 5.0, 0}
            }
        );
    }

    Y_UNIT_TEST(ConsiderAllUgramsBuckets) {
        auto merger = TUgramMerger{};
        for (double i = 1; i < 10; i += 2) {
            NZoom::NValue::TValue value {THgram::Ugram({
                TUgramBucket{i, i + 1, 1},
                TUgramBucket{i + 1, i + 2, 1}
            })};
            value.Update(merger);
        }
        auto freezer = merger.GetFreezer(5);
        UNIT_ASSERT(freezer.Defined());
        auto frozenBuckets = freezer->GetBuckets();
        TUgramBuckets oracleBuckets {
            TUgramBucket{1.0, 8.0, 7},
            TUgramBucket{8.0, 9.0, 1},
            TUgramBucket{9.0, 10.0, 1},
            TUgramBucket{10.0, 11.0, 1}
        };
        UNIT_ASSERT_VALUES_EQUAL(frozenBuckets, oracleBuckets);
    }
}
