#include "hgram.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NHgram;

Y_UNIT_TEST_SUITE(TZoomTUgramTest) {

    Y_UNIT_TEST(AddSingleValue) {
        THgram ugram = THgram::EmptyUgram();
        ugram.Mul(THgramMultiplicand(1.0));
        ugram.Mul(THgramMultiplicand(2.0));

        THgram target = THgram::Ugram({TUgramBucket(1.0, 1.0, 1.0), TUgramBucket(2.0, 2.0, 1.0)});
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(Clean) {
        THgram ugram = THgram::EmptyUgram();
        ugram.Mul(THgramMultiplicand(0.0));
        ugram.Clean();
        UNIT_ASSERT_EQUAL(ugram, THgram::EmptyUgram());
    }

    Y_UNIT_TEST(AddPointIntoBucket) {
        THgram ugram = THgram::Ugram({TUgramBucket(1.0, 3.0, 4.0)});
        ugram.Mul(THgramMultiplicand(2.0));

        THgram target = THgram::Ugram({
            TUgramBucket(1.0, 2.0, 2.0),
            TUgramBucket(2.0, 2.0, 1.0),
            TUgramBucket(2.0, 3.0, 2.0)
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(AddBucketOverPoints) {
        THgram ugram = THgram::Ugram({
            TUgramBucket(2.0, 2.0, 10.0),
            TUgramBucket(4.0, 4.0, 10.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Ugram({
            TUgramBucket(1.0, 5.0, 4.0)
        })));
        THgram target = THgram::Ugram({
            TUgramBucket(1.0, 2.0, 1.0),
            TUgramBucket(2.0, 2.0, 10.0),
            TUgramBucket(2.0, 4.0, 2.0),
            TUgramBucket(4.0, 4.0, 10.0),
            TUgramBucket(4.0, 5.0, 1.0)
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(MulUgramAndSmallHgram) {
        THgram ugram = THgram::Ugram({
            TUgramBucket(1.0, 3.0, 2.0),
            TUgramBucket(4.0, 4.0, 10.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Small({2.0, 4.0}, 5)));
        THgram target = THgram::Ugram({
            TUgramBucket(0.0, 0.0, 5.0),
            TUgramBucket(1.0, 2.0, 1.0),
            TUgramBucket(2.0, 2.0, 1.0),
            TUgramBucket(2.0, 3.0, 1.0),
            TUgramBucket(4.0, 4.0, 11.0) // 10.0 + 1.0
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(MulSmallWithUgramHgram) {
        THgram small = THgram::Small({2.0, 4.0}, 5);
        small.Mul(THgramMultiplicand(THgram::Ugram({
            TUgramBucket(1.0, 3.0, 2.0),
            TUgramBucket(4.0, 4.0, 10.0)
        })));
        THgram target = THgram::Ugram({
            TUgramBucket(0.0, 0.0, 5.0),
            TUgramBucket(1.0, 2.0, 1.0),
            TUgramBucket(2.0, 2.0, 1.0),
            TUgramBucket(2.0, 3.0, 1.0),
            TUgramBucket(4.0, 4.0, 11.0) // 10.0 + 1.0
        });
        UNIT_ASSERT_EQUAL(small, target);
    }

    Y_UNIT_TEST(MulUgramWithSlice) {
        THgram ugram = THgram::Ugram({
            TUgramBucket(1.0, 3.0, 2.0),
            TUgramBucket(4.0, 4.0, 10.0)
        });
        ugram.MulSlice(TVector<double>({2.0, 4.0}));
        THgram target = THgram::Ugram({
            TUgramBucket(1.0, 2.0, 1.0),
            TUgramBucket(2.0, 2.0, 1.0),
            TUgramBucket(2.0, 3.0, 1.0),
            TUgramBucket(4.0, 4.0, 11.0) // 10.0 + 1.0
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(MulUgramAndNormalHgram) {
        THgram ugram = THgram::Ugram({
            TUgramBucket(1.25, 1.75, 1.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Normal({2.0, 9.0}, 1, 0))); // NOTE(rocco66): real bounds are 1.0, 1.5, 2.25
        THgram target = THgram::Ugram({
            TUgramBucket(0.0,  0.0,  1.0),
            TUgramBucket(1.0,  1.25, 1.0), // 0.0+(2.0*(1.25-1.0)/(1.5-1.0))
            TUgramBucket(1.25, 1.5,  1.5), // 1.0*(1.5-1.25)/(1.75-1.25)+(2.0*(1.5-1.25)/(1.5-1.0))
            TUgramBucket(1.5,  1.75, 3.5), // 1.0*(1.75-1.5)/(1.75-1.25)+(9.0*(1.75-1.5)/(2.25-1.5))
            TUgramBucket(1.75, 2.25, 6.0) // 0.0+(9.0*(2.25-1.75)/(2.25-1.5))
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(UgramFromSmall) {
        auto empty_ugram = THgramUgram::FromSmall({}, 0);
        auto empty_target = THgramUgram();
        UNIT_ASSERT_EQUAL(*empty_ugram, empty_target);

        auto ugram = THgramUgram::FromSmall({1, 1, 10, 10, 10}, 4);
        auto target = THgramUgram({
            TUgramBucket(0.0,  0.0,  4.0),
            TUgramBucket(1.0,  1.0,  2.0),
            TUgramBucket(10.0,  10.0,  3.0),
        });
        UNIT_ASSERT_EQUAL(*ugram, target);
    }

    Y_UNIT_TEST(UgramNormalSmall) {
        auto ugram = THgramUgram::FromNormal({1, 1}, 0, 1);
        auto target = THgramUgram({
            TUgramBucket(1.5, 2.25, 1.0),
            TUgramBucket(2.25, 3.375, 1.0),
        });
        UNIT_ASSERT_EQUAL(*ugram, target);
    }

    Y_UNIT_TEST(IntersectedBucketsUgramMul) {
        THgram ugram = THgram::Ugram({
            TUgramBucket(1.0, 3.0, 2.0),
            TUgramBucket(5.0, 7.0, 2.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Ugram({
            TUgramBucket(2.0, 6.0, 4.0)
        })));
        THgram target = THgram::Ugram({
            TUgramBucket(1.0, 2.0, 1.0), // 1.0 + 0.0
            TUgramBucket(2.0, 3.0, 2.0), // 1.0 + 1.0
            TUgramBucket(3.0, 5.0, 2.0), // 0.0 + 2.0
            TUgramBucket(5.0, 6.0, 2.0), // 1.0 + 1.0
            TUgramBucket(6.0, 7.0, 1.0) // 1.0 + 0.0
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(MulUgramWithSamePoints) {
        THgram ugram = THgram::Ugram({
            TUgramBucket(1.0, 1.0, 1.0),
            TUgramBucket(2.0, 2.0, 3.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Ugram({
            TUgramBucket(2.0, 2.0, 2.0),
            TUgramBucket(4.0, 4.0, 1.0)
        })));
        THgram target = THgram::Ugram({
            TUgramBucket(1.0, 1.0, 1.0), // 1.0 + 0.0
            TUgramBucket(2.0, 2.0, 5.0), // 3.0 + 2.0
            TUgramBucket(4.0, 4.0, 1.0) // 0.0 + 1.0
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(MulUgramWithRepeatedPoints) {
        THgram ugram = THgram::Ugram({
            TUgramBucket(1.0, 5.0, 4.0)
        });
        ugram.Mul(THgramMultiplicand(2.0));
        ugram.Mul(THgramMultiplicand(3.0));
        ugram.Mul(THgramMultiplicand(2.0));
        THgram target = THgram::Ugram({
            TUgramBucket(1.0, 2.0, 1.0),
            TUgramBucket(2.0, 2.0, 2.0),
            TUgramBucket(2.0, 3.0, 1.0),
            TUgramBucket(3.0, 3.0, 1.0),
            TUgramBucket(3.0, 5.0, 2.0)
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(AccEmptyUgram) {
        THgram ugram = THgram::EmptyUgram();
        ugram.Mul(THgramMultiplicand(THgram::Default()));
        THgram target = THgram::Ugram({});
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(AccUgramWithZeroes) {
        THgram ugram = THgram::EmptyUgram();
        ugram.Mul(THgramMultiplicand(THgram::Small({}, 3)));
        ugram.Mul(THgramMultiplicand(THgram::Small({}, 2)));
        THgram target = THgram::Ugram({TUgramBucket(0.0, 0.0, 5.0)});
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(MulUgramWithSameBuckets) {
        THgram ugram = THgram::Ugram({
            TUgramBucket(1.0, 2.0, 1.0),
            TUgramBucket(3.0, 4.0, 3.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Ugram({
            TUgramBucket(3.0, 4.0, 2.0),
            TUgramBucket(5.0, 6.0, 1.0)
        })));
        THgram target = THgram::Ugram({
            TUgramBucket(1.0, 2.0, 1.0), // 1.0 + 0.0
            TUgramBucket(3.0, 4.0, 5.0), // 3.0 + 2.0
            TUgramBucket(5.0, 6.0, 1.0) // 0.0 + 1.0
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

// Tests from python https://bb.yandex-team.ru/projects/SEARCH_INFRA/repos/yasm/pull-requests/6271
    Y_UNIT_TEST(Accumulate1) {
        // Same borders
        THgram ugram = THgram::Ugram({
            TUgramBucket(10.0, 20.0, 2.0),
            TUgramBucket(20.0, 50.0, 6.0),
            TUgramBucket(50.0, 60.0, 1.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Ugram({
            TUgramBucket(5.0, 10.0, 1.0),
            TUgramBucket(10.0, 15.0, 3.0),
            TUgramBucket(15.0, 30.0, 3.0),
            TUgramBucket(30.0, 60.0, 3.0)
        })));
        THgram target = THgram::Ugram({
            TUgramBucket(5.0, 10.0, 0.0 + 1.0),
            TUgramBucket(10.0, 15.0, 1.0 + 3.0),
            TUgramBucket(15.0, 20.0, 1.0 + 1.0),
            TUgramBucket(20.0, 30.0, 2.0 + 2.0),
            TUgramBucket(30.0, 50.0, 4.0 + 2.0),
            TUgramBucket(50.0, 60.0, 1.0 + 1.0)
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(Accumulate2) {
        // Buckets nesting
        THgram ugram = THgram::Ugram({
            TUgramBucket(10.0, 20.0, 2.0),
            TUgramBucket(20.0, 30.0, 6.0),
            TUgramBucket(30.0, 50.0, 12.0),
            TUgramBucket(50.0, 50.0, 1.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Ugram({
            TUgramBucket(10.0, 30.0, 8.0),
            TUgramBucket(30.0, 40.0, 3.0),
            TUgramBucket(40.0, 45.0, 5.0),
            TUgramBucket(45.0, 50.0, 10.0)
        })));
        THgram target = THgram::Ugram({
            TUgramBucket(10.0, 20.0, 2.0 + 8.0 / 2.0),
            TUgramBucket(20.0, 30.0, 6.0 + 8.0 / 2.0),
            TUgramBucket(30.0, 40.0, 12.0 / 2.0 + 3.0),
            TUgramBucket(40.0, 45.0, 12.0 / 4.0 + 5.0),
            TUgramBucket(45.0, 50.0, 12.0 / 4.0 + 10.0),
            TUgramBucket(50.0, 50.0, 1.0 + 0.0)
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(Accumulate3) {
        // Nonintersecting borders
        THgram ugram = THgram::Ugram({
            TUgramBucket(10.0, 20.0, 2.0),
            TUgramBucket(20.0, 50.0, 6.0),
            TUgramBucket(50.0, 50.0, 1.0)
        });
        ugram.Mul(THgramMultiplicand(THgram::Ugram({
            TUgramBucket(5.0, 10.0, 1.0),
            TUgramBucket(10.0, 15.0, 3.0),
            TUgramBucket(15.0, 30.0, 3.0),
            TUgramBucket(30.0, 60.0, 3.0)
        })));
        THgram target = THgram::Ugram({
            TUgramBucket(5.0, 10.0, 0.0 + 1.0),
            TUgramBucket(10.0, 15.0, 1.0 + 3.0),
            TUgramBucket(15.0, 20.0, 1.0 + 1.0),
            TUgramBucket(20.0, 30.0, 2.0 + 2.0),
            TUgramBucket(30.0, 50.0, 4.0 + 2.0),
            TUgramBucket(50.0, 50.0, 1.0 + 0.0),
            TUgramBucket(50.0, 60.0, 0.0 + 1.0)
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

    Y_UNIT_TEST(AccumulateManySmallBuckets) {
        THgram ugram = THgram::Ugram({});
        TUgramBuckets ugrams({
            TUgramBucket(80.0, 85.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(75.0, 80.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(80.0, 85.0, 1.0),
            TUgramBucket(96.0, 97.0, 1.0),
            TUgramBucket(96.0, 97.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(85.0, 90.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(70.0, 75.0, 1.0),
            TUgramBucket(70.0, 75.0, 1.0),
            TUgramBucket(85.0, 90.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(50.0, 55.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(40.0, 45.0, 1.0),
            TUgramBucket(92.0, 94.0, 1.0),
            TUgramBucket(80.0, 85.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(70.0, 75.0, 1.0),
            TUgramBucket(45.0, 50.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(70.0, 75.0, 1.0),
            TUgramBucket(96.0, 97.0, 1.0),
            TUgramBucket(25.0, 30.0, 1.0),
            TUgramBucket(85.0, 90.0, 1.0),
            TUgramBucket(75.0, 80.0, 1.0),
            TUgramBucket(75.0, 80.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(80.0, 85.0, 1.0),
            TUgramBucket(55.0, 60.0, 1.0),
            TUgramBucket(92.0, 94.0, 1.0),
            TUgramBucket(94.0, 96.0, 1.0),
            TUgramBucket(92.0, 94.0, 1.0),
            TUgramBucket(94.0, 96.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(50.0, 55.0, 1.0),
            TUgramBucket(94.0, 96.0, 1.0),
            TUgramBucket(70.0, 75.0, 1.0),
            TUgramBucket(70.0, 75.0, 1.0),
            TUgramBucket(94.0, 96.0, 1.0),
            TUgramBucket(40.0, 45.0, 1.0),
            TUgramBucket(30.0, 35.0, 1.0),
            TUgramBucket(75.0, 80.0, 1.0),
            TUgramBucket(75.0, 80.0, 1.0),
            TUgramBucket(90.0, 92.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(70.0, 75.0, 1.0),
            TUgramBucket(96.0, 97.0, 1.0),
            TUgramBucket(92.0, 94.0, 1.0),
            TUgramBucket(70.0, 75.0, 1.0),
            TUgramBucket(75.0, 80.0, 1.0),
            TUgramBucket(92.0, 94.0, 1.0),
            TUgramBucket(55.0, 60.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(75.0, 80.0, 1.0),
            TUgramBucket(90.0, 92.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(80.0, 85.0, 1.0),
            TUgramBucket(94.0, 96.0, 1.0),
            TUgramBucket(60.0, 65.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(80.0, 85.0, 1.0),
            TUgramBucket(75.0, 80.0, 1.0),
            TUgramBucket(55.0, 60.0, 1.0),
            TUgramBucket(96.0, 97.0, 1.0),
            TUgramBucket(98.0, 99.0, 1.0),
            TUgramBucket(94.0, 96.0, 1.0),
            TUgramBucket(80.0, 85.0, 1.0),
            TUgramBucket(65.0, 70.0, 1.0),
            TUgramBucket(55.0, 60.0, 1.0),
            TUgramBucket(85.0, 90.0, 1.0),
            TUgramBucket(97.0, 98.0, 1.0),
            TUgramBucket(96.0, 97.0, 1.0),
            TUgramBucket(97.0, 98.0, 1.0),
            TUgramBucket(97.0, 98.0, 1.0),
            TUgramBucket(94.0, 96.0, 1.0),
            TUgramBucket(50.0, 55.0, 1.0)
        });
        for (const auto& oneUgram: ugrams) {
            ugram.Mul(THgramMultiplicand(THgram::Ugram({oneUgram})));
        }
        ugram.Mul(THgramMultiplicand(THgram::Small({73.0}, 0)));
        THgram target = THgram::Ugram({
            TUgramBucket(25.0, 30.0, 1.0),
            TUgramBucket(30.0, 35.0, 1.0),
            TUgramBucket(40.0, 45.0, 2.0),
            TUgramBucket(45.0, 50.0, 1.0),
            TUgramBucket(50.0, 55.0, 3.0),
            TUgramBucket(55.0, 60.0, 4.0),
            TUgramBucket(60.0, 65.0, 10.0),
            TUgramBucket(65.0, 70.0, 12.0),
            TUgramBucket(70.0, 73.0, 4.8),
            TUgramBucket(73.0, 73.0, 1.0),
            TUgramBucket(73.0, 75.0, 3.2),
            TUgramBucket(75.0, 80.0, 8.0),
            TUgramBucket(80.0, 85.0, 7.0),
            TUgramBucket(85.0, 90.0, 4.0),
            TUgramBucket(90.0, 92.0, 2.0),
            TUgramBucket(92.0, 94.0, 5.0),
            TUgramBucket(94.0, 96.0, 7.0),
            TUgramBucket(96.0, 97.0, 6.0),
            TUgramBucket(97.0, 98.0, 3.0),
            TUgramBucket(98.0, 99.0, 1.0)
        });
        UNIT_ASSERT_EQUAL(ugram, target);
    }

}
