#pragma once

#include <infra/yasm/common/points/hgram/hgram.h>

#include <library/cpp/hyperloglog/hyperloglog.h>

namespace NZoom {
    namespace NValue {

        enum class EValueType {
            NONE,
            FLOAT,
            VEC,
            COUNTED_SUM,
            SMALL_HGRAM,
            NORMAL_HGRAM,
            USER_HGRAM,
            HYPER_LOGLOG,
        };

        class IUpdatable {
        public:
            virtual ~IUpdatable() = default;

            virtual void MulNone() = 0;
            virtual void MulFloat(const double value) = 0;
            virtual void MulVec(const TVector<double>& value) = 0;
            virtual void MulCountedSum(const double sum, const ui64 count) = 0;
            virtual void MulHgram(const NHgram::THgram& value) = 0;

            virtual void MulHyperLogLog(const ::THyperLogLog& /*value*/) {
                ythrow yexception() << "Not used. Not implemented";
            }
        };

        class IMultiUpdatable : public IUpdatable, public NHgram::IHgramStorageCallback {
        public:
            void MulHgram(const NHgram::THgram& value) final {
                value.Store(*this);
            };
        };

        class IValue {
        public:
            virtual ~IValue() = default;
            virtual EValueType GetType() const noexcept = 0;
            virtual bool IsDefault() const noexcept = 0;
            virtual void Update(IUpdatable& accumulator) const = 0;
        };

        class TMultiUpdatableAdapter: public IMultiUpdatable {
        public:
            explicit TMultiUpdatableAdapter(IUpdatable& accumulator)
                : Accumulator(accumulator) {
            }
            void MulNone() override {
                Accumulator.MulNone();
            }
            void MulFloat(const double value) override {
                Accumulator.MulFloat(value);
            }
            void MulVec(const TVector<double>& value) override {
                Accumulator.MulVec(value);
            }
            void MulCountedSum(const double sum, const ui64 count) override {
                Accumulator.MulCountedSum(sum, count);
            }
            void OnStoreSmall(const TVector<double>& values, const size_t zeroes) override {
                Accumulator.MulHgram(NHgram::THgram::Small(TVector<double>(values), zeroes));
            }
            void OnStoreSmall(const TArrayRef<const double>& values, size_t zeroes) override {
                Accumulator.MulHgram(NHgram::THgram::Small(TVector<double>(values.begin(), values.end()), zeroes));
            }
            void OnStoreNormal(const TVector<double>& values, const size_t zeroes, const i16 startPower) override {
                Accumulator.MulHgram(NHgram::THgram::Normal(TVector<double>(values), zeroes, startPower));
            }
            void OnStoreNormal(const TArrayRef<const double>& values, size_t zeroes, i16 startPower) override {
                Accumulator.MulHgram(NHgram::THgram::Normal(TVector<double>(values.begin(), values.end()), zeroes, startPower));
            }
            void OnStoreUgram(const NHgram::TUgramBuckets& buckets) override {
                Accumulator.MulHgram(NHgram::THgram::Ugram(NHgram::TUgramBuckets(buckets)));
            }
        private:
            IUpdatable& Accumulator;
        };

        class IFlatValue: public IValue {
        public:
            void Update(IUpdatable& accumulator) const override {
                TMultiUpdatableAdapter adapter(accumulator);
                UpdateFlat(adapter);
            }
            virtual void UpdateFlat(IMultiUpdatable& accumulator) const = 0;
        };

    }
}
