#pragma once

#include <infra/yasm/common/points/value/abstract/value.h>

#include <util/generic/maybe.h>

namespace NZoom {
    namespace NValue {

        class TFloatValue : public IValue {
        protected:
            double Value = 0.0;

        public:
            TFloatValue(const double value);
            virtual ~TFloatValue() = default;
            void Update(IUpdatable& accumulator) const override;

            EValueType GetType() const noexcept override {
                return EValueType::FLOAT;
            }

            bool IsDefault() const noexcept override {
                return !Value;
            }
        };

        class TOptionalFloatValue : public IValue {
        protected:
            TMaybe<double> Value;

        public:
            TOptionalFloatValue();
            TOptionalFloatValue(const double value);
            virtual ~TOptionalFloatValue() = default;
            void Update(IUpdatable& accumulator) const override;

            EValueType GetType() const noexcept override {
                return Value.Defined() ? EValueType::FLOAT : EValueType::NONE;
            }

            bool IsDefault() const noexcept override {
                return Value.Empty();
            }
        };

        class TVecValue : public IValue {
        protected:
            TVector<double> Value;

        public:
            TVecValue(TVector<double> value);
            virtual ~TVecValue() = default;
            void Update(IUpdatable& accumulator) const override;

            EValueType GetType() const noexcept override {
                return EValueType::VEC;
            }

            bool IsDefault() const noexcept override {
                return Value.empty();
            }
        };

        class TNoneValue : public IValue {
        public:
            virtual ~TNoneValue() = default;
            void Update(IUpdatable& accumulator) const override;

            EValueType GetType() const noexcept override {
                return EValueType::NONE;
            }

            static const TNoneValue& GetSingleton();

            bool IsDefault() const noexcept override {
                return true;
            }
        };

        class TCountedSumValue : public IValue {
        protected:
            double Sum = 0.0;
            ui64 Count = 0;
        public:
            TCountedSumValue(const double sum, const ui64 count);
            virtual ~TCountedSumValue() = default;
            void Update(IUpdatable& accumulator) const override;

            EValueType GetType() const noexcept override {
                return EValueType::COUNTED_SUM;
            }

            bool IsDefault() const noexcept override {
                return !Sum && !Count;
            }
        };

        class THgramValue : public IValue {
        protected:
            NZoom::NHgram::THgram Value;

        public:
            THgramValue(NHgram::THgram value);
            virtual ~THgramValue() = default;
            void Update(IUpdatable& accumulator) const override;

            EValueType GetType() const noexcept override {
                using NZoom::NHgram::EHgramType;
                switch (Value.GetType()) {
                    case EHgramType::SMALL: {
                        return EValueType::SMALL_HGRAM;
                    }
                    case EHgramType::NORMAL: {
                        return EValueType::NORMAL_HGRAM;
                    }
                    case EHgramType::USER: {
                        return EValueType::USER_HGRAM;
                    }
                }
            }

            bool IsDefault() const noexcept override {
                return Value.IsDefault();
            }
        };

        class THyperLogLogValue : public IValue {
        public:
            THyperLogLogValue(const ::THyperLogLog& value);
            virtual ~THyperLogLogValue() = default;
            void Update(IUpdatable& accumulator) const override;

            EValueType GetType() const noexcept override {
                return EValueType::HYPER_LOGLOG;
            }

            bool IsDefault() const noexcept override {
                return false;
            }
        };

    }
}
