#include "types.h"
#include "impl.h"

#include <util/string/vector.h>

using namespace NZoom::NValue;

namespace {
    // For test purposes only
    class TComparingUpdatable: public IUpdatable {
    private:
        bool IsNone = false;
        TMaybe<double> FloatMaybe;
        const TVector<double>* VecPtr = nullptr;
        TMaybe<std::pair<ui64, double>> CountedSumMaybe;
        const NZoom::NHgram::THgram* HGramPtr = nullptr;

    public:
        void MulNone() final {
            IsNone = true;
        }

        void MulFloat(const double value) final {
            FloatMaybe = value;
        }

        void MulVec(const TVector<double>& value) final {
            VecPtr = &value;
        }

        void MulCountedSum(const double sum, const ui64 count) final {
            CountedSumMaybe = std::make_pair(count, sum);
        }

        void MulHgram(const NZoom::NHgram::THgram& value) final {
            HGramPtr = &value;
        }

        bool operator==(const TComparingUpdatable& other) const {
            return (IsNone == other.IsNone) &&
                (FloatMaybe == other.FloatMaybe) &&
                (VecPtr == other.VecPtr ||
                    (VecPtr != nullptr && other.VecPtr != nullptr && *VecPtr == *other.VecPtr)
                ) &&
                (CountedSumMaybe == other.CountedSumMaybe) &&
                (HGramPtr == other.HGramPtr ||
                    (HGramPtr != nullptr && other.HGramPtr != nullptr && *HGramPtr == *other.HGramPtr)
                );
        }
    };

    THolder<IValue> OwnValue(const TValueRef& value) {
        TValueCopier copier;
        value.Update(copier);
        return copier.ExtractRawValue();
    }

}

TValueRef::TValueRef()
    : Value(&TNoneValue::GetSingleton())
{
}

bool TValueRef::operator==(const TValueRef& other) const noexcept {
    if (Value == other.Value)
        return true;
    TComparingUpdatable selfUpd;
    TComparingUpdatable otherUpd;
    Update(selfUpd);
    other.Update(otherUpd);
    return selfUpd == otherUpd;
}

void TMetricManager::MulNone() {
    ++NonePoints;
}

void TMetricManager::MulFloat(const double value) {
    ++TotalPoints;
    if (value < 0.0) {
        ++NegativePoints;
    }
}

void TMetricManager::MulVec(const TVector<double>& value) {
    ++TotalLists;
    ListsLen += value.size();
}


void TMetricManager::MulCountedSum(const double sum, const ui64 /*count*/) {
    ++TotalPoints;
    if (sum < 0.0) {
        ++NegativePoints;
    }
}

void TMetricManager::MulHgram(const NHgram::THgram& value) {
    if (value.IsUgram()) {
        ++TotalUgrams;
        UgramsLen += value.Len();
    } else {
        ++TotalHgrams;
        HgramsLen += value.Len();
    }
}

void TMetricManager::MulHyperLogLog(const ::THyperLogLog& /*value*/) {
    ++TotalPoints;
}


TValue::TValue()
    : Value(new TNoneValue())
{
}

TValue::TValue(const double sum, const ui64 count)
    : Value(new TCountedSumValue(sum, count))
{
}

TValue::TValue(NHgram::THgram value)
    : Value(new THgramValue(std::move(value)))
{
}

TValue::TValue(const double value)
    : Value(new TFloatValue(value))
{
}

TValue::TValue(TVector<double> value)
    : Value(new TVecValue(std::move(value)))
{
}

TValue::TValue(const ::THyperLogLog& value)
    : Value(new THyperLogLogValue(value))
{
}

TValue::TValue(const TValueRef& other)
    : Value(OwnValue(other))
{
}

bool TValue::operator==(const TValue& other) const {
    TComparingUpdatable selfUpd;
    TComparingUpdatable otherUpd;
    Update(selfUpd);
    other.Update(otherUpd);
    return selfUpd == otherUpd;
}

void TValueCopier::MulNone() {
    ImplHolder = MakeHolder<TNoneValue>();
}

void TValueCopier::MulFloat(const double value) {
    ImplHolder = MakeHolder<TFloatValue>(value);
}

void TValueCopier::MulVec(const TVector<double>& value) {
    ImplHolder = MakeHolder<TVecValue>(value);
}

void TValueCopier::MulCountedSum(const double sum, const ui64 count) {
    ImplHolder = MakeHolder<TCountedSumValue>(sum, count);
}

void TValueCopier::MulHgram(const NZoom::NHgram::THgram& value) {
    NZoom::NHgram::THgram newValue = NZoom::NHgram::THgram::Default();
    newValue.MulHgram(value);
    ImplHolder = MakeHolder<THgramValue>(std::move(newValue));
}

void TValueCopier::MulHyperLogLog(const ::THyperLogLog& value) {
    ImplHolder = MakeHolder<THyperLogLogValue>(value);
}

TValue TValueCopier::ExtractValue() {
    return TValue(ExtractRawValue());
}

THolder<IValue> TValueCopier::ExtractRawValue() {
    return std::move(ImplHolder);
}

namespace {

    TString Vec2String(const TVector<double>& vec) {
        return JoinStrings(vec.begin(), vec.end(), ", ");
    }

    class TPrintUpdatable : public IUpdatable, public NZoom::NHgram::IHgramStorageCallback {
    private:
        IOutputStream& Stream;

    public:
        explicit TPrintUpdatable(IOutputStream& stream)
            : Stream(stream)
        {
        }

        void MulNone() final {
            Stream.Write("<None>");
        }

        void MulFloat(const double value) final {
            Stream << value;
        }

        void MulVec(const TVector<double>& value) final {
            Stream << '[' << Vec2String(value) <<']';
        }

        void MulCountedSum(const double sum, const ui64 count) final {
            Stream << "(Sum: " << sum << ", Count: " << count << ')';
        }

        void MulHgram(const NZoom::NHgram::THgram& value) final {
            value.Store(*this);
        }

        void OnStoreSmall(const TVector<double>& values, const size_t zeros) final {
            Stream << "(Small: zeroes: " << zeros << ", values: [" << Vec2String(values) << "])";
        }

        void OnStoreNormal(const TVector<double>& values, const size_t zeros, const i16 startPower) final {
            Stream << "(Small: zeroes: " << zeros << ", startPower: " << startPower << ", "
                "values: [" << Vec2String(values) << "])";
        }

        void OnStoreUgram(const NZoom::NHgram::TUgramBuckets& buckets) final {
            Stream << "(Ugram: [";
            bool first = true;
            for (const auto& bucket: buckets) {
                if (first) {
                    Stream << ", ";
                    first = false;
                }
                Stream << '(' << bucket.LowerBound << ", " << bucket.UpperBound << ", " << bucket.Weight <<')';
            }
            Stream << "])";
        }

    };
}

template <>
void Out<TValue>(IOutputStream& stream,
                 TTypeTraits<TValue>::TFuncParam value) {
    TPrintUpdatable printer(stream);
    value.Update(printer);
}
