import argparse
import logging
import logging.handlers
import socket
import time

from tornado import gen, httpserver, ioloop, netutil, web

from infra.yasm.gateway.lib.client.cluster_provider import ClusterType, SolomonGatewayClusterProvider
from infra.yasm.gateway.lib.client.cluster_provider import DEFAULT_GRPC_USER_AGENT
from infra.yasm.gateway.lib.handlers.base import BaseGatewayHandler
from infra.yasm.gateway.lib.handlers.local import ConfListHandler, FunctionsHandler
from infra.yasm.gateway.lib.handlers.meta_alert import MetaAlertHandler
from infra.yasm.gateway.lib.handlers.rt import RtHandler, RtOptions
from infra.yasm.gateway.lib.handlers.series import SeriesHandler, HistSeriesLimits
from infra.yasm.gateway.lib.handlers.top import TopHandler
from infra.yasm.gateway.lib.util import logger as logger_util
from infra.yasm.gateway.lib.util.yasmconf_lite import YasmConfLite
from infra.yasm.gateway.lib.util import common
from infra.yasm.gateway.lib.util.duostat import Duostat

log = logging.getLogger(__name__)

SOLOMON_GATEWAY_INSTANCE_REFRESH_INTERVAL = 60
SOLOMON_GATEWAY_INSTANCE_REFRESH_JITTER = 30


class ReadyHandler(BaseGatewayHandler):
    def initialize(self, unistat, front_id):
        super(ReadyHandler, self).initialize(unistat, front_id)

    @gen.coroutine
    def handle_get(self):
        self.response({"ready": True})


class StatsHandler(BaseGatewayHandler):
    RAW_RESPONSE = True

    def initialize(self, unistat, front_id):
        super(StatsHandler, self).initialize(unistat, front_id)

    @gen.coroutine
    def handle_get(self):
        target = self.get_argument("target", None)

        if target == "solomon":
            payload = self.unistat.solomon_spack()
            self.set_status(200)
            self.setup_headers()
            self.set_header("Content-Type", "application/x-solomon-spack")
            self.write(payload)
            self.finish()
        else:
            self.response(self.unistat.to_json(all_signals=True))


class TimeHandler(BaseGatewayHandler):
    """
    Return current time adjusted to five seconds.
    """

    def initialize(self, unistat, front_id):
        super(TimeHandler, self).initialize(unistat, front_id)

    @gen.coroutine
    def handle_get(self):
        self.response({"time": common.adjust_timestamp(time.time(), 5)})


def init_log(log_file):
    level = logging.INFO
    logging.root.setLevel(level)
    handler = logging.handlers.TimedRotatingFileHandler(
        filename=log_file,
        when="midnight",
        backupCount=5
    )
    handler.setLevel(level)
    handler.setFormatter(logger_util.RequestLoggerFormatter())
    logging.root.addHandler(handler)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--port",
                        dest="http_port",
                        type=int,
                        required=True,
                        help="Http port")

    parser.add_argument("--unistat-port",
                        dest="unistat_port",
                        type=int,
                        help="Unistat http port. Unistat handle will not work if the value is not specified.")

    parser.add_argument("--solomon-gateway-cluster",
                        dest="solomon_gateway_cluster",
                        type=ClusterType,
                        choices=list(ClusterType),
                        default=ClusterType.PRESTABLE,
                        help="Solomon cluster where to look for gateway instances")

    parser.add_argument("--contour",
                        dest="contour",
                        type=str,
                        default="prestable",
                        help="Current yasmgateway contour. Used to set user-agents and similar stuff.")

    parser.add_argument("--hist-max-request-points-total",
                        dest="hist_max_request_points_total",
                        type=int,
                        default=288000,
                        help="max number of points to allow in single hist series request")

    parser.add_argument("--hist-max-request-points-signal",
                        dest="hist_max_request_points_signal",
                        type=int,
                        default=2880,
                        help="max number of points to allow requesting for a single time series")

    parser.add_argument("--hist-max-signals-per-request",
                        dest="hist_max_signals_per_request",
                        type=int,
                        default=200,
                        help="max number of individual signals to allow in a single hist series request")

    parser.add_argument("--rt-signals-limit",
                        dest="rt_signals_limit",
                        type=int,
                        default=1000,
                        help="max number of signal expressions allowed in a single request")

    parser.add_argument("--rt-range-start-offset",
                        dest="rt_range_start_offset",
                        type=int,
                        default=55,
                        help="Start timestamp of /rt range offset in seconds.")

    parser.add_argument("--rt-range-end-offset",
                        dest="rt_range_end_offset",
                        type=int,
                        default=15,
                        help="End timestamp of /rt range offset in seconds.")

    parser.add_argument("--yasm-conf",
                        dest="yasm_conf_path",
                        type=str,
                        required=True,
                        help="path to yasm.conf file")

    parser.add_argument("--log",
                        dest="log_path",
                        type=str,
                        required=True,
                        help="path to log file")

    return parser.parse_args()


def create_and_register_applications(cluster_provider, args, unistat):
    front_id = socket.gethostname()
    yasmconf = YasmConfLite(path=args.yasm_conf_path)
    hist_series_limits = HistSeriesLimits(
        max_request_points_total=args.hist_max_request_points_total,
        max_request_points_signal=args.hist_max_request_points_signal,
        max_signals_per_request=args.hist_max_signals_per_request
    )
    rt_options = RtOptions(
        signals_limit=args.rt_signals_limit,
        range_start_offset=args.rt_range_start_offset,
        range_end_offset=args.rt_range_end_offset
    )

    main_app_handlers_config = [
        (r"/ready/?", ReadyHandler, dict(
            unistat=unistat,
            front_id=front_id
        )),
        (r"/time/?", TimeHandler, dict(
            unistat=unistat,
            front_id=front_id
        )),
        (r"/meta-alert/?", MetaAlertHandler, dict(
            cluster_provider=cluster_provider,
            unistat=unistat,
            front_id=front_id
        )),
        (r"/series/?", SeriesHandler, dict(
            cluster_provider=cluster_provider,
            unistat=unistat,
            front_id=front_id,
            limits=hist_series_limits,
            yasmconf=yasmconf
        )),
        (r"/data/?", SeriesHandler, dict(
            cluster_provider=cluster_provider,
            unistat=unistat,
            front_id=front_id,
            limits=hist_series_limits,
            yasmconf=yasmconf
        )),
        (r"/top/?", TopHandler, dict(
            cluster_provider=cluster_provider,
            unistat=unistat,
            front_id=front_id
        )),
        (r"/rt/?", RtHandler, dict(
            cluster_provider=cluster_provider,
            unistat=unistat,
            front_id=front_id,
            options=rt_options,
            yasmconf=yasmconf
        )),
        (r"/functions/?(.*)", FunctionsHandler, dict(
            unistat=unistat,
            front_id=front_id
        )),
        (r"/conf/conflist/?(.*)", ConfListHandler, dict(
            unistat=unistat,
            front_id=front_id,
            yasmconf=yasmconf
        ))
    ]

    for _, handler_class, _ in main_app_handlers_config:
        handler_class.prepare_unistat_signals(unistat)

    unistat_handler_config = (r"/stats/?", StatsHandler, dict(
        unistat=unistat,
        front_id=front_id
    ))
    if args.unistat_port:
        unistat_app = web.Application(
            [unistat_handler_config],
            log_function=BaseGatewayHandler.log_result
        )
        unistat_app.listen(args.unistat_port)
    app = web.Application(
        main_app_handlers_config,
        log_function=BaseGatewayHandler.log_result
    )
    server = httpserver.HTTPServer(app)
    server.add_sockets(netutil.bind_sockets(args.http_port, reuse_port=True))


def main():
    args = parse_args()
    init_log(args.log_path)
    unistat = Duostat()

    log.info("Load Solomon gateway instances")
    cluster_provider = SolomonGatewayClusterProvider(
        args.solomon_gateway_cluster,
        grpc_user_agent="{}_{}".format(DEFAULT_GRPC_USER_AGENT, args.contour),
        unistat=unistat
    )
    ioloop.IOLoop.current().run_sync(cluster_provider.reload)
    cluster_provider.schedule_reloads(
        ioloop.IOLoop.current(),
        SOLOMON_GATEWAY_INSTANCE_REFRESH_INTERVAL,
        SOLOMON_GATEWAY_INSTANCE_REFRESH_JITTER
    )

    log.info("Start yasmgateway application")
    create_and_register_applications(cluster_provider, args, unistat)
    ioloop.IOLoop.current().start()
