import json
import logging
import time
import uuid

from tornado import web, gen, httputil, escape

from infra.yasm.gateway.lib.util import stat
from infra.yasm.gateway.lib.util.logger import RequestLoggerAdapter
from infra.yasm.gateway.lib.util.common import adjust_timestamp

module_log = logging.getLogger(__name__)


class BaseGatewayHandler(web.RequestHandler):
    FQDN_HTTP_HEADER = "X-Golovan-Host"
    TIMESTAMP_HTTP_HEADER = "X-Golovan-Host-Timestamp"

    UI_USERNAME_COOKIE = 'yandex_login'
    YASMAPI_HOSTNAME_HEADER = 'X-Golovan-Hostname'
    YASMAPI_USERNAME_HEADER = 'X-Golovan-Username'
    REFERRER_HEADER = 'Referer'
    ACCESS_CONTROL_MAX_AGE = '86400'

    ALLOWED_OPTIONS_HEADERS = None  # override to provide more allowed headers for options handler
    RAW_RESPONSE = False  # override to True to avoid wrapping in standard dict
    HANDLE_STAT_NAME = "other"  # override to get detailed stats

    def initialize(self, unistat, front_id, base_log=module_log):
        self.front_id = front_id
        self.unistat = unistat
        self.request_id = str(uuid.uuid4())
        self.log = RequestLoggerAdapter(base_log, self.request_id)

    @classmethod
    def prepare_unistat_signals(cls, unistat):
        unistat.create_histogram(cls._get_request_time_signal_name(), intervals=stat.TIMING_INTERVALS_LOG_1_3)
        unistat.create_histogram(cls._get_request_handle_time_signal_name(), intervals=stat.TIMING_INTERVALS_LOG_1_3)

    def prepare(self):
        self._start_time = time.time()

    def on_finish(self):
        self.unistat.push(self._get_request_time_signal_name(), time.time() - self._start_time)

    @gen.coroutine
    def get(self, *args, **kwargs):
        # common work time measure for all GET requests
        with stat.time_it(self._get_request_handle_time_signal_name(), self.unistat):
            yield self.handle_get(*args, **kwargs)

    @gen.coroutine
    def post(self):
        # common work time measure for all POST requests
        with stat.time_it(self._get_request_handle_time_signal_name(), self.unistat):
            try:
                decoded_body = escape.json_decode(self.request.body)
            except Exception as e:
                self.error("Invalid json: {}".format(e), status_code=400)
            else:
                yield self.handle_post(decoded_body)

    @gen.coroutine
    def options(self):
        self.setup_headers(allowed_headers=self.ALLOWED_OPTIONS_HEADERS)
        self.set_header('Access-Control-Max-Age', self.ACCESS_CONTROL_MAX_AGE)
        self.set_status(204)
        self.finish()

    @gen.coroutine
    def handle_get(self, *args, **kwargs):
        self.error("Method not supported", status_code=405)

    @gen.coroutine
    def handle_post(self):
        self.error("Method not supported", status_code=405)

    def log_exception(self, typ, value, tb):
        """
        Overrides base class method to log with request id. Other logic is copied from the base implementation
        """
        if isinstance(value, web.HTTPError):
            if value.log_message:
                format = "%d %s: " + value.log_message
                args = ([value.status_code, self._request_summary()] + list(value.args))
                self.log.warning(format, *args)
        else:
            self.log.error("Uncaught exception %s\n%r", self._request_summary(), self.request, exc_info=(typ, value, tb))

    @classmethod
    def log_result(cls, handler):
        """
        Log handler result to the handler's own log, instead of tornado's internal one
        """
        cur_log = handler.log if isinstance(handler, cls) else module_log
        if handler.get_status() < 400:
            log_method = cur_log.info
        elif handler.get_status() < 500:
            log_method = cur_log.warning
        else:
            log_method = cur_log.error
        request_time = 1000.0 * handler.request.request_time()
        request_summary = "%s %s (%s)" % (handler.request.method, handler.request.uri, handler.request.remote_ip)
        log_method("%d %s %.2fms", handler.get_status(), request_summary, request_time)

    @classmethod
    def _get_request_time_signal_name(cls):
        return "handlers.{}_request_time".format(cls.HANDLE_STAT_NAME)

    @classmethod
    def _get_request_handle_time_signal_name(cls):
        return "handlers.{}_handle_time".format(cls.HANDLE_STAT_NAME)

    def error(self, message, status_code=None):
        """
        Return given error to the client.
        """
        self._response({"error": message}, status="error", status_code=status_code)

    def write_error(self, status_code, **kwargs):
        self.error(httputil.responses.get(status_code), status_code)

    def response(self, response):
        """
        Return response to the client.

        :type response: dict[str, X]
        """
        self._response(response, status="ok")

    def setup_headers(self, allowed_headers=None):
        """
        :type allowed_headers: list[str]
        """
        self.set_header(self.FQDN_HTTP_HEADER, self.front_id)
        self.set_header(self.TIMESTAMP_HTTP_HEADER, adjust_timestamp(time.time(), 5))
        # setup CORS
        allowed_headers = ['Origin', 'Content-Type', 'Cookie'] + (allowed_headers if allowed_headers else [])
        origin = self.request.headers.get('Origin', '*')
        self.set_header('Access-Control-Allow-Origin', origin)
        self.set_header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
        self.set_header('Access-Control-Allow-Headers', ', '.join(allowed_headers))
        self.set_header('Access-Control-Allow-Credentials', 'true')

    def _response(self, response, status, status_code=None):
        if not response:
            response = {}

        if status_code is not None:
            self.smart_set_status(status_code)

        self.setup_headers()
        self.set_header("Content-Type", "application/json")

        if self.RAW_RESPONSE:
            response_body = response
        else:
            data = {"status": status, "response": response, "fqdn": self.front_id}

            if status == "error":
                data["error"] = response.get("error")

            response_body = json.dumps(data) if self.is_xhr else json.dumps(data, indent=2)

        self.write(response_body)

        self.finish()

    @property
    def is_xhr(self):
        """
        True if the request was triggered via a JavaScript XMLHttpRequest.
        This only works with libraries that support the `X-Requested-With`
        header and set it to "XMLHttpRequest".  Libraries that do that are
        prototype, jQuery and Mochikit and probably some more.

        :rtype: bool
        """
        return self.request.headers.get("x-requested-with", "").lower() == "xmlhttprequest"

    def smart_set_status(self, status_code):
        if status_code == 521:
            self.set_status(status_code, "Web Server Is Down")
        else:
            self.set_status(status_code)
