import logging

from tornado import web

from infra.yasm.gateway.lib.util.common import AnyArgs

from infra.yasm.gateway.lib.handlers.base import BaseGatewayHandler

module_log = logging.getLogger(__name__)


class HistFrontHandler(BaseGatewayHandler):

    FQDN_HTTP_HEADER = "X-HistFront-Fqdn"
    REQUESTS_CONNECTION_CLOSED_SIGNAL_NAME = "hist.requests_connection_closed"

    def initialize(self, unistat, front_id):
        super(HistFrontHandler, self).initialize(unistat, front_id)
        self._cleaned = False

    @classmethod
    def prepare_unistat_signals(cls, unistat):
        super(HistFrontHandler, cls).prepare_unistat_signals(unistat)
        unistat.create_float(cls.REQUESTS_CONNECTION_CLOSED_SIGNAL_NAME)

    def get_int_argument(self, name, default=None):
        try:
            str_int = self.get_argument(name)
        except web.MissingArgumentError:
            return default
        try:
            return int(str_int)
        except (ValueError, TypeError):
            self.error('Wrong argument "{}", it must be int'.format(name), status_code=400)

    def get_query_args(self, allowed_args=AnyArgs):
        if allowed_args is AnyArgs:
            return {
                a: self.get_argument(a) for a
                in self.request.query_arguments
            }
        return {
            a: self.get_argument(a) for a in allowed_args
            if a in self.request.query_arguments
        }

    def on_finish(self):
        super(HistFrontHandler, self).on_finish()
        self.log.info('request finished')

    def on_connection_close(self):
        super(HistFrontHandler, self).on_connection_close()
        self.unistat.push(self.REQUESTS_CONNECTION_CLOSED_SIGNAL_NAME, 1)
        self.log.info('[%s] connection closed by client', self.request_id)
