"""
Handlers that do not require communication with external services.
"""
import json
import re

from tornado import gen, web

from infra.yasm.gateway.lib.handlers.base import BaseGatewayHandler
from infra.yasm.gateway.lib.util.functions import FUNCS


def _recursive_get(data, path):
    if isinstance(data, basestring):
        return data

    if path:
        if not isinstance(data, dict):
            raise web.HTTPError(status_code=400, log_message="Cannot get any field from data that is not a dict")

        if path[0].startswith('list'):
            return data.keys()

        if len(path) > 1:
            return _recursive_get(data.get(path[0], {}), path[1:])
        else:
            if path[0]:
                return data.get(path[0], {})
            else:
                return data


def extract_sub_response(response, path):
    # Get contents of response dict (?) by parsing the request path.
    # This is old logic from yasmcollector that existed since https://st.yandex-team.ru/GOLOVAN-2536
    path = path.strip('/').split('/')
    return _recursive_get(response, path)


class BaseHandler(BaseGatewayHandler):
    RAW_RESPONSE = True

    @gen.coroutine
    def handle_get(self, path):
        full_response = self.make_full_response()
        response = extract_sub_response(full_response, path)

        if response and not isinstance(response, basestring):
            response = json.dumps(response, sort_keys=True)
        self.response(response)

    def make_full_response(self):
        raise Exception("Not implemented")


class ConfListHandler(BaseHandler):
    HANDLE_STAT_NAME = "conflist"

    def initialize(self, unistat, front_id, yasmconf):
        super(ConfListHandler, self).initialize(unistat, front_id)
        self._yasmconf = yasmconf

    def make_full_response(self):
        return self._yasmconf.get_conflist()


class FunctionsHandler(BaseHandler):
    HANDLE_STAT_NAME = "functions"

    @staticmethod
    def _get_int(args, opt):
        try:
            return int(args[opt][0])
        except (KeyError, IndexError, ValueError):
            return None

    @staticmethod
    def _filter_functions(functions, args):
        try:
            match = re.compile(args['function_pattern'][0]).match
        except (KeyError, IndexError):
            return functions
        else:
            return [f for f in functions if match(f['name'])]

    @classmethod
    def _slice_result(cls, result, args):
        try:
            if args['sorted'][0] == 'desc':
                # already sorted ascendant
                result = list(reversed(result))
        except (KeyError, IndexError):
            pass

        limit = cls._get_int(args, 'limit')

        if not limit:
            return result

        offset = cls._get_int(args, 'offset') or 0
        return result[offset:offset + limit]

    @classmethod
    def make_full_response_for_args(cls, arguments):
        filtered = cls._filter_functions(FUNCS, arguments)
        total = len(filtered)
        sliced = cls._slice_result(filtered, arguments)
        return {'result': sliced, 'total': total}

    def make_full_response(self):
        return self.make_full_response_for_args(self.request.arguments)
