import itertools

from collections import namedtuple
from tornado import gen
from tornado import web

from solomon.services.gateway.api import yasm_gateway_service_pb2 as gateway_service_proto
from solomon.services.gateway.api import yasm_gateway_service_pb2_grpc as gateway_service_grpc

from infra.yasm.gateway.lib.client.requester import MultiAttemptRpc, ReadyHostAttemptSequence
from infra.yasm.gateway.lib.client.requester import SOLOMON_GATEWAY_CLUSTER_ATTEMPTS
from infra.yasm.gateway.lib.client.requester import SOLOMON_GATEWAY_IN_CLUSTER_REQUEST_ATTEMPTS
from infra.yasm.gateway.lib.handlers.base import BaseGatewayHandler

STATUS_MAPPING = {
    gateway_service_proto.AlertStatusResponse.AlertStatus.OK: "ok",
    gateway_service_proto.AlertStatusResponse.AlertStatus.WARN: "warn",
    gateway_service_proto.AlertStatusResponse.AlertStatus.ALARM: "crit"
}
META_ALERT_REQUEST_TIMEOUT = 5

AlertID = namedtuple("AlertID", ["name", "itype"])


def parse_meta_alert_input(request_body):
    requested_alerts = []
    try:
        for requested_alert in request_body["meta-alert"]:
            name = requested_alert["name"]
            itype = requested_alert["itype"]
            if not isinstance(name, basestring) or not isinstance(name, basestring):
                raise web.HTTPError(status_code=400, log_message="'name' or 'itype' is not a string")
            requested_alerts.append(AlertID(name, itype))
    except web.HTTPError:
        raise
    except Exception as e:
        raise web.HTTPError(status_code=400, log_message=repr(e))
    return requested_alerts


class MetaAlertHandler(BaseGatewayHandler):
    HANDLE_STAT_NAME = "meta_alert"
    RAW_RESPONSE = True

    def initialize(self, cluster_provider, unistat, front_id):
        super(MetaAlertHandler, self).initialize(unistat, front_id)
        self._cluster_provider = cluster_provider

    @gen.coroutine
    def handle_post(self, parsed_body):
        requested_alert_ids = parse_meta_alert_input(parsed_body)

        proto_request = gateway_service_proto.AlertStatusRequest()
        for requested_alert in requested_alert_ids:
            alert_key = proto_request.alert_keys.add()
            alert_key.name = requested_alert.name
            alert_key.itype = requested_alert.itype

        call = MultiAttemptRpc(
            ReadyHostAttemptSequence.make_random(
                self._cluster_provider.get_cluster_hosts(),
                SOLOMON_GATEWAY_CLUSTER_ATTEMPTS,
                SOLOMON_GATEWAY_IN_CLUSTER_REQUEST_ATTEMPTS
            ),
            gateway_service_grpc.YasmGatewayServiceStub,
            rpc_name="AlertStatus",
            request=proto_request,
            timeout=META_ALERT_REQUEST_TIMEOUT,
            client_id=self._cluster_provider.grpc_user_agent,
            unistat=self.unistat,
            log=self.log
        )
        result = yield call.future()
        states_response = {}
        for alert_id, status in itertools.izip(requested_alert_ids, result.alert_statuses):
            state = STATUS_MAPPING.get(status.alert_state)
            if state is not None:
                states_response[alert_id.name] = {
                    "state": state,
                    "value": status.thresholds_value,
                    "meta": {
                        "thresholds_value": status.thresholds_value,
                        "visible_value": status.visible_value
                    },
                    "description": status.description
                }
            elif status.alert_state != gateway_service_proto.AlertStatusResponse.AlertStatus.NO_DATA:
                self.log.warning("Alert {} has unsupported status: {}. Details: '{}'".format(
                    alert_id.name,
                    status.alert_state,
                    status.description
                ))

        self.response({
            "format": "dict",
            "meta-alert": {
                "frontId": self.front_id,
                "states": states_response
            }
        })
