"""
    Common functions and constants for both `InstanceKey` and `RequestKey`.
"""

from infra.yasm.gateway.lib.util.verification import is_correct_tag_value, is_correct_itype_value

from errors import EmptyItype, ItypeError, ItypeTypeError, InvalidTagSet, EmptyTagValue, TagValueError


class TagSet(object):
    """
    Underscored format can be respresented in two forms:
        * long, for example upper_prod_imgs-main_sas_self
        * short, for example upper_self

    This class provides names and validators for one form.

    :param list[str] tags: tag names
    :param list[(str) -> bool] predicates: value validators
    :param list[str]: aggregated tag if tier is self
    """

    def __init__(self, tags, predicates, aggregated):
        assert tags[-1] == "tier"  # Please do not remove this assert without RequestKey and InstanceKey audit

        self.tags = tags
        self.predicates = predicates
        self.sorted_aggregated = sorted(aggregated)

        self.tags_set = set(tags)

        self.sorted_tags, self.sorted_predicates = zip(*sorted(zip(tags, predicates)))
        self.sorted_tags = list(self.sorted_tags)
        self.sorted_predicates = list(self.sorted_predicates)

        self.sorted_tags_without_tier = self.sorted_tags[:]
        if self.sorted_tags_without_tier[-1] == "tier":
            self.sorted_tags_without_tier.pop()


#: Ordered set of tag names in underscored format, short version, e.g. upper_self
SHORT_TAGS = TagSet(["tier"], [lambda tier: tier == "self"], ["ctype", "prj", "geo", "tier"])

#: Ordered set of tag names in underscored format, long version, e.g. upper_prod_imgs-main_sas_self
LONG_TAGS = TagSet(["ctype", "prj", "geo", "tier"],
                   [is_correct_tag_value, is_correct_tag_value, is_correct_tag_value, is_correct_tag_value],
                   ["tier"])


def parse_itype(raw, str):
    """
    Helper function, parses and validates itype or raise an exception.

    :param T raw: Full source, used for better error reporting
    :param str str: Itype stirng

    :raises EmptyItype: Itype is an empty string
    :raises ItypeError: Itype doesn't match restrictions
    :raises ItypeTypeError: `str` must be a string
    :rtype: str
    """
    try:
        if not is_correct_itype_value(str):
            if not str:
                raise EmptyItype(raw)
            raise ItypeError(raw, str)
    except TypeError:
        raise ItypeTypeError(raw, type(str))
    return str


def parse_underscored_tags(raw_str):
    """
    Helper function, parses tags in undercored format,
    used in both instance and request modules.

    :param str raw_str: Full tag string, used for better error reporting

    :raises EmptyItype: Itype is an empty string
    :raises ItypError: Itype doesn't match restrictions
    :raises InvalidTagSet: Tag is not in short or long form
    :rtype: (str, TagSet, typing.Iterable[(str, str)])
    """
    splitted = raw_str.split("_")
    raw_itype = splitted[0]
    itype = parse_itype(raw_str, raw_itype)

    tag_values = splitted[1:]

    if len(tag_values) == 1:
        tag_set = SHORT_TAGS
    elif len(tag_values) == 4:
        tag_set = LONG_TAGS
    else:
        raise InvalidTagSet(raw_str, len(tag_values))

    return itype, tag_set, _iter_underscored_tags(raw_str, tag_set, tag_values)


def _iter_underscored_tags(raw_str, tag_set, tag_values):
    for tag_name, predicate, tag_value in zip(tag_set.tags, tag_set.predicates, tag_values):
        if not predicate(tag_value):
            if not tag_value:
                raise EmptyTagValue(raw_str, tag_name)
            raise TagValueError(raw_str, tag_value)
        yield tag_name, tag_value
